"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
class TestApp {
    constructor() {
        const account = '123456789012';
        const region = 'bermuda-triangle';
        const context = {
            [`availability-zones:${account}:${region}`]: `${region}-1a`,
        };
        this.app = new cdk.App({ context });
        this.stack = new cdk.Stack(this.app, 'MyStack', { env: { account, region } });
    }
}
module.exports = {
    'default properties': {
        'public hosted zone'(test) {
            const app = new TestApp();
            new lib_1.PublicHostedZone(app.stack, 'HostedZone', { zoneName: 'test.public' });
            assert_1.expect(app.stack).to(assert_1.exactlyMatchTemplate({
                Resources: {
                    HostedZoneDB99F866: {
                        Type: 'AWS::Route53::HostedZone',
                        Properties: {
                            Name: 'test.public.',
                        },
                    },
                },
            }));
            test.done();
        },
        'private hosted zone'(test) {
            const app = new TestApp();
            const vpcNetwork = new ec2.Vpc(app.stack, 'VPC');
            new lib_1.PrivateHostedZone(app.stack, 'HostedZone', { zoneName: 'test.private', vpc: vpcNetwork });
            assert_1.expect(app.stack).to(assert_1.beASupersetOfTemplate({
                Resources: {
                    HostedZoneDB99F866: {
                        Type: 'AWS::Route53::HostedZone',
                        Properties: {
                            Name: 'test.private.',
                            VPCs: [{
                                    VPCId: { Ref: 'VPCB9E5F0B4' },
                                    VPCRegion: 'bermuda-triangle',
                                }],
                        },
                    },
                },
            }));
            test.done();
        },
        'when specifying multiple VPCs'(test) {
            const app = new TestApp();
            const vpcNetworkA = new ec2.Vpc(app.stack, 'VPC1');
            const vpcNetworkB = new ec2.Vpc(app.stack, 'VPC2');
            new lib_1.PrivateHostedZone(app.stack, 'HostedZone', { zoneName: 'test.private', vpc: vpcNetworkA })
                .addVpc(vpcNetworkB);
            assert_1.expect(app.stack).to(assert_1.beASupersetOfTemplate({
                Resources: {
                    HostedZoneDB99F866: {
                        Type: 'AWS::Route53::HostedZone',
                        Properties: {
                            Name: 'test.private.',
                            VPCs: [{
                                    VPCId: { Ref: 'VPC17DE2CF87' },
                                    VPCRegion: 'bermuda-triangle',
                                },
                                {
                                    VPCId: { Ref: 'VPC2C1F0E711' },
                                    VPCRegion: 'bermuda-triangle',
                                }],
                        },
                    },
                },
            }));
            test.done();
        },
    },
    'exporting and importing works'(test) {
        const stack2 = new cdk.Stack();
        const importedZone = lib_1.HostedZone.fromHostedZoneAttributes(stack2, 'Imported', {
            hostedZoneId: 'hosted-zone-id',
            zoneName: 'cdk.local',
        });
        new lib_1.TxtRecord(importedZone, 'Record', {
            zone: importedZone,
            recordName: 'lookHere',
            values: ['SeeThere'],
        });
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::Route53::RecordSet', {
            HostedZoneId: 'hosted-zone-id',
            Name: 'lookHere.cdk.local.',
            ResourceRecords: ['"SeeThere"'],
            Type: 'TXT',
        }));
        test.done();
    },
    'adds period to name if not provided'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new lib_1.HostedZone(stack, 'MyHostedZone', {
            zoneName: 'zonename',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::HostedZone', {
            Name: 'zonename.',
        }));
        test.done();
    },
    'fails if zone name ends with a trailing dot'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.throws(() => new lib_1.HostedZone(stack, 'MyHostedZone', { zoneName: 'zonename.' }), /zone name must not end with a trailing dot/);
        test.done();
    },
    'a hosted zone can be assiciated with a VPC either upon creation or using "addVpc"'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc1 = new ec2.Vpc(stack, 'VPC1');
        const vpc2 = new ec2.Vpc(stack, 'VPC2');
        const vpc3 = new ec2.Vpc(stack, 'VPC3');
        // WHEN
        const zone = new lib_1.HostedZone(stack, 'MyHostedZone', {
            zoneName: 'zonename',
            vpcs: [vpc1, vpc2],
        });
        zone.addVpc(vpc3);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::HostedZone', {
            VPCs: [
                {
                    VPCId: {
                        Ref: 'VPC17DE2CF87',
                    },
                    VPCRegion: {
                        Ref: 'AWS::Region',
                    },
                },
                {
                    VPCId: {
                        Ref: 'VPC2C1F0E711',
                    },
                    VPCRegion: {
                        Ref: 'AWS::Region',
                    },
                },
                {
                    VPCId: {
                        Ref: 'VPC3CB5FCDA8',
                    },
                    VPCRegion: {
                        Ref: 'AWS::Region',
                    },
                },
            ],
        }));
        test.done();
    },
    'public zone cannot be associated with a vpc (runtime error)'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const zone = new lib_1.PublicHostedZone(stack, 'MyHostedZone', { zoneName: 'zonename' });
        const vpc = new ec2.Vpc(stack, 'VPC');
        // THEN
        test.throws(() => zone.addVpc(vpc), /Cannot associate public hosted zones with a VPC/);
        test.done();
    },
    'setting up zone delegation'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const zone = new lib_1.PublicHostedZone(stack, 'TopZone', { zoneName: 'top.test' });
        const delegate = new lib_1.PublicHostedZone(stack, 'SubZone', { zoneName: 'sub.top.test' });
        // WHEN
        zone.addDelegation(delegate, { ttl: cdk.Duration.seconds(1337) });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
            Type: 'NS',
            Name: 'sub.top.test.',
            HostedZoneId: stack.resolve(zone.hostedZoneId),
            ResourceRecords: stack.resolve(delegate.hostedZoneNameServers),
            TTL: '1337',
        }));
        test.done();
    },
    'public hosted zone wiht caaAmazon set to true'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new lib_1.PublicHostedZone(stack, 'MyHostedZone', {
            zoneName: 'protected.com',
            caaAmazon: true,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
            Type: 'CAA',
            Name: 'protected.com.',
            ResourceRecords: [
                '0 issue "amazon.com"',
            ],
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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