# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['enchanter',
 'enchanter.addons',
 'enchanter.addons.criterions',
 'enchanter.addons.layers',
 'enchanter.callbacks',
 'enchanter.engine',
 'enchanter.metrics',
 'enchanter.preprocessing',
 'enchanter.preprocessing.signal',
 'enchanter.tasks',
 'enchanter.utils']

package_data = \
{'': ['*']}

install_requires = \
['comet_ml>=3.1.3,<4.0.0',
 'netron>=4.5,<5.0',
 'numpy>=1.17,<2.0',
 'pandas>=1.0,<2.0',
 'scikit-learn>=0.23,<0.24',
 'tensorboard>=2.1,<3.0',
 'tqdm>=4.41,<5.0']

setup_kwargs = {
    'name': 'enchanter',
    'version': '0.8.1',
    'description': 'Enchanter is a library for machine learning tasks for comet.ml users.',
    'long_description': '<div align="center">\n\n<img src="docs/_static/images/Enchanter-Logo.png" width="200px">\n\n# Enchanter\nEnchanter is a library for machine learning tasks for comet.ml users.\n\n<p align="center">\n  <a href="#Getting-Started">Getting Started</a> •\n  <a href="https://enchanter.readthedocs.io/en/stable/">Docs</a> •\n  <a href="https://enchanter.readthedocs.io/en/stable/tutorial/modules.html">Tutorial</a> •\n  <a href="LICENSE">Licence</a>\n</p>\n\n[![Codacy Badge](https://api.codacy.com/project/badge/Grade/84197fb283924f02a1667cea49dd031a)](https://app.codacy.com/manual/KawashimaHirotaka/enchanter?utm_source=github.com&utm_medium=referral&utm_content=khirotaka/enchanter&utm_campaign=Badge_Grade_Dashboard)\n[![CI testing](https://github.com/khirotaka/enchanter/workflows/CI/badge.svg)](https://github.com/khirotaka/enchanter/actions)\n[![Build & Publish](https://github.com/khirotaka/enchanter/workflows/Build%20&%20Publish/badge.svg)](https://github.com/khirotaka/enchanter/actions)\n[![Documentation Status](https://readthedocs.org/projects/enchanter/badge/?version=latest)](https://enchanter.readthedocs.io/)\n\n[![PyPI](https://img.shields.io/pypi/v/enchanter?color=brightgreen)](https://pypi.org/project/enchanter/)\n[![Dependabot Status](https://api.dependabot.com/badges/status?host=github&repo=khirotaka/enchanter)](https://dependabot.com)\n[![license](https://img.shields.io/github/license/khirotaka/enchanter?color=light)](LICENSE)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n[![Using PyTorch](https://img.shields.io/badge/PyTorch-red.svg?labelColor=f3f4f7&logo=data:image/svg+xml;base64,PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0iVVRGLTgiPz4KPHN2ZyB3aWR0aD0iMjU2cHgiIGhlaWdodD0iMzEwcHgiIHZpZXdCb3g9IjAgMCAyNTYgMzEwIiB2ZXJzaW9uPSIxLjEiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyIgeG1sbnM6eGxpbms9Imh0dHA6Ly93d3cudzMub3JnLzE5OTkveGxpbmsiIHByZXNlcnZlQXNwZWN0UmF0aW89InhNaWRZTWlkIj4KICAgIDxnPgogICAgICAgIDxwYXRoIGQ9Ik0yMTguMjgxMDM3LDkwLjEwNjQxMiBDMjY4LjU3Mjk4OCwxNDAuMzk4MzYzIDI2OC41NzI5ODgsMjIxLjA3NTAzNCAyMTguMjgxMDM3LDI3MS43MTYyMzUgQzE2OS4wMzY4MzUsMzIyLjAwODE4NiA4OC4wMTA5MTQxLDMyMi4wMDgxODYgMzcuNzE4OTYzMiwyNzEuNzE2MjM1IEMtMTIuNTcyOTg3NywyMjEuNDI0Mjg0IC0xMi41NzI5ODc3LDE0MC4zOTgzNjMgMzcuNzE4OTYzMiw5MC4xMDY0MTIgTDEyNy44MjUzNzUsMCBMMTI3LjgyNTM3NSw0NS4wNTMyMDYgTDExOS40NDMzODMsNTMuNDM1MTk3OCBMNTkuNzIxNjkxNywxMTMuMTU2ODg5IEMyMi4wMDI3Mjg1LDE1MC4xNzczNTMgMjIuMDAyNzI4NSwyMTAuOTQ2Nzk0IDU5LjcyMTY5MTcsMjQ4LjY2NTc1NyBDOTYuNzQyMTU1NSwyODYuMzg0NzIgMTU3LjUxMTU5NiwyODYuMzg0NzIgMTk1LjIzMDU1OSwyNDguNjY1NzU3IEMyMzIuOTQ5NTIzLDIxMS42NDUyOTMgMjMyLjk0OTUyMywxNTAuODc1ODUzIDE5NS4yMzA1NTksMTEzLjE1Njg4OSBMMjE4LjI4MTAzNyw5MC4xMDY0MTIgWiBNMTczLjIyNzgzMSw4NC41MTg0MTc1IEMxNjMuOTY5MzM4LDg0LjUxODQxNzUgMTU2LjQ2Mzg0Nyw3Ny4wMTI5MjYzIDE1Ni40NjM4NDcsNjcuNzU0NDMzOCBDMTU2LjQ2Mzg0Nyw1OC40OTU5NDEzIDE2My45NjkzMzgsNTAuOTkwNDUwMiAxNzMuMjI3ODMxLDUwLjk5MDQ1MDIgQzE4Mi40ODYzMjMsNTAuOTkwNDUwMiAxODkuOTkxODE0LDU4LjQ5NTk0MTMgMTg5Ljk5MTgxNCw2Ny43NTQ0MzM4IEMxODkuOTkxODE0LDc3LjAxMjkyNjMgMTgyLjQ4NjMyMyw4NC41MTg0MTc1IDE3My4yMjc4MzEsODQuNTE4NDE3NSBaIiBmaWxsPSIjRUU0QzJDIj48L3BhdGg+CiAgICA8L2c+Cjwvc3ZnPgo=)](https://pytorch.org/)\n\n</div>\n\n---\n\n## Installation\nTo get started, [install PyTorch](https://pytorch.org) for your environment.\nThen install Enchanter in the following way:  \n\nTo install the stable release.\n```shell script\npip install enchanter\n```\n\nor\n\nTo install the latest(unstable) release. \n```shell script\npip install git+https://github.com/khirotaka/enchanter.git\n```\n\nIf you want to install with a specific branch, you can use the following.\n```shell script\n# e.g.) Install enchanter from develop branch.\npip install git+https://github.com/khirotaka/enchanter.git@develop\n```\n\n### Supported Platforms\nEnchanter supports:\n*   macOS 10.15\n*   Ubuntu 18.04 or later\n\n## Getting Started\nTry your first Enchanter Program. To train a neural network written in PyTorch on Enchanter, use the `Runner`.  \nThere are 2 ways to define a `Runner`:\n\n1.  To use a `Runner` already implemented under `enchanter.tasks`\n2.  To define a custom `Runner` that inherit `enchanter.engine.BaseRunner`.\n\nLet\'s see how to use the `enchanter.tasks.ClassificationRunner`, which is the easiest way.\n\n### Training Neural Network\n\n```python\nimport comet_ml\nimport torch\nimport enchanter\n\nmodel = torch.nn.Linear(6, 10)\noptimizer = torch.optim.Adam(model.parameters())\n\nrunner = enchanter.tasks.ClassificationRunner(\n    model, \n    optimizer,\n    criterion=torch.nn.CrossEntropyLoss(),\n    experiment=comet_ml.Experiment()\n)\n\nrunner.add_loader("train", train_loader)\nrunner.train_config(epochs=10)\nrunner.run()\n```\n\nRegister a `torch.utils.data.DataLoader` with the `Runner` by using `.add_loader()`.  \nSet up the number of epochs using `.train_config()`, and execute `Runner` with `.run()`.\n\n### Training Unsupervised Time Series Feature Learning\nThe wonderful algorithms for unsupervised time series representation learning, adopted at [NeurIPS 2019](https://papers.nips.cc/paper/8713-unsupervised-scalable-representation-learning-for-multivariate-time-series), are now easily available.\n\nPlease prepare the following:\n\n1.  PyTorch Model that can output feature vectors of the same length regardless of the input series.\n2.  time series data consisting of `[N, F, L]`.\n3.  (Optional) A teacher label for each sample in `2.`\n\n\n```python\nimport comet_ml\nimport torch.nn as nn\nimport torch.optim as optim\nimport enchanter.tasks as tasks\nimport enchanter.addons.layers as L\n\n\nclass Encoder(nn.Module):\n    def __init__(self, in_features, mid_features, out_features):\n        super(Encoder, self).__init__()\n        self.conv = nn.Sequential(\n            L.CausalConv1d(in_features, mid_features, 3),\n            nn.LeakyReLU(),\n            L.CausalConv1d(mid_features, mid_features, 3),\n            nn.LeakyReLU(),\n            L.CausalConv1d(mid_features, mid_features, 3),\n            nn.LeakyReLU(),\n            nn.AdaptiveMaxPool1d(1)\n        )\n        self.fc = nn.Linear(mid_features, out_features)\n\n    def forward(self, x):\n        batch = x.shape[0]\n        out = self.conv(x).reshape(batch, -1)\n        return self.fc(out)\n\n\nexperiment = comet_ml.Experiment()\nmodel = Encoder(...)\noptimizer = optim.Adam(model.parameters())\n\nrunner = tasks.TimeSeriesUnsupervisedRunner(model, optimizer, experiment)\nrunner.add_loader("train", ...)\nrunner.run()\n```\n\nA teacher label is required for validation. Also, Use `enchanter.callbacks.EarlyStoppingForTSUS` for early stopping. \n \n### Hyper parameter searching using Comet.ml\n\n```python\nfrom comet_ml import Optimizer\n\nimport torch\nimport torch.nn as nn\nimport torch.optim as optim\nfrom sklearn.datasets import load_iris\n\nimport enchanter.tasks as tasks\nimport enchanter.addons as addons\nimport enchanter.addons.layers as layers\nfrom enchanter.utils import comet\n\n\nconfig = comet.TunerConfigGenerator(\n    algorithm="bayes",\n    metric="train_avg_loss",\n    objective="minimize",\n    seed=0,\n    trials=1,\n    max_combo=10\n)\n\nconfig.suggest_categorical("activation", ["addons.mish", "torch.relu", "torch.sigmoid"])\nopt = Optimizer(config.generate())\n\nx, y = load_iris(return_X_y=True)\nx = x.astype("float32")\ny = y.astype("int64")\n\n\nfor experiment in opt.get_experiments():\n    model = layers.MLP([4, 512, 128, 3], eval(experiment.get_parameter("activation")))\n    optimizer = optim.Adam(model.parameters())\n    runner = tasks.ClassificationRunner(\n        model, optimizer=optimizer, criterion=nn.CrossEntropyLoss(), experiment=experiment\n    )\n\n    runner.fit(x, y, epochs=1, batch_size=32)\n    runner.quite()\n\n    # or \n    # with runner:\n    #   runner.fit(...)\n    # or\n    #   runner.run()\n\n```\n\n### Training with Mixed Precision\nRunners with defined in `enchanter.tasks` are now support Auto Mixed Precision.  \nWrite the following.\n\n\n```python\nfrom torch.cuda import amp\nfrom enchanter.tasks import ClassificationRunner\n\n\nrunner = ClassificationRunner(...)\nrunner.scaler = amp.GradScaler()\n```\n\n\nIf you want to define a custom runner that supports mixed precision, do the following.\n```python\nfrom torch.cuda import amp\nimport torch.nn.functional as F\nfrom enchanter.engine import BaseRunner\n\n\nclass CustomRunner(BaseRunner):\n    # ...\n    def train_step(self, batch):\n        x, y = batch\n        with amp.autocast():        # REQUIRED\n            out = self.model(x)\n            loss = F.nll_loss(out, y)\n        \n        return {"loss": loss}\n\n\nrunner = CustomRunner(...)\nrunner.scaler = amp.GradScaler()\n```\n\nThat is, you can enable AMP by using `torch.cuda.amp.autocast()` in `.train_step()`, `.val_step()` and `.test_step()`.\n\n### with-statement training\n\n```python\nfrom comet_ml import Experiment\n\nimport torch\nimport torch.nn as nn\nimport torch.optim as optim\nfrom torch.utils.data import DataLoader\nfrom sklearn.datasets import load_iris\nfrom tqdm.auto import tqdm\n\nimport enchanter.tasks as tasks\nimport enchanter.engine.modules as modules\nimport enchanter.addons as addons\nimport enchanter.addons.layers as layers\n\n\nexperiment = Experiment()\nmodel = layers.MLP([4, 512, 128, 3], addons.mish)\noptimizer = optim.Adam(model.parameters())\n\nx, y = load_iris(return_X_y=True)\nx = x.astype("float32")\ny = y.astype("int64")\n\ntrain_ds = modules.get_dataset(x, y)\nval_ds = modules.get_dataset(x, y)\ntest_ds = modules.get_dataset(x, y)\n\ntrain_loader = DataLoader(train_ds, batch_size=32)\nval_loader = DataLoader(val_ds, batch_size=32)\ntest_loader = DataLoader(test_ds, batch_size=32)\n\nrunner = tasks.ClassificationRunner(\n    model, optimizer, nn.CrossEntropyLoss(), experiment\n)\n\nwith runner:\n    for epoch in tqdm(range(10)):\n        with runner.experiment.train():\n            for train_batch in train_loader:\n                runner.optimizer.zero_grad()\n                train_out = runner.train_step(train_batch)\n                runner.backward(train_out["loss"])\n                runner.update_optimizer()\n    \n                with runner.experiment.validate(), torch.no_grad():\n                    for val_batch in val_loader:\n                        val_out = runner.val_step(val_batch)["loss"]\n                        runner.experiment.log_metric("val_loss", val_out)\n\n        with runner.experiment.test(), torch.no_grad():\n            for test_batch in test_loader:\n                test_out = runner.test_step(test_batch)["loss"]\n                runner.experiment.log_metric("test_loss", test_out)\n\n# The latest checkpoints (model_state & optim_state) are stored\n# in comet.ml after the with statement.\n```\n\n## Graph visualization\n\n```python\nimport torch\nfrom enchanter.utils import visualize\nfrom enchanter.addons.layers import AutoEncoder\n\nx = torch.randn(1, 32)  # [N, in_features]\nmodel = AutoEncoder([32, 16, 8, 2])\nvisualize.with_netron(model, (x, ))\n```\n\n![netron_graph](docs/tutorial/assets/netron_viewer.png)\n\n## License\n[Apache License 2.0](LICENSE)\n',
    'author': 'Hirotaka Kawashima',
    'author_email': 'khirotaka@vivaldi.net',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://enchanter.readthedocs.io/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6.5,<3.9',
}


setup(**setup_kwargs)
