"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PgStacDatabase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const bootstrapper_1 = require("../bootstrapper");
const instanceSizes = require("./instance-memory.json");
/**
 * An RDS instance with pgSTAC installed. This is a wrapper around the
 * `rds.DatabaseInstance` higher-level construct making use
 * of the BootstrapPgStac construct.
 */
class PgStacDatabase extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const defaultParameters = this.getParameters(props.instanceType?.toString() || "m5.large", props.parameters);
        const parameterGroup = new aws_cdk_lib_1.aws_rds.ParameterGroup(this, "parameterGroup", {
            engine: props.engine,
            parameters: {
                shared_buffers: defaultParameters.sharedBuffers,
                effective_cache_size: defaultParameters.effectiveCacheSize,
                work_mem: defaultParameters.workMem,
                maintenance_work_mem: defaultParameters.maintenanceWorkMem,
                max_locks_per_transaction: defaultParameters.maxLocksPerTransaction,
                temp_buffers: defaultParameters.tempBuffers,
                seq_page_cost: defaultParameters.seqPageCost,
                random_page_cost: defaultParameters.randomPageCost,
                ...props.parameters,
            },
        });
        this.db = new aws_cdk_lib_1.aws_rds.DatabaseInstance(this, "db", {
            instanceIdentifier: aws_cdk_lib_1.Stack.of(this).stackName,
            parameterGroup,
            ...props,
        });
        const bootstrap = new bootstrapper_1.BootstrapPgStac(this, "bootstrap-pgstac-instance", {
            vpc: props.vpc,
            database: this.db,
            dbSecret: this.db.secret,
            pgstacDbName: props.pgstacDbName,
            pgstacVersion: props.pgstacVersion,
            pgstacUsername: props.pgstacUsername,
            secretsPrefix: props.secretsPrefix,
        });
        this.pgstacSecret = bootstrap.secret;
    }
    getParameters(instanceType, parameters) {
        // https://github.com/aws/aws-cli/issues/1279#issuecomment-909318236
        const memory_in_kb = instanceSizes[instanceType] * 1024;
        // It's only necessary to consider passed in parameters for any value that used to
        // derive subsequent values. Values that don't have dependencies will be overriden
        // when we unpack the passed-in user parameters
        const maxConnections = parameters?.maxConnections
            ? Number.parseInt(parameters.maxConnections)
            : // https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.MaxConnections
                Math.min(Math.round((memory_in_kb * 1024) / 9531392), 5000);
        const sharedBuffers = parameters?.sharedBufers
            ? Number.parseInt(parameters.sharedBufers)
            : Math.round(0.25 * memory_in_kb);
        const effectiveCacheSize = Math.round(0.75 * memory_in_kb);
        const workMem = Math.floor(sharedBuffers / maxConnections);
        const maintenanceWorkMem = Math.round(0.25 * sharedBuffers);
        const tempBuffers = 128 * 1024;
        const seqPageCost = 1;
        const randomPageCost = 1.1;
        return {
            maxConnections: `${maxConnections}`,
            sharedBuffers: `${sharedBuffers / 8}`,
            effectiveCacheSize: `${effectiveCacheSize}`,
            workMem: `${workMem}`,
            maintenanceWorkMem: `${maintenanceWorkMem}`,
            maxLocksPerTransaction: "1024",
            tempBuffers: `${tempBuffers}`,
            seqPageCost: `${seqPageCost}`,
            randomPageCost: `${randomPageCost}`,
        };
    }
}
exports.PgStacDatabase = PgStacDatabase;
_a = JSII_RTTI_SYMBOL_1;
PgStacDatabase[_a] = { fqn: "cdk-pgstac.PgStacDatabase", version: "2.6.2" };
//# sourceMappingURL=data:application/json;base64,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