import * as events from '@aws-cdk/aws-events';
import * as iam from '@aws-cdk/aws-iam';
import { IResource, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
export interface IRepository extends IResource {
    /**
     * The ARN of this Repository.
     * @attribute
     */
    readonly repositoryArn: string;
    /**
     * The human-visible name of this Repository.
     * @attribute
     */
    readonly repositoryName: string;
    /**
     * The HTTP clone URL
     * @attribute
     */
    readonly repositoryCloneUrlHttp: string;
    /**
     * The SSH clone URL
     * @attribute
     */
    readonly repositoryCloneUrlSsh: string;
    /**
     * Defines a CloudWatch event rule which triggers for repository events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a "CodeCommit
     * Repository State Change" event occurs.
     */
    onStateChange(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a reference is
     * created (i.e. a new branch/tag is created) to the repository.
     */
    onReferenceCreated(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a reference is
     * updated (i.e. a commit is pushed to an existing or new branch) from the repository.
     */
    onReferenceUpdated(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a reference is
     * delete (i.e. a branch/tag is deleted) from the repository.
     */
    onReferenceDeleted(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a pull request state is changed.
     */
    onPullRequestStateChange(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a comment is made on a pull request.
     */
    onCommentOnPullRequest(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a comment is made on a commit.
     */
    onCommentOnCommit(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a commit is pushed to a branch.
     */
    onCommit(id: string, options?: OnCommitOptions): events.Rule;
    /**
     * Grant the given principal identity permissions to perform the actions on this repository
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Grant the given identity permissions to pull this repository.
     */
    grantPull(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant the given identity permissions to pull and push this repository.
     */
    grantPullPush(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant the given identity permissions to read this repository.
     */
    grantRead(grantee: iam.IGrantable): iam.Grant;
}
/**
 * Options for the onCommit() method
 */
export interface OnCommitOptions extends events.OnEventOptions {
    /**
     * The branch to monitor.
     *
     * @default - All branches
     */
    readonly branches?: string[];
}
/**
 * Represents a reference to a CodeCommit Repository.
 *
 * If you want to create a new Repository managed alongside your CDK code,
 * use the {@link Repository} class.
 *
 * If you want to reference an already existing Repository,
 * use the {@link Repository.import} method.
 */
declare abstract class RepositoryBase extends Resource implements IRepository {
    /** The ARN of this Repository. */
    abstract readonly repositoryArn: string;
    /** The human-visible name of this Repository. */
    abstract readonly repositoryName: string;
    /** The HTTP clone URL */
    abstract readonly repositoryCloneUrlHttp: string;
    /** The SSH clone URL */
    abstract readonly repositoryCloneUrlSsh: string;
    /**
     * Defines a CloudWatch event rule which triggers for repository events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a "CodeCommit
     * Repository State Change" event occurs.
     */
    onStateChange(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a reference is
     * created (i.e. a new branch/tag is created) to the repository.
     */
    onReferenceCreated(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a reference is
     * updated (i.e. a commit is pushed to an existing or new branch) from the repository.
     */
    onReferenceUpdated(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a reference is
     * delete (i.e. a branch/tag is deleted) from the repository.
     */
    onReferenceDeleted(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a pull request state is changed.
     */
    onPullRequestStateChange(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a comment is made on a pull request.
     */
    onCommentOnPullRequest(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a comment is made on a commit.
     */
    onCommentOnCommit(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a commit is pushed to a branch.
     */
    onCommit(id: string, options?: OnCommitOptions): events.Rule;
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    grantPull(grantee: iam.IGrantable): iam.Grant;
    grantPullPush(grantee: iam.IGrantable): iam.Grant;
    grantRead(grantee: iam.IGrantable): iam.Grant;
}
export interface RepositoryProps {
    /**
     * Name of the repository.
     *
     * This property is required for all CodeCommit repositories.
     */
    readonly repositoryName: string;
    /**
     * A description of the repository. Use the description to identify the
     * purpose of the repository.
     *
     * @default - No description.
     */
    readonly description?: string;
}
/**
 * Provides a CodeCommit Repository
 */
export declare class Repository extends RepositoryBase {
    /**
     * Imports a codecommit repository.
     * @param repositoryArn (e.g. `arn:aws:codecommit:us-east-1:123456789012:MyDemoRepo`)
     */
    static fromRepositoryArn(scope: Construct, id: string, repositoryArn: string): IRepository;
    static fromRepositoryName(scope: Construct, id: string, repositoryName: string): IRepository;
    private static makeCloneUrl;
    private static arnForLocalRepository;
    readonly repositoryArn: string;
    readonly repositoryName: string;
    private readonly repository;
    private readonly triggers;
    constructor(scope: Construct, id: string, props: RepositoryProps);
    get repositoryCloneUrlHttp(): string;
    get repositoryCloneUrlSsh(): string;
    /**
     * Create a trigger to notify another service to run actions on repository events.
     * @param arn   Arn of the resource that repository events will notify
     * @param options Trigger options to run actions
     */
    notify(arn: string, options?: RepositoryTriggerOptions): Repository;
}
/**
 * Creates for a repository trigger to an SNS topic or Lambda function.
 */
export interface RepositoryTriggerOptions {
    /**
     * A name for the trigger.Triggers on a repository must have unique names
     */
    readonly name?: string;
    /**
     * The repository events for which AWS CodeCommit sends information to the
     * target, which you specified in the DestinationArn property.If you don't
     * specify events, the trigger runs for all repository events.
     */
    readonly events?: RepositoryEventTrigger[];
    /**
     * The names of the branches in the AWS CodeCommit repository that contain
     * events that you want to include in the trigger. If you don't specify at
     * least one branch, the trigger applies to all branches.
     */
    readonly branches?: string[];
    /**
     * When an event is triggered, additional information that AWS CodeCommit
     * includes when it sends information to the target.
     */
    readonly customData?: string;
}
/**
 * Repository events that will cause the trigger to run actions in another service.
 */
export declare enum RepositoryEventTrigger {
    ALL = "all",
    UPDATE_REF = "updateReference",
    CREATE_REF = "createReference",
    DELETE_REF = "deleteReference"
}
export {};
