from ApiMethod import ApiMethod


class Photos(ApiMethod):
    def __init__(self, access_token, v, session, calls_per_second):
        super(Photos, self).__init__(access_token, v, session, calls_per_second)
        self._base_method = 'photos.'

    def confirmTag(self, owner_id: str = None, photo_id: str = None, tag_id: str = None) -> dict:
        """
        Подтверждает отметку на фотографии.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. Обратите
            внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например,
            owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)
        :param photo_id: идентификатор фотографии.
        :param tag_id: идентификатор отметки на фотографии.

        :return: После успешного выполнения возвращает 1
        """

        params = locals()
        method_name = self._base_method + 'confirmTag'
        return self._call(method_name, params)

    def copy(self, owner_id: str = None, photo_id: str = None, access_key: str = None) -> dict:
        """
        Позволяет скопировать фотографию в альбом "Сохраненные фотографии"

        :param owner_id: идентификатор владельца фотографии
        :param photo_id: индентификатор фотографии
        :param access_key: специальный код доступа для приватных фотографий

        :return: Возвращает идентификатор созданной фотографии
        """

        params = locals()
        method_name = self._base_method + 'copy'
        return self._call(method_name, params)

    def createAlbum(self, title: str = None, group_id: str = None, description: str = None, privacy_view: str = None,
                    privacy_comment: str = None, upload_by_admins_only: str = None,
                    comments_disabled: str = None) -> dict:
        """
        Создает пустой альбом для фотографий.

        :param title: название альбома.
        :param group_id: идентификатор сообщества, в котором создаётся альбом.
        :param description: текст описания альбома.
        :param privacy_view: настройки приватности просмотра альбома в специальном формате.
        :param privacy_comment: настройки приватности комментирования альбома в специальном формате.
        :param upload_by_admins_only: кто может загружать фотографии в альбом (только для альбома сообщества).
            Возможные значения:   0 — фотографии могут добавлять все пользователи;  1 — фотографии могут добавлять
            только редакторы и администраторы.
        :param comments_disabled: отключено ли комментирование альбома (только для альбома сообщества).
            Возможные значения:   0 — комментирование включено;  1 — комментирование отключено.

        :return: После успешного выполнения возвращает объект, который содержит следующие поля:   id —
            идентификатор созданного альбома;  thumb_id — идентификатор фотографии, которая является обложкой альбома
            (0, если обложка отсутствует);  owner_id идентификатор пользователя или сообщества, которому принадлежит
            альбом;  title— название альбома;  description — описание альбома;  created — дата создания альбома в
            формате unixtime;  updated — дата обновления альбома в формате unixtime;  size — количество фотографий в
            альбоме;  privacy_view — настройки приватности для альбома в формате настроек приватности (только для
            альбома пользователя);  privacy_comment — настройки приватности для альбома в формате настроек
            приватности (только для альбома пользователя);  upload_by_admins_only — кто может загружать фотографии в
            альбом (только для альбома сообщества);  comments_disabled — отключено ли комментирование альбома (только
            для альбома сообщества);  can_upload — может ли текущий пользователь добавлять фотографии в альбом
        """

        params = locals()
        method_name = self._base_method + 'createAlbum'
        return self._call(method_name, params)

    def createComment(self, owner_id: str = None, photo_id: str = None, message: str = None, attachments: str = None,
                      from_group: str = None, reply_to_comment: str = None, sticker_id: str = None,
                      access_key: str = None, guid: str = None) -> dict:
        """
        Создает новый комментарий к фотографии.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. Обратите
            внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например,
            owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)
        :param photo_id: идентификатор фотографии.
        :param message: текст комментария (является обязательным, если не задан параметр attachments).
            Максимальное количество символов: 2048.
        :param attachments: список объектов, приложенных к комментарию и разделённых символом ",". Поле
            attachments представляется в формате: <type><owner_id>_<media_id>,<type><owner_id>_<media_id> <type> —
            тип медиа-вложения: photo — фотография  video — видеозапись  audio — аудиозапись  doc — документ
            <owner_id> — идентификатор владельца медиа-вложения  <media_id> — идентификатор медиа-вложения.
            Например: photo100172_166443618,photo66748_265827614 Параметр является обязательным, если не задан
            параметр message.
        :param from_group: Данный параметр учитывается, если owner_id < 0 (комментарий к фотографии группы).
            Возможные значения:   1 — комментарий будет опубликован от имени группы;  0 — комментарий будет
            опубликован от имени пользователя.  По умолчанию: 0.
        :param reply_to_comment: идентификатор комментария, в ответ на который нужно оставить текущий.
        :param sticker_id: идентификатор стикера, который нужно прикрепить к комментарию.
        :param access_key: ключ доступа.
        :param guid: уникальное значение для предотвращения повторной отправки одного и того же комментария.

        :return: После успешного выполнения возвращает идентификатор созданного комментария
        """

        params = locals()
        method_name = self._base_method + 'createComment'
        return self._call(method_name, params)

    def delete(self, owner_id: str = None, photo_id: str = None) -> dict:
        """
        Удаление фотографии на сайте.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. Обратите
            внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например,
            owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)
        :param photo_id: идентификатор фотографии.

        :return: После успешного выполнения возвращает 1
        """

        params = locals()
        method_name = self._base_method + 'delete'
        return self._call(method_name, params)

    def deleteAlbum(self, album_id: str = None, group_id: str = None) -> dict:
        """
        Удаляет указанный альбом для фотографий у текущего пользователя

        :param album_id: идентификатор альбома.
        :param group_id: идентификатор сообщества, в котором размещен альбом.

        :return: После успешного выполнения возвращает 1
        """

        params = locals()
        method_name = self._base_method + 'deleteAlbum'
        return self._call(method_name, params)

    def deleteComment(self, owner_id: str = None, comment_id: str = None) -> dict:
        """
        Удаляет комментарий к фотографии.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. Обратите
            внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например,
            owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)
        :param comment_id: идентификатор комментария.

        :return: После успешного выполнения возвращает 1 (0, если комментарий не найден)
        """

        params = locals()
        method_name = self._base_method + 'deleteComment'
        return self._call(method_name, params)

    def edit(self, owner_id: str = None, photo_id: str = None, caption: str = None, latitude: str = None,
             longitude: str = None, place_str: str = None, foursquare_id: str = None, delete_place: str = None) -> dict:
        """
        Редактирует описание или геометку у фотографии.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. Обратите
            внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например,
            owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)
        :param photo_id: идентификатор фотографии.
        :param caption: новый текст описания к фотографии. Если параметр не задан, то считается, что он равен
            пустой строке.
        :param latitude: географическая широта.
        :param longitude: географическая долгота.
        :param place_str: название места.
        :param foursquare_id: id в Foursquare.
        :param delete_place: удалить место (0 — не удалять, 1 — удалить).

        :return: После успешного выполнения возвращает 1
        """

        params = locals()
        method_name = self._base_method + 'edit'
        return self._call(method_name, params)

    def editAlbum(self, album_id: str = None, title: str = None, description: str = None, owner_id: str = None,
                  privacy_view: str = None, privacy_comment: str = None, upload_by_admins_only: str = None,
                  comments_disabled: str = None) -> dict:
        """
        Редактирует данные альбома для фотографий.

        :param album_id: идентификатор альбома.
        :param title: новое название альбома.
        :param description: новый текст описания альбома.
        :param owner_id: идентификатор владельца альбома (пользователь или сообщество). Обратите внимание,
            идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1
            соответствует идентификатору сообщества ВКонтакте API (club1)
        :param privacy_view: настройки приватности просмотра альбома в специальном формате.
        :param privacy_comment: настройки приватности комментирования альбома в специальном формате.
        :param upload_by_admins_only: кто может загружать фотографии в альбом (только для альбома сообщества).
            Возможные значения:   0 — фотографии могут добавлять все пользователи;  1 — фотографии могут добавлять
            только редакторы и администраторы.
        :param comments_disabled: отключено ли комментирование альбома (только для альбома сообщества).
            Возможные значения:   0 — комментирование включено;  1 — комментирование отключено.

        :return: После успешного выполнения возвращает 1
        """

        params = locals()
        method_name = self._base_method + 'editAlbum'
        return self._call(method_name, params)

    def editComment(self, owner_id: str = None, comment_id: str = None, message: str = None,
                    attachments: str = None) -> dict:
        """
        Изменяет текст комментария к фотографии.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. Обратите
            внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например,
            owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)
        :param comment_id: идентификатор комментария.
        :param message: новый текст комментария. Обязательный параметр, если не задан параметр attachments.
            Максимальное количество символов: 2048.
        :param attachments: новый список объектов, приложенных к комментарию и разделённых символом ",". Поле
            attachments представляется в формате: <type><owner_id>_<media_id>,<type><owner_id>_<media_id> <type> —
            тип медиа-вложения: photo — фотография  video — видеозапись  audio — аудиозапись  doc — документ
            <owner_id> — идентификатор владельца медиа-вложения  <media_id> — идентификатор медиа-вложения.
            Например: photo100172_166443618,photo66748_265827614 Параметр является обязательным, если не задан
            параметр message.

        :return: После успешного выполнения возвращает 1
        """

        params = locals()
        method_name = self._base_method + 'editComment'
        return self._call(method_name, params)

    def get(self, owner_id: str = None, album_id: str = None, photo_ids: str = None, rev: str = None,
            extended: str = None, feed_type: str = None, feed: str = None, photo_sizes: str = None, offset: str = None,
            count: str = None) -> dict:
        """
        Возвращает список фотографий в альбоме.

        :param owner_id: идентификатор владельца альбома. Обратите внимание, идентификатор сообщества в параметре
            owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору
            сообщества ВКонтакте API (club1)
        :param album_id: идентификатор альбома. Для служебных альбомов используются следующие идентификаторы:
            wall — фотографии со стены;  profile — фотографии профиля;  saved — сохраненные фотографии. Возвращается
            только с ключом доступа пользователя.
        :param photo_ids: идентификаторы фотографий, информацию о которых необходимо вернуть.
        :param rev: порядок сортировки фотографий. Возможные значения:   1 — антихронологический;  0 —
            хронологический.
        :param extended: 1 — будут возвращены дополнительные поля likes, comments, tags, can_comment, reposts. По
            умолчанию: 0.
        :param feed_type: тип новости, получаемый в поле type метода newsfeed.get, для получения только
            загруженных пользователем фотографий, либо только фотографий, на которых он был отмечен. Может принимать
            значения photo, photo_tag.
        :param feed: время в формате, который может быть получен методом newsfeed.get в поле date, для получения
            всех фотографий загруженных пользователем в определённый день либо на которых пользователь был отмечен.
            Также нужно указать параметр uid пользователя, с которым произошло событие.  Значение должно отличаться
            от текущего времени не более, чем на месяц.
        :param photo_sizes: 1 — возвращать доступные размеры фотографии в специальном формате. По умолчанию: 0.
        :param offset: отступ, необходимый для получения определенного подмножества записей.
        :param count: количество записей, которое будет получено.

        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив
            объектов фотографий в поле items.  Если к фотографии прикреплено местоположение, также возвращаются поля
            lat и long, содержащие географические координаты отметки.  Если был задан параметр extended=1,
            возвращаются дополнительные поля:   likes — количество отметок Мне нравится и информация о том, поставил
            ли лайк текущий пользователь;  comments — количество комментариев к фотографии;  tags — количество
            отметок на фотографии;  can_comment — может ли текущий пользователь комментировать фото (1 — может, 0 —
            не может);  reposts — число репостов фотографии.   Если был задан параметр photo_sizes=1, вместо полей
            width и height возвращаются размеры копий фотографии в специальном формате
        """

        params = locals()
        method_name = self._base_method + 'get'
        return self._call(method_name, params)

    def getAlbums(self, owner_id: str = None, album_ids: str = None, offset: str = None, count: str = None,
                  need_system: str = None, need_covers: str = None, photo_sizes: str = None) -> dict:
        """
        Возвращает список фотоальбомов пользователя или сообщества.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежат альбомы. Обратите
            внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например,
            owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)
        :param album_ids: перечисленные через запятую идентификаторы альбомов.
        :param offset: смещение, необходимое для выборки определенного подмножества альбомов.
        :param count: количество альбомов, которое нужно вернуть. (по умолчанию возвращаются все альбомы)
        :param need_system: 1 — будут возвращены системные альбомы, имеющие отрицательные идентификаторы.
            Обратите внимание, что информация о системных альбомах возвращается даже в том случае, если они не
            содержат фотографий.
        :param need_covers: 1 — будет возвращено дополнительное поле thumb_src с адресом изображения-обложки. По
            умолчанию поле thumb_src не возвращается.
        :param photo_sizes: 1 — размеры фотографий будут возвращены в специальном формате.

        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив
            объектов, описывающих альбомы, в поле items. Каждый из этих объектов содержит следующие поля:   id —
            идентификатор альбома;  thumb_id — идентификатор фотографии, которая является обложкой (0, если обложка
            отсутствует);  owner_id — идентификатор владельца альбома;  title — название альбома;  description —
            описание альбома; (не приходит для системных альбомов)  created — дата создания альбома в формате
            unixtime; (не приходит для системных альбомов);  updated — дата последнего обновления альбома в формате
            unixtime; (не приходит для системных альбомов);  size — количество фотографий в альбоме;  can_upload — 1,
            если текущий пользователь может загружать фотографии в альбом (при запросе информации об альбомах
            сообщества);  privacy_view* — настройки приватности для альбома в формате настроек приватности (только
            для альбома пользователя, не приходит для системных альбомов);  privacy_comment* — настройки приватности
            для альбома в формате настроек приватности (только для альбома пользователя, не приходит для системных
            альбомов);  upload_by_admins_only* — кто может загружать фотографии в альбом (только для альбома
            сообщества, не приходит для системных альбомов);  comments_disabled* — отключено ли комментирование
            альбома (только для альбома сообщества, не приходит для системных альбомов);  thumb_src — ссылка на
            изображение обложки альбома (если был указан параметр need_covers).   * — поля возвращаются только при
            запросе информации об альбомах текущего пользователя или альбомах администрируемых им сообществ
        """

        params = locals()
        method_name = self._base_method + 'getAlbums'
        return self._call(method_name, params)

    def getAlbumsCount(self, user_id: str = None, group_id: str = None) -> dict:
        """
        Возвращает количество доступных альбомов пользователя или сообщества.

        :param user_id: идентификатор пользователя, количество альбомов которого необходимо получить.
        :param group_id: идентификатор сообщества, количество альбомов которого необходимо получить.

        :return: После успешного выполнения возвращает количество альбомов с учетом настроек приватности
        """

        params = locals()
        method_name = self._base_method + 'getAlbumsCount'
        return self._call(method_name, params)

    def getAll(self, owner_id: str = None, extended: str = None, offset: str = None, count: str = None,
               photo_sizes: str = None, no_service_albums: str = None, need_hidden: str = None,
               skip_hidden: str = None) -> dict:
        """
        Возвращает все фотографии пользователя или сообщества в антихронологическом порядке.

        :param owner_id: идентификатор пользователя или сообщества, фотографии которого нужно получить. Обратите
            внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например,
            owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)
        :param extended: 1 — возвращать расширенную информацию о фотографиях.
        :param offset: смещение, необходимое для выборки определенного подмножества фотографий. По умолчанию — 0.
        :param count: число фотографий, информацию о которых необходимо получить.
        :param photo_sizes: 1 — будут возвращены размеры фотографий в специальном формате.
        :param no_service_albums: 0 — вернуть все фотографии, включая находящиеся в сервисных альбомах, таких как
            "Фотографии на моей стене" (по умолчанию);  1 — вернуть фотографии только из стандартных альбомов
            пользователя или сообщества.
        :param need_hidden: 1 — возвращает информацию от том, скрыта ли фотография из блока над стеной
            пользователя.
        :param skip_hidden: 1 — не возвращать фотографии, скрытые из блока над стеной пользователя (параметр
            учитывается только при owner_id > 0, параметр no_service_albums игнорируется).

        :return: После успешного выполнения возвращает список объектов photo.  Если был задан параметр
            extended=1, возвращаются дополнительные поля:   likes — количество отметок Мне нравится и информация о
            том, поставил ли лайк текущий пользователь;  reposts — число репостов фотографии.   Если был задан
            параметр photo_sizes=1, вместо полей width и height возвращаются размеры копий фотографии в специальном
            формате.  Если был задан параметр need_hidden=1, будет возвращено поле hidden = 1 в случае, если
            фотография скрыта из блока над стеной пользователя.  Если был задан параметр skip_hidden=1, для объекта
            ответа будет возвращено поле more, принимающее значения 1 или 0, обозначающее наличие или отсутствие
            следующих страниц с фотографиями пользователя. Также для объекта фотографии будет возвращено поле
            real_offset, обозначающее настоящую позицию фотографии из всех фотографий пользователя. Максимальное
            полученное значение этого поля необходимо передать в offset запроса для следующей страницы
        """

        params = locals()
        method_name = self._base_method + 'getAll'
        return self._call(method_name, params)

    def getAllComments(self, owner_id: str = None, album_id: str = None, need_likes: str = None, offset: str = None,
                       count: str = None) -> dict:
        """
        Возвращает отсортированный в антихронологическом порядке список всех комментариев к конкретному альбому или ко
        всем альбомам пользователя.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежат фотографии. Обратите
            внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например,
            owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)
        :param album_id: идентификатор альбома. Если параметр не задан, то считается, что необходимо получить
            комментарии ко всем альбомам пользователя или сообщества.
        :param need_likes: 1 — будет возвращено дополнительное поле likes. По умолчанию поле likes не
            возвращается.
        :param offset: смещение, необходимое для выборки определенного подмножества комментариев. Если параметр
            не задан, то считается, что он равен 0.
        :param count: количество комментариев, которое необходимо получить. Если параметр не задан, то считается
            что он равен 20. Максимальное значение параметра 100.  Обратите внимание, даже при использовании
            параметра offset для получения доступны только первые 10000 комментариев.

        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив
            объектов комментариев в поле items с дополнительным полем pid, содержащим идентификатор фотографии, к
            которой был оставлен комментарий.  Если был задан параметр need_likes=1, возвращается дополнительное поле
            likes:   count — число пользователей, которым понравился комментарий;  user_likes — наличие отметки «Мне
            нравится» от текущего пользователя  (1 — есть, 0 — нет);  can_like — информация о том, может ли текущий
            пользователь поставить отметку «Мне нравится»  (1 — может, 0 — не может)
        """

        params = locals()
        method_name = self._base_method + 'getAllComments'
        return self._call(method_name, params)

    def getById(self, photos: str = None, extended: str = None, photo_sizes: str = None) -> dict:
        """
        Возвращает информацию о фотографиях по их идентификаторам.

        :param photos: перечисленные через запятую идентификаторы, которые представляют собой идущие через знак
            подчеркивания id пользователей, разместивших фотографии, и id самих фотографий. Чтобы получить информацию
            о фотографии в альбоме группы, вместо id пользователя следует указать -id группы. Пример значения photos:
            1_129207899,6492_135055734,  -20629724_271945303  Некоторые фотографии, идентификаторы которых могут быть
            получены через API, закрыты приватностью, и не будут получены. В этом случае следует использовать ключ
            доступа фотографии (access_key) в её идентификаторе. Пример значения photos:
            1_129207899_220df2876123d3542f, 6492_135055734_e0a9bcc31144f67fbd  Поле access_key будет возвращено
            вместе с остальными данными фотографии в методах, которые возвращают фотографии, закрытые приватностью но
            доступные в данном контексте. Например данное поле имеют фотографии, возвращаемые методом newsfeed.get.
        :param extended: 1 — будут возвращены дополнительные поля likes, comments, tags, can_comment, can_repost.
            Поля comments и tags содержат только количество объектов. По умолчанию данные поля не возвращается.
        :param photo_sizes: возвращать ли доступные размеры фотографии в специальном формате.

        :return: После успешного выполнения возвращает массив объектов photo.  Если к фотографии прикреплено
            местоположение, также возвращаются поля lat и long, содержащие географические координаты отметки.  Если
            был задан параметр extended=1, возвращаются дополнительные поля:   likes — количество отметок Мне
            нравится и информация о том, поставил ли лайк текущий пользователь;  comments — количество комментариев к
            фотографии;  tags — количество отметок на фотографии;  can_comment — может ли текущий пользователь
            комментировать фото (1 — может, 0 — не может);  can_repost — может ли текущий пользователь сделать репост
            фотографии (1 — может, 0 — не может).   Если был задан параметр photo_sizes, вместо полей width и height
            возвращаются размеры копий фотографии в специальном формате
        """

        params = locals()
        method_name = self._base_method + 'getById'
        return self._call(method_name, params)

    def getChatUploadServer(self, chat_id: str = None, crop_x: str = None, crop_y: str = None,
                            crop_width: str = None) -> dict:
        """
        Позволяет получить адрес для загрузки обложки чата.

        :param chat_id: идентификатор беседы, для которой нужно загрузить фотографию.
        :param crop_x: координата x для обрезки фотографии (верхний правый угол).
        :param crop_y: координата y для обрезки фотографии (верхний правый угол).
        :param crop_width: ширина фотографии после обрезки в px.

        :return: После успешного выполнения возвращает объект с единственным полем upload_url
        """

        params = locals()
        method_name = self._base_method + 'getChatUploadServer'
        return self._call(method_name, params)

    def getComments(self, owner_id: str = None, photo_id: str = None, need_likes: str = None,
                    start_comment_id: str = None, offset: str = None, count: str = None, sort: str = None,
                    access_key: str = None, extended: str = None, fields: str = None) -> dict:
        """
        Возвращает список комментариев к фотографии.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. Обратите
            внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например,
            owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)
        :param photo_id: идентификатор фотографии.
        :param need_likes: 1 — будет возвращено дополнительное поле likes. По умолчанию: 0.
        :param start_comment_id: идентификатор комментария, начиная с которого нужно вернуть список (подробности
            см. ниже).
        :param offset: смещение, необходимое для выборки определенного подмножества комментариев. По умолчанию:
            0.
        :param count: количество комментариев, которое необходимо получить.
        :param sort: порядок сортировки комментариев. Возможные значения:   asc — от старых к новым;  desc — от
            новых к старым.
        :param access_key: ключ доступа к фотографии.
        :param extended: 1 — в ответе будут возвращены дополнительные поля profiles и groups, содержащие
            информацию о пользователях и сообществах. По умолчанию: 0.
        :param fields: список дополнительных полей профилей, которые необходимо вернуть. См. подробное описание.
            Доступные значения: photo_id, verified, sex, bdate, city, country, home_town, has_photo, photo_50,
            photo_100, photo_200_orig, photo_200, photo_400_orig, photo_max, photo_max_orig, online, lists, domain,
            has_mobile, contacts, site, education, universities, schools, status, last_seen, followers_count,
            common_count, occupation, nickname, relatives, relation, personal, connections, exports, wall_comments,
            activities, interests, music, movies, tv, books, games, about, quotes, can_post, can_see_all_posts,
            can_see_audio, can_write_private_message, can_send_friend_request, is_favorite, is_hidden_from_feed,
            timezone, screen_name, maiden_name, crop_photo, is_friend, friend_status, career, military, blacklisted,
            blacklisted_by_me.

        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив
            объектов комментариев в поле items.  Если был задан параметр extended=1, возвращает число результатов в
            поле count, отдельно массив объектов комментариев в поле items, пользователей в поле profiles и сообществ
            в поле groups.  Если был задан параметр need_likes=1, у объектов комментариев возвращается дополнительное
            поле likes:   count — число пользователей, которым понравился комментарий;  user_likes — наличие отметки
            «Мне нравится» от текущего пользователя  (1 — есть, 0 — нет);  can_like — информация о том, может ли
            текущий пользователь поставить отметку «Мне нравится»  (1 — может, 0 — не может).   Если был передан
            параметр start_comment_id, будет также возвращено поле real_offset – итоговое смещение данного
            подмножества комментариев (оно может быть отрицательным, если был указан отрицательный offset)
        """

        params = locals()
        method_name = self._base_method + 'getComments'
        return self._call(method_name, params)

    def getMarketAlbumUploadServer(self, group_id: str = None) -> dict:
        """
        Возвращает адрес сервера для загрузки фотографии подборки товаров в сообществе.

        :param group_id: идентификатор сообщества, для которого необходимо загрузить фотографию подборки товаров.

        :return: После успешного выполнения возвращает объект с единственным полем upload_url
        """

        params = locals()
        method_name = self._base_method + 'getMarketAlbumUploadServer'
        return self._call(method_name, params)

    def getMarketUploadServer(self, group_id: str = None, main_photo: str = None, crop_x: str = None,
                              crop_y: str = None, crop_width: str = None) -> dict:
        """
        Возвращает адрес сервера для загрузки фотографии товара.

        :param group_id: идентификатор сообщества, для которого необходимо загрузить фотографию товара.
        :param main_photo: является ли фотография обложкой товара (1 — фотография для обложки, 0 — дополнительная
            фотография)
        :param crop_x: координата x для обрезки фотографии (верхний правый угол).
        :param crop_y: координата y для обрезки фотографии (верхний правый угол).
        :param crop_width: ширина фотографии после обрезки в px.

        :return: После успешного выполнения возвращает объект с единственным полем upload_url
        """

        params = locals()
        method_name = self._base_method + 'getMarketUploadServer'
        return self._call(method_name, params)

    def getMessagesUploadServer(self, peer_id: str = None) -> dict:
        """
        Возвращает адрес сервера для загрузки фотографии в личное сообщение.

        :param peer_id: идентификатор назначения (для загрузки фотографии в сообщениях сообществ).

        :return: После успешного выполнения возвращает объект с полями upload_url, album_id (id альбома),
            group_id (идентификатор сообщества, если используются сообщения сообщества)
        """

        params = locals()
        method_name = self._base_method + 'getMessagesUploadServer'
        return self._call(method_name, params)

    def getNewTags(self, offset: str = None, count: str = None) -> dict:
        """
        Возвращает список фотографий, на которых есть непросмотренные отметки.

        :param offset: смещение, необходимое для получения определённого подмножества фотографий.
        :param count: количество фотографий, которые необходимо вернуть.

        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив
            объектов фотографий в поле items.  Объекты фотографий содержат дополнительные поля:   placer_id —
            идентификатор пользователя, сделавшего отметку;  tag_created — дата создания отметки в формате unixtime;
            tag_id — идентификатор отметки
        """

        params = locals()
        method_name = self._base_method + 'getNewTags'
        return self._call(method_name, params)

    def getOwnerCoverPhotoUploadServer(self, group_id: str = None, crop_x: str = None, crop_y: str = None,
                                       crop_x2: str = None, crop_y2: str = None) -> dict:
        """
        Получает адрес для загрузки обложки сообщества.

        :param group_id: идентификатор сообщества.
        :param crop_x: координата X верхнего левого угла для обрезки изображения.
        :param crop_y: координата Y верхнего левого угла для обрезки изображения.
        :param crop_x2: координата X нижнего правого угла для обрезки изображения.
        :param crop_y2: координата Y нижнего правого угла для обрезки изображения.

        :return: Возвращает объект с единственным полем upload_url, содержащим адрес сервера для загрузки фото
        """

        params = locals()
        method_name = self._base_method + 'getOwnerCoverPhotoUploadServer'
        return self._call(method_name, params)

    def getOwnerPhotoUploadServer(self, owner_id: str = None) -> dict:
        """
        Возвращает адрес сервера для загрузки главной фотографии на страницу пользователя или сообщества.

        :param owner_id: идентификатор сообщества или текущего пользователя. Обратите внимание, идентификатор
            сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует
            идентификатору сообщества ВКонтакте API (club1)

        :return: После успешного выполнения возвращает объект с единственным полем upload_url
        """

        params = locals()
        method_name = self._base_method + 'getOwnerPhotoUploadServer'
        return self._call(method_name, params)

    def getTags(self, owner_id: str = None, photo_id: str = None, access_key: str = None) -> dict:
        """
        Возвращает список отметок на фотографии.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. Обратите
            внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например,
            owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)
        :param photo_id: идентификатор фотографии.
        :param access_key: строковой ключ доступа, который может быть получен при получении объекта фотографии.

        :return: После успешного выполнения возвращает массив объектов tag, каждый из которых содержит следующие
            поля:   user_id — идентификатор пользователя, которому соответствует отметка;  id — идентификатор
            отметки;  placer_id — идентификатор пользователя, сделавшего отметку;  tagged_name — название отметки;
            date — дата добавления отметки в формате unixtime;  x, y, x2, y2 — координаты прямоугольной области, на
            которой сделана отметка (верхний левый угол и нижний правый угол) в процентах;  viewed — статус отметки
            (1 — подтвержденная, 0 — неподтвержденная)
        """

        params = locals()
        method_name = self._base_method + 'getTags'
        return self._call(method_name, params)

    def getUploadServer(self, album_id: str = None, group_id: str = None) -> dict:
        """
        Возвращает адрес сервера для загрузки фотографий.

        :param album_id: идентификатор альбома.
        :param group_id: идентификатор сообщества, которому принадлежит альбом (если необходимо загрузить
            фотографию в альбом сообщества).  Если group_id не указан, возвращается адрес для загрузки на стену
            текущего пользователя.

        :return: После успешного выполнения возвращает объект, содержащий следующие поля:   upload_url — адрес
            для загрузки фотографий;  album_id — идентификатор альбома, в который будет загружена фотография;
            user_id — идентификатор пользователя, от чьего имени будет загружено фото
        """

        params = locals()
        method_name = self._base_method + 'getUploadServer'
        return self._call(method_name, params)

    def getUserPhotos(self, user_id: str = None, offset: str = None, count: str = None, extended: str = None,
                      sort: str = None) -> dict:
        """
        Возвращает список фотографий, на которых отмечен пользователь

        :param user_id: идентификатор пользователя, список фотографий для которого нужно получить.
        :param offset: смещение, необходимое для выборки определенного подмножества фотографий. По умолчанию — 0.
        :param count: количество фотографий, которое необходимо получить.
        :param extended: 1 — будут возвращены дополнительные поля likes, comments, tags, can_comment. Поля
            comments и tags содержат только количество объектов. По умолчанию данные поля не возвращается.
        :param sort: сортировка результатов (0 — по дате добавления отметки в порядке убывания, 1 — по дате
            добавления отметки в порядке возрастания).

        :return: После успешного выполнения возвращает список объектов photo.  Если был задан параметр
            extended=1, возвращаются дополнительные поля:   likes — количество отметок Мне нравится и информация о
            том, поставил ли лайк текущий пользователь;  comments — количество комментариев к фотографии;  tags —
            количество отметок на фотографии;  can_comment — может ли текущий пользователь комментировать фото (1 —
            может, 0 — не может).   Если был задан параметр photo_sizes=1, вместо полей width и height возвращаются
            размеры копий фотографии в специальном формате
        """

        params = locals()
        method_name = self._base_method + 'getUserPhotos'
        return self._call(method_name, params)

    def getWallUploadServer(self, group_id: str = None) -> dict:
        """
        Возвращает адрес сервера для загрузки фотографии на стену пользователя или сообщества.

        :param group_id: идентификатор сообщества, на стену которого нужно загрузить фото (без знака «минус»).

        :return: После успешного выполнения возвращает объект с полями upload_url, album_id, user_id
        """

        params = locals()
        method_name = self._base_method + 'getWallUploadServer'
        return self._call(method_name, params)

    def makeCover(self, owner_id: str = None, photo_id: str = None, album_id: str = None) -> dict:
        """
        Делает фотографию обложкой альбома.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. Обратите
            внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например,
            owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)
        :param photo_id: идентификатор фотографии. Фотография должна находиться в альбоме album_id.
        :param album_id: идентификатор альбома.

        :return: После успешного выполнения возвращает 1
        """

        params = locals()
        method_name = self._base_method + 'makeCover'
        return self._call(method_name, params)

    def move(self, owner_id: str = None, target_album_id: str = None, photo_id: str = None) -> dict:
        """
        Переносит фотографию из одного альбома в другой.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. Обратите
            внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например,
            owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)
        :param target_album_id: идентификатор альбома, в который нужно переместить фотографию.
        :param photo_id: идентификатор фотографии, которую нужно перенести.

        :return: После успешного выполнения возвращает 1
        """

        params = locals()
        method_name = self._base_method + 'move'
        return self._call(method_name, params)

    def putTag(self, owner_id: str = None, photo_id: str = None, user_id: str = None, x: str = None, y: str = None,
               x2: str = None, y2: str = None) -> dict:
        """
        Добавляет отметку на фотографию.

        :param owner_id: идентификатор пользователя, которому принадлежит фотография.
        :param photo_id: идентификатор фотографии.
        :param user_id: идентификатор пользователя, которого нужно отметить на фотографии.
        :param x: координата верхнего левого угла области с отметкой в % от ширины фотографии.
        :param y: координата верхнего левого угла области с отметкой в % от высоты фотографии.
        :param x2: координата правого нижнего угла области с отметкой в % от ширины фотографии.
        :param y2: координата правого нижнего угла области с отметкой в % от высоты фотографии.

        :return: После успешного выполнения возвращает идентификатор созданной отметки (tag id)
        """

        params = locals()
        method_name = self._base_method + 'putTag'
        return self._call(method_name, params)

    def removeTag(self, owner_id: str = None, photo_id: str = None, tag_id: str = None) -> dict:
        """
        Удаляет отметку с фотографии.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. Обратите
            внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например,
            owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)
        :param photo_id: идентификатор фотографии.
        :param tag_id: идентификатор отметки.

        :return: После успешного выполнения возвращает 1
        """

        params = locals()
        method_name = self._base_method + 'removeTag'
        return self._call(method_name, params)

    def reorderAlbums(self, owner_id: str = None, album_id: str = None, before: str = None, after: str = None) -> dict:
        """
        Меняет порядок альбома в списке альбомов пользователя.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит альбом. Обратите
            внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например,
            owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)
        :param album_id: идентификатор альбома.
        :param before: идентификатор альбома, перед которым следует поместить альбом.
        :param after: идентификатор альбома, после которого следует поместить альбом.

        :return: После успешного выполнения возвращает 1
        """

        params = locals()
        method_name = self._base_method + 'reorderAlbums'
        return self._call(method_name, params)

    def reorderPhotos(self, owner_id: str = None, photo_id: str = None, before: str = None, after: str = None) -> dict:
        """
        Меняет порядок фотографии в списке фотографий альбома пользователя.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. Обратите
            внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например,
            owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)
        :param photo_id: идентификатор фотографии.
        :param before: идентификатор фотографии, перед которой следует поместить фотографию. Если параметр не
            указан, фотография будет помещена последней.
        :param after: идентификатор фотографии, после которой следует поместить фотографию. Если параметр не
            указан, фотография будет помещена первой.

        :return: После успешного выполнения возвращает 1
        """

        params = locals()
        method_name = self._base_method + 'reorderPhotos'
        return self._call(method_name, params)

    def report(self, owner_id: str = None, photo_id: str = None, reason: str = None) -> dict:
        """
        Позволяет пожаловаться на фотографию.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография.
        :param photo_id: идентификатор фотографии.
        :param reason: причина жалобы:   0 — спам;  1 — детская порнография;  2 — экстремизм;  3 — насилие;  4 —
            пропаганда наркотиков;  5 — материал для взрослых;  6 — оскорбление.

        :return: После успешного выполнения возвращает 1
        """

        params = locals()
        method_name = self._base_method + 'report'
        return self._call(method_name, params)

    def reportComment(self, owner_id: str = None, comment_id: str = None, reason: str = None) -> dict:
        """
        Позволяет пожаловаться на комментарий к фотографии.

        :param owner_id: идентификатор владельца фотографии к которой оставлен комментарий.
        :param comment_id: идентификатор комментария.
        :param reason: причина жалобы:   0 — спам;  1 — детская порнография;  2 — экстремизм;  3 — насилие;  4 —
            пропаганда наркотиков;  5 — материал для взрослых;  6 — оскорбление.

        :return: После успешного выполнения возвращает 1
        """

        params = locals()
        method_name = self._base_method + 'reportComment'
        return self._call(method_name, params)

    def restore(self, owner_id: str = None, photo_id: str = None) -> dict:
        """
        Восстанавливает удаленную фотографию.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. Обратите
            внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например,
            owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)
        :param photo_id: идентификатор фотографии.

        :return: После успешного выполнения возвращает 1
        """

        params = locals()
        method_name = self._base_method + 'restore'
        return self._call(method_name, params)

    def restoreComment(self, owner_id: str = None, comment_id: str = None) -> dict:
        """
        Восстанавливает удаленный комментарий к фотографии.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. Обратите
            внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например,
            owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)
        :param comment_id: идентификатор удаленного комментария.

        :return: После успешного выполнения возвращает 1 (0, если комментарий с таким идентификатором не является
            удаленным)
        """

        params = locals()
        method_name = self._base_method + 'restoreComment'
        return self._call(method_name, params)

    def save(self, album_id: str = None, group_id: str = None, server: str = None, photos_list: str = None,
             hash: str = None, latitude: str = None, longitude: str = None, caption: str = None) -> dict:
        """
        Сохраняет фотографии после успешной загрузки.

        :param album_id: идентификатор альбома, в который необходимо сохранить фотографии.
        :param group_id: идентификатор сообщества, в которое необходимо сохранить фотографии.
        :param server: параметр, возвращаемый в результате загрузки фотографий на сервер.
        :param photos_list: параметр, возвращаемый в результате загрузки фотографий на сервер.
        :param hash: параметр, возвращаемый в результате загрузки фотографий на сервер.
        :param latitude: географическая широта, заданная в градусах (от -90 до 90);
        :param longitude: географическая долгота, заданная в градусах (от -180 до 180);
        :param caption: текст описания фотографии (максимум 2048 символов).

        :return: После успешного выполнения возвращает массив объектов фотографий
        """

        params = locals()
        method_name = self._base_method + 'save'
        return self._call(method_name, params)

    def saveMarketAlbumPhoto(self, group_id: str = None, photo: str = None, server: str = None,
                             hash: str = None) -> dict:
        """
        Сохраняет фотографии после успешной загрузки на URI, полученный методом photos.getMarketAlbumUploadServer.

        :param group_id: идентификатор группы, для которой нужно загрузить фотографию.
        :param photo: параметр, возвращаемый в результате загрузки фотографии на сервер.  Минимальный размер
            фотографии — 1280x720 пикселей.
        :param server: параметр, возвращаемый в результате загрузки фотографии на сервер.
        :param hash: параметр, возвращаемый в результате загрузки фотографии на сервер.

        :return: После успешного выполнения возвращает массив, содержащий объект с загруженной фотографией
        """

        params = locals()
        method_name = self._base_method + 'saveMarketAlbumPhoto'
        return self._call(method_name, params)

    def saveMarketPhoto(self, group_id: str = None, photo: str = None, server: str = None, hash: str = None,
                        crop_data: str = None, crop_hash: str = None) -> dict:
        """
        Сохраняет фотографии после успешной загрузки на URI, полученный методом photos.getMarketUploadServer.

        :param group_id: идентификатор группы, для которой нужно загрузить фотографию.
        :param photo: параметр, возвращаемый в результате загрузки фотографии на сервер.
        :param server: параметр, возвращаемый в результате загрузки фотографии на сервер.
        :param hash: параметр, возвращаемый в результате загрузки фотографии на сервер.
        :param crop_data: параметр, возвращаемый в результате загрузки фотографии на сервер.
        :param crop_hash: параметр, возвращаемый в результате загрузки фотографии на сервер.

        :return: После успешного выполнения возвращает массив, содержащий объект с загруженной фотографией
        """

        params = locals()
        method_name = self._base_method + 'saveMarketPhoto'
        return self._call(method_name, params)

    def saveMessagesPhoto(self, photo: str = None, server: str = None, hash: str = None) -> dict:
        """
        Сохраняет фотографию после успешной загрузки на URI, полученный методом photos.getMessagesUploadServer.

        :param photo: параметр, возвращаемый в результате загрузки фотографии на сервер.
        :param server: параметр, возвращаемый в результате загрузки фотографии на сервер.
        :param hash: параметр, возвращаемый в результате загрузки фотографии на сервер.

        :return: После успешного выполнения возвращает массив с загруженной фотографией, возвращённый объект
            имеет поля id, pid, aid, owner_id, src, src_big, src_small, created. В случае наличия фотографий в
            высоком разрешении также будут возвращены адреса с названиями src_xbig и src_xxbig
        """

        params = locals()
        method_name = self._base_method + 'saveMessagesPhoto'
        return self._call(method_name, params)

    def saveOwnerCoverPhoto(self, hash: str = None, photo: str = None) -> dict:
        """
        Сохраняет изображение для обложки сообщества после успешной загрузки.

        :param hash: параметр hash, полученный в результате загрузки фотографии на сервер.
        :param photo: параметр photo, полученный в результате загрузки фотографии на сервер.

        :return: Возвращает массив images объектов, описывающих копии загруженной фотографии. Каждый объект
            содержит следующие поля:   url (string) — URL копии фотографии;  width (integer) — ширина копии в px;
            height (integer) — высота копии в px
        """

        params = locals()
        method_name = self._base_method + 'saveOwnerCoverPhoto'
        return self._call(method_name, params)

    def saveOwnerPhoto(self, server: str = None, hash: str = None, photo: str = None) -> dict:
        """
        Позволяет сохранить главную фотографию пользователя или сообщества.

        :param server: параметр, возвращаемый в результате загрузки фотографии на сервер.
        :param hash: параметр, возвращаемый в результате загрузки фотографии на сервер.
        :param photo: параметр, возвращаемый в результате загрузки фотографии на сервер.

        :return: После успешного выполнения возвращает объект, содержащий поля photo_hash и photo_src (при работе
            через VK.api метод вернёт поля photo_src, photo_src_big, photo_src_small). Параметр photo_hash необходим
            для подтверждения пользователем изменения его фотографии через вызов метода saveProfilePhoto Javascript
            API. Поле photo_src содержит путь к загруженной фотографии
        """

        params = locals()
        method_name = self._base_method + 'saveOwnerPhoto'
        return self._call(method_name, params)

    def saveWallPhoto(self, user_id: str = None, group_id: str = None, photo: str = None, server: str = None,
                      hash: str = None, latitude: str = None, longitude: str = None, caption: str = None) -> dict:
        """
        Сохраняет фотографии после успешной загрузки на URI, полученный методом photos.getWallUploadServer.

        :param user_id: идентификатор пользователя, на стену которого нужно сохранить фотографию.
        :param group_id: идентификатор сообщества, на стену которого нужно сохранить фотографию.
        :param photo: параметр, возвращаемый в результате загрузки фотографии на сервер.
        :param server: параметр, возвращаемый в результате загрузки фотографии на сервер.
        :param hash: параметр, возвращаемый в результате загрузки фотографии на сервер.
        :param latitude: географическая широта, заданная в градусах (от -90 до 90);
        :param longitude: географическая долгота, заданная в градусах (от -180 до 180);
        :param caption: текст описания фотографии (максимум 2048 символов).

        :return: После успешного выполнения возвращает массив, содержащий объект с загруженной фотографией
        """

        params = locals()
        method_name = self._base_method + 'saveWallPhoto'
        return self._call(method_name, params)

    def search(self, q: str = None, lat: str = None, long: str = None, start_time: str = None, end_time: str = None,
               sort: str = None, offset: str = None, count: str = None, radius: str = None) -> dict:
        """
        Осуществляет поиск изображений по местоположению или описанию.

        :param q:  поискового запроса, например, Nature.
        :param lat: географическая широта отметки, заданная в градусах (от -90 до 90).
        :param long: географическая долгота отметки, заданная в градусах (от -180 до 180).
        :param start_time: время в формате unixtime, не раньше которого должны были быть загружены найденные
            фотографии.
        :param end_time: время в формате unixtime, не позже которого должны были быть загружены найденные
            фотографии.
        :param sort: сортировка результатов. Возможные значения:   1 — по количеству отметок «Мне нравится»;  0 —
            по дате добавления фотографии.
        :param offset: смещение относительно первой найденной фотографии для выборки определенного подмножества.
        :param count: количество возвращаемых фотографий.  Обратите внимание — даже при использовании параметра
            offset для получения информации доступны не больше 3000 результатов.
        :param radius: радиус поиска в метрах. (работает очень приближенно, поэтому реальное расстояние до цели
            может отличаться от заданного). Может принимать значения: 10, 100, 800, 6000, 50000

        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив
            объектов фотографий в поле items
        """

        params = locals()
        method_name = self._base_method + 'search'
        return self._call(method_name, params)
