"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Stage = exports.MethodLoggingLevel = void 0;
const core_1 = require("@aws-cdk/core");
const access_log_1 = require("./access-log");
const apigateway_generated_1 = require("./apigateway.generated");
const util_1 = require("./util");
/**
 *
 */
var MethodLoggingLevel;
(function (MethodLoggingLevel) {
    MethodLoggingLevel["OFF"] = "OFF";
    MethodLoggingLevel["ERROR"] = "ERROR";
    MethodLoggingLevel["INFO"] = "INFO";
})(MethodLoggingLevel = exports.MethodLoggingLevel || (exports.MethodLoggingLevel = {}));
/**
 *
 */
class Stage extends core_1.Resource {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.enableCacheCluster = props.cacheClusterEnabled;
        const methodSettings = this.renderMethodSettings(props); // this can mutate `this.cacheClusterEnabled`
        // custom access logging
        let accessLogSetting;
        const accessLogDestination = props.accessLogDestination;
        const accessLogFormat = props.accessLogFormat;
        if (!accessLogDestination && !accessLogFormat) {
            accessLogSetting = undefined;
        }
        else {
            if (accessLogFormat !== undefined &&
                !core_1.Token.isUnresolved(accessLogFormat.toString()) &&
                !/.*\$context.requestId.*/.test(accessLogFormat.toString())) {
                throw new Error('Access log must include at least `AccessLogFormat.contextRequestId()`');
            }
            if (accessLogFormat !== undefined && accessLogDestination === undefined) {
                throw new Error('Access log format is specified without a destination');
            }
            accessLogSetting = {
                destinationArn: accessLogDestination === null || accessLogDestination === void 0 ? void 0 : accessLogDestination.bind(this).destinationArn,
                format: (accessLogFormat === null || accessLogFormat === void 0 ? void 0 : accessLogFormat.toString()) ? accessLogFormat === null || accessLogFormat === void 0 ? void 0 : accessLogFormat.toString() : access_log_1.AccessLogFormat.clf().toString(),
            };
        }
        // enable cache cluster if cacheClusterSize is set
        if (props.cacheClusterSize !== undefined) {
            if (this.enableCacheCluster === undefined) {
                this.enableCacheCluster = true;
            }
            else if (this.enableCacheCluster === false) {
                throw new Error(`Cannot set "cacheClusterSize" to ${props.cacheClusterSize} and "cacheClusterEnabled" to "false"`);
            }
        }
        const cacheClusterSize = this.enableCacheCluster ? (props.cacheClusterSize || '0.5') : undefined;
        const resource = new apigateway_generated_1.CfnStage(this, 'Resource', {
            stageName: props.stageName || 'prod',
            accessLogSetting,
            cacheClusterEnabled: this.enableCacheCluster,
            cacheClusterSize,
            clientCertificateId: props.clientCertificateId,
            deploymentId: props.deployment.deploymentId,
            restApiId: props.deployment.api.restApiId,
            description: props.description,
            documentationVersion: props.documentationVersion,
            variables: props.variables,
            tracingEnabled: props.tracingEnabled,
            methodSettings,
        });
        this.stageName = resource.ref;
        this.restApi = props.deployment.api;
    }
    /**
     * Returns the invoke URL for a certain path.
     *
     * @param path The resource path.
     */
    urlForPath(path = '/') {
        if (!path.startsWith('/')) {
            throw new Error(`Path must begin with "/": ${path}`);
        }
        return `https://${this.restApi.restApiId}.execute-api.${core_1.Stack.of(this).region}.${core_1.Stack.of(this).urlSuffix}/${this.stageName}${path}`;
    }
    renderMethodSettings(props) {
        const settings = new Array();
        const self = this;
        // extract common method options from the stage props
        const commonMethodOptions = {
            metricsEnabled: props.metricsEnabled,
            loggingLevel: props.loggingLevel,
            dataTraceEnabled: props.dataTraceEnabled,
            throttlingBurstLimit: props.throttlingBurstLimit,
            throttlingRateLimit: props.throttlingRateLimit,
            cachingEnabled: props.cachingEnabled,
            cacheTtl: props.cacheTtl,
            cacheDataEncrypted: props.cacheDataEncrypted,
        };
        // if any of them are defined, add an entry for '/*/*'.
        const hasCommonOptions = Object.keys(commonMethodOptions).map(v => commonMethodOptions[v]).filter(x => x).length > 0;
        if (hasCommonOptions) {
            settings.push(renderEntry('/*/*', commonMethodOptions));
        }
        if (props.methodOptions) {
            for (const path of Object.keys(props.methodOptions)) {
                settings.push(renderEntry(path, props.methodOptions[path]));
            }
        }
        return settings.length === 0 ? undefined : settings;
        function renderEntry(path, options) {
            if (options.cachingEnabled) {
                if (self.enableCacheCluster === undefined) {
                    self.enableCacheCluster = true;
                }
                else if (self.enableCacheCluster === false) {
                    throw new Error(`Cannot enable caching for method ${path} since cache cluster is disabled on stage`);
                }
            }
            const { httpMethod, resourcePath } = util_1.parseMethodOptionsPath(path);
            return {
                httpMethod,
                resourcePath,
                cacheDataEncrypted: options.cacheDataEncrypted,
                cacheTtlInSeconds: options.cacheTtl && options.cacheTtl.toSeconds(),
                cachingEnabled: options.cachingEnabled,
                dataTraceEnabled: options.dataTraceEnabled,
                loggingLevel: options.loggingLevel,
                metricsEnabled: options.metricsEnabled,
                throttlingBurstLimit: options.throttlingBurstLimit,
                throttlingRateLimit: options.throttlingRateLimit,
            };
        }
    }
}
exports.Stage = Stage;
//# sourceMappingURL=data:application/json;base64,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