# **************************************************************************
# *
# * Authors:     J.M. De la Rosa Trevin (delarosatrevin@scilifelab.se) [1]
# *              Based on script from Javi Chichon
# *
# * [1] SciLifeLab, Stockholm University
# *
# * This program is free software; you can redistribute it and/or modify
# * it under the terms of the GNU General Public License as published by
# * the Free Software Foundation; either version 2 of the License, or
# * (at your option) any later version.
# *
# * This program is distributed in the hope that it will be useful,
# * but WITHOUT ANY WARRANTY; without even the implied warranty of
# * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# * GNU General Public License for more details.
# *
# * You should have received a copy of the GNU General Public License
# * along with this program; if not, write to the Free Software
# * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
# * 02111-1307  USA
# *
# *  All comments concerning this program package may be sent to the
# *  e-mail address 'scipion@cnb.csic.es'
# *
# **************************************************************************

import sys
import os
import argparse


parser = argparse.ArgumentParser(
    description="Create movie stacks from the individual "
                "frame files.")
add = parser.add_argument  # shortcut
add('inputTs', metavar='INPUT_STACK',
    help='Input tilt series stack file. ')

add('--widthz', type=int, default=500,
    help='Estimated witdh in Z. ')

add('--bin', type=int, default=2,
    help='Bin factor.')

add('--rotation_angle', type=float, default=0.0,
    help='Angle from the vertical to the tilt axis in raw images.')

add('--raptor', action='store_true',
    help='Try to automatically track fiducials with RAPTOR ')
add('--markers_diameter', type=int, default=20, metavar='VALUE',
    help='Estimated diameter of fiducial markers. ')
add('--markers_number', type=int, default=20, metavar='VALUE',
    help='Number of markers to track. ')

add('--output', default='',
    help='Define the output tomogram name. If not provided,'
         'the basename from the input will be used + "_tomo.mrc". ')

args = parser.parse_args()


def _getVar(key):
    """ Retrieve an environment variable. """
    value = os.environ.get(key, None)

    if value is None:
        raise Exception("%s var should be defined" % key)

    return value


IMOD_DIR = _getVar('IMOD_DIR')
TOMO3D_DIR = _getVar('TOMO3D_DIR')

IMOD_BIN = os.path.join(IMOD_DIR, 'bin')
TOMO3D_BIN = os.path.join(TOMO3D_DIR, 'bin')


name = os.path.splitext(os.path.basename(args.inputTs))[0]
fidFile = '%s.fid' % name
outputTomo = args.output or name + '_tomo.mrc'

argsDict = {
    'input': args.inputTs,
    'name': name,
    'fidFile': fidFile,
    'IMOD_BIN': IMOD_BIN,
    'rawtlt': name + '.rawtlt',  # this is required, name like this for RAPTOR
    'tomoName': outputTomo,
    'outRaptor': 'outRAPTOR',
    'markersDiameter': args.markers_diameter,
    'markersNumber': args.markers_number,
    'binFactor': args.bin,
    'zWidth': args.widthz,
    'rotationAngle': args.rotation_angle
}


def run(program, args):
    cmd = "%s %s" % (program, args % argsDict)
    print("\n>>>  %s" % cmd)
    retCode = os.system(cmd)
    # if retCode:
    #     raise Exception("An error occurred when running command: %s" % cmd)


def header(msg):
    """ Simply print some noticeable header message. """
    line = "-" * 80 + '\n' + '-' * 80
    print('\n' + line)
    print(' ' * 20 + msg)
    print(line + '\n')


# Generate file with transforms used to align images for patch tracking (*prefx)
run('tiltxcorr',
    '-input %(input)s -output %(name)s.prexf -tiltfile %(rawtlt)s -rotation 0.0 '
    '-sigma1 0.03 -radius2 0.25 -sigma2 0.05')

header('STEP 1: PREALIGNMENT FILE GENERATION')
# Generate preali
run('xftoxg',
    '-input %(name)s.prexf -nfit 0 -goutput %(name)s.prexg')
run('newstack',
    '-input %(input)s -output %(name)s.preali -mode 0 -xform %(name)s.prexg -float 2')

if args.raptor:
    print("Using RAPTOR to align the tilt-series")
    run("RAPTOR",
        '-execPath %(IMOD_BIN)s -path . -input %(name)s.preali -tracking -output %(outRaptor)s '
        '-diameter %(markersDiameter)s -markers %(markersNumber)s ')
    run('cp', '%(outRaptor)s/IMOD/%(name)s.fid.txt %(fidFile)s')
    fidErr = "Fiducial file was not properly generated by RAPTOR"
else:
    fidErr = "Should provide input fiducial file"

if not os.path.exists(fidFile):
    raise Exception("Missing fiducials file\n   Expecting: %s" % fidFile)

header('STEP 2: alignment FILE GENERATION')
# Generate ali: tiltalign (align using model); xfproduct (Forms product of two lists of transformations)
run('tiltalign',
    '-ModelFile %(fidFile)s -ImageFile %(name)s.preali -OutputTiltFile %(name)s.tlt '
    '-OutputTransformFile %(name)s.tltxf -RotationAngle %(rotationAngle)s -tiltfile %(rawtlt)s -AngleOffset 0.0 '
    '-RotOption -1 -RotDefaultGrouping 5 -TiltOption 0 -MagReferenceView 1 -MagOption 0 -MagDefaultGrouping 4 '
    '-XStretchOption 0 -XStretchDefaultGrouping 7 -SkewOption 0 -SkewDefaultGrouping 11 -ResidualReportCriterion 3.0 '
    '-SurfacesToAnalyze 1 -MetroFactor 0.25 -MaximumCycles 1000 -AxisZShift 0.0 -LocalAlignments 0 '
    '-MinFidsTotalAndEachSurface 8,3 -LocalOutputOptions 1,0,1 -LocalRotOption 3 -LocalRotDefaultGrouping 6 '
    '-LocalTiltOption 5 -LocalTiltDefaultGrouping 6 -LocalMagReferenceView 1 -LocalMagOption 3 '
    '-LocalMagDefaultGrouping 7 -LocalXStretchOption 0 -LocalXStretchDefaultGrouping 7 -LocalSkewOption 0 '
    '-LocalSkewDefaultGrouping 11 -BeamTiltOption 0')
run('xfproduct',
    '-in1 %(name)s.prexg -in2 %(name)s.tltxf -output %(name)s_fid.xf')
run('cp',
    '%(name)s_fid.xf %(name)s.xf')
run('newstack',
    '-input %(input)s -output %(name)s.ali -offset 0,0 -xform %(name)s.xf -origin -taper 1,0')
header('ALIGNMENT DONE.')

header('STEP3: Starting RECONSTRUCTION')
run('newstack', '-bin %(binFactor)s %(name)s.ali %(name)s.ali')
run(os.path.join(TOMO3D_BIN, 'tomo3d'),
    '-o %(tomoName)s -t 8 -z %(zWidth)s -a %(name)s.tlt -i %(name)s.ali -S -l 30')
run('trimvol', '-yz %(tomoName)s %(tomoName)s')

# Some clean up
run('rm', '*~')

