# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['textfsm_aos', 'textfsm_aos.templates']

package_data = \
{'': ['*']}

install_requires = \
['PyYAML>=6.0,<7.0', 'textfsm>=1.1.2,<2.0.0']

setup_kwargs = {
    'name': 'textfsm-aos',
    'version': '1.0.0',
    'description': 'Alcatel-Lucent Enterprise AOS CLI parsing (TextFSM)',
    'long_description': '# TEXTFSM-AOS\n\n> Alcatel-Lucent Enterprise AOS CLI parsing\n\nPython package for Alcatel-Lucent Enterprise aos6 and aos8 parsing based on TextFSM templates.\n\n## Why TextFSM-AOS?\n\nParse semi-structured cli data to structured data ready to be ingested by your network automation pipeline. Autmatically transform gathered output from screen-scraping tools like Netmiko, Scrapli and Paramiko. Receive uniform data accross Alcatel-Lucent Enterprise devices running aos6 or aos8.\n\n## Installing / Getting started\n\nTextfsm-aos can be installed using Git + Poetry or PyPI.\n\n## Git\n\n```bash\ngit clone https://github.com/jefvantongerloo/textfsm-aos\npoetry install\n```\n\n## PyPI\n\n```bash\npip install textfsm-aos\n```\n\nProvide screen-scraped data to parser\n\n```python\nfrom textfsm_aos.parser import parse\n\nsample_data = """\n   Package           Release       Size     Description\n-----------------+---------------+--------+-----------------------------------\nKFbase.img        6.7.2.89.R06    18059551 Alcatel-Lucent Enterprise Base Softw\nKFos.img          6.7.2.89.R06     3566798 Alcatel-Lucent Enterprise OS\nKFeni.img         6.7.2.89.R06     6123991 Alcatel-Lucent Enterprise NI softwar\nKFsecu.img        6.7.2.89.R06      649383 Alcatel-Lucent Enterprise Security M\n"""\n\nparse("ale_aos6", "show microcode", sample_data)\n```\n\nparsed result\n\n```python\n[\n   {\n      "package":"KFbase.img",\n      "release":"6.7.2.89.R06",\n      "size":"18059551",\n      "description":"Alcatel-Lucent Enterprise Base Softw"\n   },\n   {\n      "package":"KFos.img",\n      "release":"6.7.2.89.R06",\n      "size":"3566798",\n      "description":"Alcatel-Lucent Enterprise OS"\n   },\n   {\n      "package":"KFeni.img",\n      "release":"6.7.2.89.R06",\n      "size":"6123991",\n      "description":"Alcatel-Lucent Enterprise NI softwar"\n   },\n   {\n      "package":"KFsecu.img",\n      "release":"6.7.2.89.R06",\n      "size":"649383",\n      "description":"Alcatel-Lucent Enterprise Security M"\n   }\n]\n```\n\n## Supported commands\n\n| command                        |                 aos6                |                aos8               |\n|--------------------------------|:-----------------------------------:|:---------------------------------:|\n| history                        |        `alias: show history`        |         :heavy_check_mark:        |\n| show 802.1x users              |          :heavy_check_mark:         |       `alias: show unp user`      |\n| show 802.1x users unp          |          :heavy_check_mark:         |                :x:                |\n| show 802.1x non-supplicant     |          :heavy_check_mark:         |                :x:                |\n| show 802.1x non-supplicant unp |          :heavy_check_mark:         |                :x:                |\n| show chassis                   |          :heavy_check_mark:         |         :heavy_check_mark:        |\n| show cmm                       |                 :x:                 |         :heavy_check_mark:        |\n| show command-log               |          :heavy_check_mark:         |         :heavy_check_mark:        |\n| show hardware-info             |                 :x:                 |         :heavy_check_mark:        |\n| show health                    |          :heavy_check_mark:         |         :heavy_check_mark:        |\n| show history                   |          :heavy_check_mark:         |          `alias: history`         |\n| show interface status          |          :heavy_check_mark:         |         :heavy_check_mark:        |\n| show ip interface              |          :heavy_check_mark:         |         :heavy_check_mark:        |\n| show ip route                  |          :heavy_check_mark:         |       `alias: show ip routes`     |\n| show ip routes                 |        `alias: show ip route`       |         :heavy_check_mark:        |\n| show lld remote system         |          :heavy_check_mark:         |                :x:                |\n| show log events                |                 :x:                 |         :heavy_check_mark:        |\n| show mac-address-table         |          :heavy_check_mark:         |     `alias: show mac-learning`    |\n| show mac-learning              |   `alias: show mac-address-table`   |         :heavy_check_mark:        |\n| show microcode                 |          :heavy_check_mark:         |         :heavy_check_mark:        |\n| show ntp server status         |          :heavy_check_mark:         |         :heavy_check_mark:        |\n| show unp user                  |                 :x:                 |         :heavy_check_mark:        |\n| show user                      |          :heavy_check_mark:         |         :heavy_check_mark:        |\n| show running-directory         |                 :x:                 |         :heavy_check_mark:        |\n| show snmp station              |          :heavy_check_mark:         |         :heavy_check_mark:        |\n| show snmp community map        |          :heavy_check_mark:         |  `alias: show snmp community-map` |\n| show snmp community-map        |   `alias: show snmp community map`  |         :heavy_check_mark:        |\n| show system                    |          :heavy_check_mark:         |         :heavy_check_mark:        |\n| show transceivers              |                 :x:                 |         :heavy_check_mark:        |\n| show vlan                      |          :heavy_check_mark:         |         :heavy_check_mark:        |\n| show vlan members              |                 :x:                 |         :heavy_check_mark:        |\n| show vlan port mobile          |          :heavy_check_mark:         |                :x:                |\n\n## Direct TextFSM example usage\n\nBypass the build-in parser functionality and use the TextFSM templates directly in network cli scraping and orchestration tools like Netmiko, Scrapli and Ansible.\n\n### Scrapli\n\nPython script:\n\n```python\nfrom scrapli import Scrapli\nfrom scrapli.helper import textfsm_parse\n\ndevice = {\n    "host": "<host ip>",\n    "auth_username": "<username>",\n    "auth_password": "<password>",\n    "auth_strict_key": False,\n    "transport": "ssh2",\n    "platform": "alcatel_aos",\n}\n\nwith Scrapli(**device) as conn:\n    response = conn.send_command("show health").result\n    structured_response = textfsm_parse(\n        "templates/ale_aos6_show_health.textfsm", response\n    )\n```\n\nExample output:\n\n```python\n[\n   {\n      "resource":"Receive",\n      "limit":"80",\n      "current":"01",\n      "min_avg":"01",\n      "hr_avg":"01",\n      "hr_max":"01"\n   },\n   {\n      "resource":"Transmit/Receive",\n      "limit":"80",\n      "current":"01",\n      "min_avg":"01",\n      "hr_avg":"01",\n      "hr_max":"01"\n   },\n   {\n      "resource":"Memory",\n      "limit":"80",\n      "current":"76",\n      "min_avg":"76",\n      "hr_avg":"76",\n      "hr_max":"76"\n   },\n   {\n      "resource":"Cpu",\n      "limit":"80",\n      "current":"32",\n      "min_avg":"33",\n      "hr_avg":"29",\n      "hr_max":"97"\n   }\n]\n```\n\n### Netmiko\n\nPython script:\n\n```python\nfrom netmiko import ConnectHandler\n\ndevice = {\n    \'device_type\': \'alcatel_aos\',\n    \'host\': \'<host ip>\',\n    \'username\': \'<username>\',\n    \'password\': \'<password>\'\n}\n\nwith ConnectHandler(**device) as conn:\n    output = conn.send_command("show health", use_textfsm=True, textfsm_template="textfsm-aos/templates/ale_aos6_show_health.textfsm")\n```\n\nExample Output:\n\n```python\n[\n   {\n      "resource":"Receive",\n      "limit":"80",\n      "current":"01",\n      "min_avg":"01",\n      "hr_avg":"01",\n      "hr_max":"01"\n   },\n   {\n      "resource":"Transmit/Receive",\n      "limit":"80",\n      "current":"01",\n      "min_avg":"01",\n      "hr_avg":"01",\n      "hr_max":"01"\n   },\n   {\n      "resource":"Memory",\n      "limit":"80",\n      "current":"76",\n      "min_avg":"76",\n      "hr_avg":"76",\n      "hr_max":"76"\n   },\n   {\n      "resource":"Cpu",\n      "limit":"80",\n      "current":"32",\n      "min_avg":"33",\n      "hr_avg":"29",\n      "hr_max":"97"\n   }\n]\n```\n\n### Ansible\n\nAnsible task:\n\n```yaml\n- name: AOS6 >> parsed with textfsm\n  set_fact:\n    health: "{{ health-aos6 | ansible.netcommon.parse_cli_textfsm(\'textfsm/templates/ale_aos6_show_health.textfsm\') }}"\n```\n\nExample Output:\n\n```yaml\n    health:\n    - healthModuleCpu1HrAvg: \'29\'\n      healthModuleCpu1HrMax: \'98\'\n      healthModuleCpu1MinAvg: \'26\'\n      healthModuleCpuLatest: \'31\'\n      healthModuleCpuLimit: \'80\'\n      healthModuleMemory1HrAvg: \'76\'\n      healthModuleMemory1HrMax: \'76\'\n      healthModuleMemory1MinAvg: \'76\'\n      healthModuleMemoryLatest: \'76\'\n      healthModuleMemoryLimit: \'80\'\n      healthModuleRx1HrAvg: \'01\'\n      healthModuleRx1HrMax: \'01\'\n      healthModuleRx1MinAvg: \'01\'\n      healthModuleRxLatest: \'01\'\n      healthModuleRxLimit: \'80\'\n      healthModuleRxTxRx1HrAvg: \'01\'\n      healthModuleRxTxRx1HrMax: \'01\'\n      healthModuleRxTxRx1MinAvg: \'01\'\n      healthModuleRxTxRxLatest: \'01\'\n      healthModuleRxTxRxLimit: \'80\'\n      healthModuleSlot: \'1\'\n```\n\n## How to contribute\n\n1. Create branch with naming `<platform>_<command>` (for example: ale_aos8_show_system).\n\n2. Add TextFSM template file in templates folder with naming `<platform>_<command>.textfsm`.\n\n3. Add entry in templates_index with attribute command and platform.\n\n4. Add test folder in \'templates\' with naming `<platform>_<command>`.\n\n5. Add sample cli output file in newly created folder `<platform>_<command>.txt`.\n\n6. Add expected parsed data from sample cli output in `<platform>_<command>.yml`.\n\n7. Run linting `tox` and tests `pytest`.\n\n## How to setup development environment\n\n1. Create virtual Python environment `python -m venv .venv`\n\n2. Activate environment `source .venv/bin/activate`\n\n3. Install Python dependencies `pip install -r requirements.txt`\n\n4. Install textfsm_aos package in development mode `pip install -e .`\n\n## Related projects\n\n- Google TextFSM: [https://github.com/google/textfsm](https://github.com/google/textfsm)\n- Scrapli: [https://github.com/carlmontanari/scrapli](https://github.com/carlmontanari/scrapli)\n- Netmiko: [https://github.com/ktbyers/netmiko](https://github.com/ktbyers/netmiko)\n- Getting started with TextFSM: [https://pyneng.readthedocs.io](https://pyneng.readthedocs.io/en/latest/book/21_textfsm/index.html)\n',
    'author': 'Jef Vantongerloo',
    'author_email': 'jefvantongerloo@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/jefvantongerloo/textfsm-aos',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
