"""
3D implementation of: \n
1. Channel Squeeze and Excitation \n
2. Spatial Squeeze and Excitation \n
3. Concurrent Spatial and Channel Squeeze & Excitation

Oringinal file is on `Github.
<https://github.com/maodong2056/squeeze_and_excitation/blob/master/squeeze_and_excitation/squeeze_and_excitation.py>`_
"""
from __future__ import print_function, division

from enum import Enum
import torch
import torch.nn as nn
import torch.nn.functional as F

class ChannelSELayer3D(nn.Module):
    """
    3D implementation of Squeeze-and-Excitation (SE) block.
    
    * Reference: Jie Hu, Li Shen, Gang Sun: Squeeze-and-Excitation Networks.
      `CVPR 2018. <https://ieeexplore.ieee.org/document/8578843>`_

    :param num_channels: Number of input channels
    :param reduction_ratio: By how much should the num_channels should be reduced
    """
    def __init__(self, num_channels, reduction_ratio=2):       
        super(ChannelSELayer3D, self).__init__()
        num_channels_reduced = num_channels // reduction_ratio
        self.reduction_ratio = reduction_ratio
        self.fc1 = nn.Linear(num_channels, num_channels_reduced, bias=True)
        self.fc2 = nn.Linear(num_channels_reduced, num_channels, bias=True)
        self.relu = nn.ReLU()
        self.sigmoid = nn.Sigmoid()

    def forward(self, input_tensor):
        """
        :param input_tensor: X, shape = (batch_size, num_channels, D, H, W)
        :return: output tensor
        """

        batch_size, num_channels, D, H, W = input_tensor.size()
        # Average along each channel
        squeeze_tensor = input_tensor.view(batch_size, num_channels, -1).mean(dim=2)
        # channel excitation
        fc_out_1 = self.relu(self.fc1(squeeze_tensor))
        fc_out_2 = self.sigmoid(self.fc2(fc_out_1))

        a, b = squeeze_tensor.size()
        output_tensor = torch.mul(input_tensor, fc_out_2.view(a, b, 1, 1, 1))

        return output_tensor

class SpatialSELayer3D(nn.Module):
    """
    3D Re-implementation of SE block -- squeezing spatially and exciting channel-wise described in:

    * Reference: Roy et al., Concurrent Spatial and Channel Squeeze & Excitation in 
      Fully Convolutional Networks, MICCAI 2018.
    
    :param num_channels: Number of input channels
    """
    def __init__(self, num_channels):
        super(SpatialSELayer3D, self).__init__()
        self.conv = nn.Conv3d(num_channels, 1, 1)
        self.sigmoid = nn.Sigmoid()

    def forward(self, input_tensor, weights=None):
        """
        :param weights: weights for few shot learning
        :param input_tensor: X, shape = (batch_size, num_channels, D, H, W)
        :return: output_tensor
        """
        # spatial squeeze
        batch_size, channel, D, H, W = input_tensor.size()
        if weights is not None:
            weights = torch.mean(weights, dim=0)
            weights = weights.view(1, channel, 1, 1)
            out = F.conv3d(input_tensor, weights)
        else:
            out = self.conv(input_tensor)
        squeeze_tensor = self.sigmoid(out)

        # spatial excitation
        # print(input_tensor.size(), squeeze_tensor.size())
        squeeze_tensor = squeeze_tensor.view(batch_size, 1, D, H, W)
        output_tensor = torch.mul(input_tensor, squeeze_tensor)

        #output_tensor = torch.mul(input_tensor, squeeze_tensor)
        return output_tensor


class ChannelSpatialSELayer3D(nn.Module):
    """
    3D Re-implementation of concurrent spatial and channel squeeze & excitation.

    * Reference: Roy et al., Concurrent Spatial and Channel Squeeze & Excitation in 
      Fully Convolutional Networks, MICCAI 2018.
    
    :param num_channels: Number of input channels
    :param reduction_ratio: By how much should the num_channels should be reduced
    """
    def __init__(self, num_channels, reduction_ratio=2):
        super(ChannelSpatialSELayer3D, self).__init__()
        self.cSE = ChannelSELayer3D(num_channels, reduction_ratio)
        self.sSE = SpatialSELayer3D(num_channels)

    def forward(self, input_tensor):
        """
        :param input_tensor: X, shape = (batch_size, num_channels, D, H, W)
        :return: output_tensor
        """
        output_tensor = torch.max(self.cSE(input_tensor), self.sSE(input_tensor))
        return output_tensor

class SELayer(Enum):
    """
    Enum restricting the type of SE Blockes available. So that type checking can be adding when adding these blockes to
    a neural network::
        if self.se_block_type == se.SELayer.CSE.value:
            self.SELayer = se.ChannelSpatialSELayer(params['num_filters'])
        elif self.se_block_type == se.SELayer.SSE.value:
            self.SELayer = se.SpatialSELayer(params['num_filters'])
        elif self.se_block_type == se.SELayer.CSSE.value:
            self.SELayer = se.ChannelSpatialSELayer(params['num_filters'])
    """
    NONE = 'NONE'
    CSE = 'CSE'
    SSE = 'SSE'
    CSSE = 'CSSE'