"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NxMonorepoProject = exports.TargetDependencyProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const fs = require("fs");
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const python_1 = require("projen/lib/python");
const typescript_1 = require("projen/lib/typescript");
const syncpack_options_1 = require("./syncpack-options");
const NX_MONOREPO_PLUGIN_PATH = ".nx/plugins/nx-monorepo-plugin.js";
/**
 * Supported enums for a TargetDependency.
 */
var TargetDependencyProject;
(function (TargetDependencyProject) {
    /**
     * Only rely on the package where the target is called.
     *
     * This is usually done for test like targets where you only want to run unit
     * tests on the target packages without testing all dependent packages.
     */
    TargetDependencyProject["SELF"] = "self";
    /**
     * Target relies on executing the target against all dependencies first.
     *
     * This is usually done for build like targets where you want to build all
     * dependant projects first.
     */
    TargetDependencyProject["DEPENDENCIES"] = "dependencies";
})(TargetDependencyProject = exports.TargetDependencyProject || (exports.TargetDependencyProject = {}));
/**
 * This project type will bootstrap a NX based monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid nx-monorepo
 */
class NxMonorepoProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            github: options.github ?? false,
            package: options.package ?? false,
            prettier: options.prettier ?? true,
            projenrcTs: true,
            release: options.release ?? false,
            jest: options.jest ?? false,
            defaultReleaseBranch: options.defaultReleaseBranch ?? "mainline",
            sampleCode: false,
        });
        this.targetOverrides = {};
        this.nxConfig = options.nxConfig;
        this.workspaceConfig = options.workspaceConfig;
        this.additionalWorkspacePackages =
            options.workspaceConfig?.additionalPackages ?? [];
        this.implicitDependencies = this.nxConfig?.implicitDependencies || {};
        // Never publish a monorepo root package.
        this.package.addField("private", true);
        // No need to compile or test a monorepo root package.
        this.compileTask.reset();
        this.testTask.reset();
        this.addDevDeps("@nrwl/cli", "@nrwl/workspace");
        this.addDeps("aws-cdk-lib", "constructs", "cdk-nag"); // Needed as this can be bundled in aws-prototyping-sdk
        if (options.monorepoUpgradeDeps !== false) {
            this.addDevDeps("npm-check-updates", "syncpack");
            const upgradeDepsTask = this.addTask(options.monorepoUpgradeDepsOptions?.taskName || "upgrade-deps");
            upgradeDepsTask.exec("npx npm-check-updates --deep --rejectVersion 0.0.0 -u");
            upgradeDepsTask.exec("npx syncpack fix-mismatches");
            upgradeDepsTask.exec(`${this.package.packageManager} install`);
            upgradeDepsTask.exec("npx projen");
            new projen_1.JsonFile(this, ".syncpackrc.json", {
                obj: options.monorepoUpgradeDepsOptions?.syncpackConfig || syncpack_options_1.DEFAULT_CONFIG,
                readonly: true,
            });
        }
        options.nxConfig?.nxCloudReadOnlyAccessToken &&
            this.addDevDeps("@nrwl/nx-cloud");
        new projen_1.IgnoreFile(this, ".nxignore").exclude("test-reports", "target", ".env", ".pytest_cache", ...(this.nxConfig?.nxIgnore || []));
        new projen_1.TextFile(this, NX_MONOREPO_PLUGIN_PATH, {
            readonly: true,
            lines: fs.readFileSync(getPluginPath()).toString("utf-8").split("\n"),
        });
        this.nxJson = new projen_1.JsonFile(this, "nx.json", {
            obj: {
                extends: "@nrwl/workspace/presets/npm.json",
                plugins: [`./${NX_MONOREPO_PLUGIN_PATH}`],
                npmScope: "monorepo",
                tasksRunnerOptions: {
                    default: {
                        runner: options.nxConfig?.nxCloudReadOnlyAccessToken
                            ? "@nrwl/nx-cloud"
                            : "@nrwl/workspace/tasks-runners/default",
                        options: {
                            useDaemonProcess: false,
                            cacheableOperations: ["build", "test"],
                            accessToken: options.nxConfig?.nxCloudReadOnlyAccessToken,
                        },
                    },
                },
                implicitDependencies: this.implicitDependencies,
                targetDependencies: {
                    build: [
                        {
                            target: "build",
                            projects: "dependencies",
                        },
                    ],
                    ...(this.nxConfig?.targetDependencies || {}),
                },
                affected: {
                    defaultBase: this.nxConfig?.affectedBranch || "mainline",
                },
            },
        });
    }
    /**
     * Create an implicit dependency between two Project's. This is typically
     * used in polygot repos where a Typescript project wants a build dependency
     * on a Python project as an example.
     *
     * @param dependent project you want to have the dependency.
     * @param dependee project you wish to depend on.
     */
    addImplicitDependency(dependent, dependee) {
        if (this.implicitDependencies[dependent.name]) {
            this.implicitDependencies[dependent.name].push(dependee.name);
        }
        else {
            this.implicitDependencies[dependent.name] = [dependee.name];
        }
    }
    /**
     * Add one or more additional package globs to the workspace.
     * @param packageGlobs paths to the package to include in the workspace (for example packages/my-package)
     */
    addWorkspacePackages(...packageGlobs) {
        this.additionalWorkspacePackages.push(...packageGlobs);
    }
    /**
     * Allow project specific target overrides.
     */
    overrideProjectTargets(project, targets) {
        const _package = project.tryFindObjectFile("package.json");
        _package?.addOverride("nx", {
            targets: targets,
        });
        !_package && (this.targetOverrides[project.outdir] = targets);
    }
    // Remove this hack once subProjects is made public in Projen
    get subProjects() {
        // @ts-ignore
        const subProjects = this.subprojects || [];
        return subProjects.sort((a, b) => a.name.localeCompare(b.name));
    }
    /**
     * @inheritDoc
     */
    synth() {
        this.validateSubProjects();
        this.updateWorkspace();
        this.wirePythonDependencies();
        this.synthesizeNonNodePackageJson();
        super.synth();
    }
    /**
     * Ensures subprojects don't have a default task and that all packages use the same package manager.
     */
    validateSubProjects() {
        this.subProjects.forEach((subProject) => {
            // Disable default task on subprojects as this isn't supported in a monorepo
            subProject.defaultTask?.reset();
            if (isNodeProject(subProject) &&
                subProject.package.packageManager !== this.package.packageManager) {
                throw new Error(`${subProject.name} packageManager does not match the monorepo packageManager: ${this.package.packageManager}.`);
            }
        });
    }
    /**
     * For non-node projects, a package.json is required in order to be discovered by NX.
     */
    synthesizeNonNodePackageJson() {
        this.subProjects
            .filter((subProject) => !isNodeProject(subProject))
            .filter((subProject) => !subProject.tryFindFile("package.json"))
            .forEach((subProject) => {
            // generate a package.json if not found
            const manifest = {};
            (manifest.nx = this.targetOverrides[subProject.outdir]
                ? { targets: this.targetOverrides[subProject.outdir] }
                : undefined),
                (manifest.name = subProject.name);
            manifest.private = true;
            manifest.__pdk__ = true;
            manifest.scripts = subProject.tasks.all.reduce((p, c) => ({
                [c.name]: `npx projen ${c.name}`,
                ...p,
            }), {});
            manifest.version = "0.0.0";
            new projen_1.JsonFile(subProject, "package.json", {
                obj: manifest,
                readonly: true,
            });
        });
    }
    /**
     * Add a submodule entry to the appropriate workspace file.
     */
    updateWorkspace() {
        // Add workspaces for each subproject
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            new projen_1.YamlFile(this, "pnpm-workspace.yaml", {
                readonly: true,
                obj: {
                    packages: this.subProjects
                        .map((subProject) => path.relative(this.outdir, subProject.outdir))
                        .concat(this.additionalWorkspacePackages),
                },
            });
        }
        else {
            this.package.addField("workspaces", {
                packages: this.subProjects
                    .map((subProject) => path.relative(this.outdir, subProject.outdir))
                    .concat(this.additionalWorkspacePackages),
                nohoist: this.workspaceConfig?.noHoist,
            });
        }
    }
    /**
     * Updates the install task for python projects so that they are run serially and in dependency order such that python
     * projects within the monorepo can declare dependencies on one another.
     * @private
     */
    wirePythonDependencies() {
        // Find any python projects
        const pythonProjects = this.subProjects.filter((project) => project instanceof python_1.PythonProject);
        if (pythonProjects.length === 0) {
            // Nothing to do for no python projects
            return;
        }
        // Move all install commands to install-py so that they are not installed in parallel by the monorepo package manager.
        // eg yarn install will run the install task for all packages in parallel which can lead to conflicts for python.
        pythonProjects.forEach((pythonProject) => {
            const installPyTask = pythonProject.tasks.tryFind("install-py") ??
                pythonProject.addTask("install-py");
            const installTask = pythonProject.tasks.tryFind("install");
            (installTask?.steps || []).forEach((step) => {
                this.copyStepIntoTask(step, installPyTask, pythonProject);
            });
            installTask?.reset();
        });
        // Add an install task to the monorepo to include running the install-py command serially to avoid conflicting writes
        // to a shared virtual env. This is also managed by nx so that installs occur in dependency order.
        const monorepoInstallTask = this.tasks.tryFind("install") ?? this.addTask("install");
        monorepoInstallTask.exec(`npx nx run-many --target install-py --projects ${pythonProjects
            .map((project) => project.name)
            .join(",")} --parallel=1`);
        // Update the nx.json to ensure that install-py follows dependency order
        this.nxJson.addOverride("targetDependencies.install-py", [
            {
                target: "install-py",
                projects: "dependencies",
            },
        ]);
    }
    /**
     * Copies the given step into the given task. Step and Task must be from the given Project
     * @private
     */
    copyStepIntoTask(step, task, project) {
        if (step.exec) {
            task.exec(step.exec, { name: step.name, cwd: step.cwd });
        }
        else if (step.say) {
            task.say(step.say, { name: step.name, cwd: step.cwd });
        }
        else if (step.spawn) {
            const stepToSpawn = project.tasks.tryFind(step.spawn);
            if (stepToSpawn) {
                task.spawn(stepToSpawn, { name: step.name, cwd: step.cwd });
            }
        }
        else if (step.builtin) {
            task.builtin(step.builtin);
        }
    }
}
exports.NxMonorepoProject = NxMonorepoProject;
_a = JSII_RTTI_SYMBOL_1;
NxMonorepoProject[_a] = { fqn: "aws-prototyping-sdk.nx_monorepo.NxMonorepoProject", version: "0.11.9" };
/**
 * Determines if the passed in project is of type NodeProject.
 *
 * @param project Project instance.
 * @returns true if the project instance is of type NodeProject.
 */
function isNodeProject(project) {
    return project instanceof javascript_1.NodeProject || project.package;
}
function getPluginPath() {
    return path.join(__dirname, "plugin/nx-monorepo-plugin.js");
}
//# sourceMappingURL=data:application/json;base64,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