"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Operator = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Use condition operators in the `Condition` element to match the condition key and value in the policy against values in the request context. For more information about the `Condition` element, see [IAM JSON Policy Elements: Condition](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
 *
 * The condition operator that you can use in a policy depends on the condition key you choose. You can choose a global condition key or a service-specific condition key. To learn which condition operator you can use for a global condition key, see [AWS Global Condition Context Keys](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html). To learn which condition operator you can use for a service-specific condition key, see [Actions, Resources, and Condition Keys for AWS Services](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_actions-resources-contextkeys.html) and choose the service that you want to view.
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html
 */
class Operator {
    constructor() {
        this.base = '';
        this.hasIfExists = false;
        this.hasForAllValues = false;
        this.hasForAnyValue = false;
    }
    setBase(base) {
        if (this.base.length) {
            throw new Error(`Operator already set: ${this.base}`);
        }
        this.base = base;
        return this;
    }
    toString() {
        let value = this.base;
        if (this.base.length == 0) {
            throw new Error('No operator set');
        }
        if (this.hasIfExists) {
            value += 'IfExists';
        }
        if (this.hasForAllValues) {
            value = `ForAllValues:${value}`;
        }
        if (this.hasForAnyValue) {
            value = `ForAnyValue:${value}`;
        }
        if (this.base == 'Null' && value != this.base) {
            throw new Error(`You cannot add modifiers to the "Null" operator: ${value}`);
        }
        return value;
    }
    /**
     * You can add `IfExists` to the end of any condition operator name except the `Null` condition. For example, `StringLikeIfExists`. You do this to say "If the policy key is present in the context of the request, process the key as specified in the policy. If the key is not present, evaluate the condition element as true." Other condition elements in the statement can still result in a nonmatch, but not a missing key when checked with `...IfExists`.
     */
    ifExists() {
        this.hasIfExists = true;
        return this;
    }
    /**
     * Tests whether the value of every member of the request set is a subset of the condition key set. The condition returns true if every key value in the request matches at least one value in the policy. It also returns true if there are no keys in the request, or if the key values resolve to a null data set, such as an empty string.
     */
    forAllValues() {
        this.hasForAllValues = true;
        return this;
    }
    /**
     * Tests whether at least one member of the set of request values matches at least one member of the set of condition key values. The condition returns true if any one of the key values in the request matches any one of the condition values in the policy. For no matching key or a null dataset, the condition returns false.
     */
    forAnyValue() {
        this.hasForAnyValue = true;
        return this;
    }
    /**
     * Exact match, case sensitive.
     */
    stringEquals() {
        return this.setBase('StringEquals');
    }
    /**
     * Negated exact match, case sensitive.
     */
    stringNotEquals() {
        return this.setBase('StringNotEquals');
    }
    /**
     * Exact match, ignore case.
     */
    stringEqualsIgnoreCase() {
        return this.setBase('StringEqualsIgnoreCase');
    }
    /**
     * Exact exact match, ignore case.
     */
    stringNotEqualsIgnoreCase() {
        return this.setBase('StringNotEqualsIgnoreCase');
    }
    /**
     * Case-sensitive match.
     *
     * The values can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`) anywhere in the string.
     */
    stringLike() {
        return this.setBase('StringLike');
    }
    /**
     * Negated case-sensitive matching.
     *
     * The values can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`) anywhere in the string.
     */
    stringNotLike() {
        return this.setBase('StringNotLike');
    }
    /**
     * Exact match.
     */
    numericEquals() {
        return this.setBase('NumericEquals');
    }
    /**
     * Negated exact match.
     */
    numericNotEquals() {
        return this.setBase('NumericNotEquals');
    }
    /**
     * Match numbers lower than value: `<`
     */
    numericLessThan() {
        return this.setBase('NumericLessThan');
    }
    /**
     * Match numbers lower or equal to value: `<=`
     */
    numericLessThanEquals() {
        return this.setBase('NumericLessThanEquals');
    }
    /**
     * Match numbers higher than value: `>`
     */
    numericGreaterThan() {
        return this.setBase('NumericGreaterThan');
    }
    /**
     * Match numbers higher or equal value: `>=`
     */
    numericGreaterThanEquals() {
        return this.setBase('NumericGreaterThanEquals');
    }
    /**
     * Match a specific date: `=`
     */
    dateEquals() {
        return this.setBase('DateEquals');
    }
    /**
     * Negated match for a specific date: `!=`
     */
    dateNotEquals() {
        return this.setBase('DateNotEquals');
    }
    /**
     * Match before a specific date and time: `<`
     */
    dateLessThan() {
        return this.setBase('DateLessThan');
    }
    /**
     * Match at or before a specific date and time: `<=`
     */
    dateLessThanEquals() {
        return this.setBase('DateLessThanEquals');
    }
    /**
     * Match after a specific a date and time: `>`
     */
    dateGreaterThan() {
        return this.setBase('DateGreaterThan');
    }
    /**
     * Match at or after a specific date and time: `>=`
     */
    dateGreaterThanEquals() {
        return this.setBase('DateGreaterThanEquals');
    }
    /**
     * Boolean match
     */
    bool() {
        return this.setBase('Bool');
    }
    /**
     * The BinaryEquals condition operator lets you construct Condition elements that test key values that are in binary format. It compares the value of the specified key byte for byte against a [base-64](https://en.wikipedia.org/wiki/Base64) encoded representation of the binary value in the policy.
     */
    binaryEquals() {
        return this.setBase('BinaryEquals');
    }
    /**
     * Match an IP address or range: `=`
     */
    ipAddress() {
        return this.setBase('IpAddress');
    }
    /**
     * All IP addresses except the specified IP address or range `!=`
     */
    notIpAddress() {
        return this.setBase('NotIpAddress');
    }
    /**
     * Match of the ARN, case sensitive.
     *
     * Each of the six colon-delimited components of the ARN is checked separately and each can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`).
     *
     * `ArnEquals` and `ArnLike` behave identical.
     */
    arnEquals() {
        return this.setBase('ArnEquals');
    }
    /**
     * Negated match of the ARN, case sensitive.
     *
     * Each of the six colon-delimited components of the ARN is checked separately and each can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`).
     *
     * `ArnNotEquals` and `ArnNotLike` behave identical.
     */
    arnNotEquals() {
        return this.setBase('ArnNotEquals');
    }
    /**
     * Match of the ARN, case sensitive.
     *
     * Each of the six colon-delimited components of the ARN is checked separately and each can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`).
     */
    arnLike() {
        return this.setBase('ArnLike');
    }
    /**
     * Negated match of the ARN, case sensitive.
     *
     * Each of the six colon-delimited components of the ARN is checked separately and each can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`).
     *
     * `ArnNotEquals` and `ArnNotLike` behave identical.
     */
    arnNotLike() {
        return this.setBase('ArnNotLike');
    }
    /**
     * Check if a key is present at the time of authorization. In the policy statement, use either true (the key doesn't exist — it is null) or false (the key exists and its value is not null).
     */
    null() {
        return this.setBase('Null');
    }
}
exports.Operator = Operator;
_a = JSII_RTTI_SYMBOL_1;
Operator[_a] = { fqn: "iam-floyd.Operator", version: "0.427.0" };
//# sourceMappingURL=data:application/json;base64,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