from enum import Enum, unique


@unique
class Currency(Enum):
    """Currencies as for ISO 4217."""
    AFN = 1
    ARS = 2
    AWG = 3
    AUD = 4
    AZN = 5
    BSD = 6
    BBD = 7
    BDT = 8
    BYR = 9
    BZD = 10
    BMD = 11
    BOB = 12
    BAM = 13
    BWP = 14
    BGN = 15
    BRL = 16
    BND = 17
    KHR = 18
    CAD = 19
    KYD = 20
    CLP = 21
    CNY = 22
    COP = 23
    CRC = 24
    HRK = 25
    CUP = 26
    CZK = 27
    DKK = 28
    DOP = 29
    XCD = 30
    EGP = 31
    SVC = 32
    EEK = 33
    EUR = 34
    FKP = 35
    FJD = 36
    GHC = 37
    GIP = 38
    GTQ = 39
    GGP = 40
    GYD = 41
    HNL = 42
    HKD = 43
    HUF = 44
    ISK = 45
    INR = 46
    IDR = 47
    IRR = 48
    IMP = 49
    ILS = 50
    JMD = 51
    JPY = 52
    JEP = 53
    KZT = 54
    KPW = 55
    KRW = 56
    KGS = 57
    LAK = 58
    LVL = 59
    LBP = 60
    LRD = 61
    LTL = 62
    MKD = 63
    MYR = 64
    MUR = 65
    MXN = 66
    MNT = 67
    MZN = 68
    NAD = 69
    NPR = 70
    ANG = 71
    NZD = 72
    NIO = 73
    NGN = 74
    NOK = 75
    OMR = 76
    PKR = 77
    PAB = 78
    PYG = 79
    PEN = 80
    PHP = 81
    PLN = 82
    QAR = 83
    RON = 84
    RUB = 85
    SHP = 86
    SAR = 87
    RSD = 88
    SCR = 89
    SGD = 90
    SBD = 91
    SOS = 92
    ZAR = 93
    LKR = 94
    SEK = 95
    CHF = 96
    SRD = 97
    SYP = 98
    TWD = 99
    THB = 100
    TTD = 101
    TRY = 102
    TRL = 103
    TVD = 104
    UAH = 105
    GBP = 106
    USD = 107
    UYU = 108
    UZS = 109
    VEF = 110
    VND = 111
    YER = 112
    ZWD = 113

    def __str__(self):
        if self == Currency.EUR:
            return '€'
        else:
            return self.name


@unique
class Continent(Enum):
    """
    Continent codes.

    From `Data hub <https://datahub.io/core/continent-codes#data>`_.
    """
    AF = 'Africa'
    NA = 'North America'
    OC = 'Oceania'
    AN = 'Antartica'
    AS = 'Asia'
    EU = 'Europe'
    SA = 'South America'


@unique
class Country(Enum):
    """
    Countries as ISO 3166-1 alpha 2.

    Taken from table from `iso-3616-1 commit
    8e31d749b9ce331cfa50c280a29b04ae2d805b7e  <https://github.com/spoqa/
    iso-3166-1/blob/master/iso3166/table.csv>`_.
    """
    AF = "Afghanistan"
    AX = "Åland Islands"
    AL = "Albania"
    DZ = "Algeria"
    AS = "American Samoa"
    AD = "Andorra"
    AO = "Angola"
    AI = "Anguilla"
    AQ = "Antarctica"
    AG = "Antigua and Barbuda"
    AR = "Argentina"
    AM = "Armenia"
    AW = "Aruba"
    AU = "Australia"
    AT = "Austria"
    AZ = "Azerbaijan"
    BS = "Bahamas"
    BH = "Bahrain"
    BD = "Bangladesh"
    BB = "Barbados"
    BY = "Belarus"
    BE = "Belgium"
    BZ = "Belize"
    BJ = "Benin"
    BM = "Bermuda"
    BT = "Bhutan"
    BO = "Bolivia (Plurinational State of)"
    BQ = "Bonaire, Sint Eustatius and Saba"
    BA = "Bosnia and Herzegovina"
    BW = "Botswana"
    BV = "Bouvet Island"
    BR = "Brazil"
    IO = "British Indian Ocean Territory"
    BN = "Brunei Darussalam"
    BG = "Bulgaria"
    BF = "Burkina Faso"
    BI = "Burundi"
    KH = "Cambodia"
    CM = "Cameroon"
    CA = "Canada"
    CV = "Cabo Verde"
    KY = "Cayman Islands"
    CF = "Central African Republic"
    TD = "Chad"
    CL = "Chile"
    CN = "China"
    CX = "Christmas Island"
    CC = "Cocos (Keeling) Islands"
    CO = "Colombia"
    KM = "Comoros"
    CG = "Congo"
    CD = "Congo (Democratic Republic of the)"
    CK = "Cook Islands"
    CR = "Costa Rica"
    CI = "Côte d'Ivoire"
    HR = "Croatia"
    CU = "Cuba"
    CW = "Curaçao"
    CY = "Cyprus"
    CZ = "Czech Republic"
    DK = "Denmark"
    DJ = "Djibouti"
    DM = "Dominica"
    DO = "Dominican Republic"
    EC = "Ecuador"
    EG = "Egypt"
    SV = "El Salvador"
    GQ = "Equatorial Guinea"
    ER = "Eritrea"
    EE = "Estonia"
    ET = "Ethiopia"
    FK = "Falkland Islands (Malvinas)"
    FO = "Faroe Islands"
    FJ = "Fiji"
    FI = "Finland"
    FR = "France"
    GF = "French Guiana"
    PF = "French Polynesia"
    TF = "French Southern Territories"
    GA = "Gabon"
    GM = "Gambia"
    GE = "Georgia"
    DE = "Germany"
    GH = "Ghana"
    GI = "Gibraltar"
    GR = "Greece"
    GL = "Greenland"
    GD = "Grenada"
    GP = "Guadeloupe"
    GU = "Guam"
    GT = "Guatemala"
    GG = "Guernsey"
    GN = "Guinea"
    GW = "Guinea-Bissau"
    GY = "Guyana"
    HT = "Haiti"
    HM = "Heard Island and McDonald Islands"
    VA = "Holy See"
    HN = "Honduras"
    HK = "Hong Kong"
    HU = "Hungary"
    IS = "Iceland"
    IN = "India"
    ID = "Indonesia"
    IR = "Iran (Islamic Republic of)"
    IQ = "Iraq"
    IE = "Ireland"
    IM = "Isle of Man"
    IL = "Israel"
    IT = "Italy"
    JM = "Jamaica"
    JP = "Japan"
    JE = "Jersey"
    JO = "Jordan"
    KZ = "Kazakhstan"
    KE = "Kenya"
    KI = "Kiribati"
    KP = "Korea (Democratic People's Republic of)"
    KR = "Korea (Republic of)"
    KW = "Kuwait"
    KG = "Kyrgyzstan"
    LA = "Lao People's Democratic Republic"
    LV = "Latvia"
    LB = "Lebanon"
    LS = "Lesotho"
    LR = "Liberia"
    LY = "Libya"
    LI = "Liechtenstein"
    LT = "Lithuania"
    LU = "Luxembourg"
    MO = "Macao"
    MK = "Macedonia (the former Yugoslav Republic of)"
    MG = "Madagascar"
    MW = "Malawi"
    MY = "Malaysia"
    MV = "Maldives"
    ML = "Mali"
    MT = "Malta"
    MH = "Marshall Islands"
    MQ = "Martinique"
    MR = "Mauritania"
    MU = "Mauritius"
    YT = "Mayotte"
    MX = "Mexico"
    FM = "Micronesia (Federated States of)"
    MD = "Moldova (Republic of)"
    MC = "Monaco"
    MN = "Mongolia"
    ME = "Montenegro"
    MS = "Montserrat"
    MA = "Morocco"
    MZ = "Mozambique"
    MM = "Myanmar"
    NA = "Namibia"
    NR = "Nauru"
    NP = "Nepal"
    NL = "Netherlands"
    NC = "New Caledonia"
    NZ = "New Zealand"
    NI = "Nicaragua"
    NE = "Niger"
    NG = "Nigeria"
    NU = "Niue"
    NF = "Norfolk Island"
    MP = "Northern Mariana Islands"
    NO = "Norway"
    OM = "Oman"
    PK = "Pakistan"
    PW = "Palau"
    PS = "Palestine, State of"
    PA = "Panama"
    PG = "Papua New Guinea"
    PY = "Paraguay"
    PE = "Peru"
    PH = "Philippines"
    PN = "Pitcairn"
    PL = "Poland"
    PT = "Portugal"
    PR = "Puerto Rico"
    QA = "Qatar"
    RE = "Réunion"
    RO = "Romania"
    RU = "Russian Federation"
    RW = "Rwanda"
    BL = "Saint Barthélemy"
    SH = "Saint Helena, Ascension and Tristan da Cunha"
    KN = "Saint Kitts and Nevis"
    LC = "Saint Lucia"
    MF = "Saint Martin (French part)"
    PM = "Saint Pierre and Miquelon"
    VC = "Saint Vincent and the Grenadines"
    WS = "Samoa"
    SM = "San Marino"
    ST = "Sao Tome and Principe"
    SA = "Saudi Arabia"
    SN = "Senegal"
    RS = "Serbia"
    SC = "Seychelles"
    SL = "Sierra Leone"
    SG = "Singapore"
    SX = "Sint Maarten (Dutch part)"
    SK = "Slovakia"
    SI = "Slovenia"
    SB = "Solomon Islands"
    SO = "Somalia"
    ZA = "South Africa"
    GS = "South Georgia and the South Sandwich Islands"
    SS = "South Sudan"
    ES = "Spain"
    LK = "Sri Lanka"
    SD = "Sudan"
    SR = "Suriname"
    SJ = "Svalbard and Jan Mayen"
    SZ = "Swaziland"
    SE = "Sweden"
    CH = "Switzerland"
    SY = "Syrian Arab Republic"
    TW = "Taiwan, Province of China"
    TJ = "Tajikistan"
    TZ = "Tanzania, United Republic of"
    TH = "Thailand"
    TL = "Timor-Leste"
    TG = "Togo"
    TK = "Tokelau"
    TO = "Tonga"
    TT = "Trinidad and Tobago"
    TN = "Tunisia"
    TR = "Turkey"
    TM = "Turkmenistan"
    TC = "Turks and Caicos Islands"
    TV = "Tuvalu"
    UG = "Uganda"
    UA = "Ukraine"
    AE = "United Arab Emirates"
    GB = "United Kingdom of Great Britain and Northern Ireland"
    US = "United States of America"
    UM = "United States Minor Outlying Islands"
    UY = "Uruguay"
    UZ = "Uzbekistan"
    VU = "Vanuatu"
    VE = "Venezuela (Bolivarian Republic of)"
    VN = "Viet Nam"
    VG = "Virgin Islands (British)"
    VI = "Virgin Islands (U.S.)"
    WF = "Wallis and Futuna"
    EH = "Western Sahara"
    YE = "Yemen"
    ZM = "Zambia"
    ZW = "Zimbabwe"

    def __contains__(self, item: 'Subdivision'):
        """Checks if a Subdivision is inside of this Country."""
        if not isinstance(item, Subdivision):
            raise TypeError('Only subdivisions can be inside a country.')
        return item.country == self

    def __str__(self):
        return self.value


class SubdivisionMixin:
    @property
    def country(self: Enum) -> Country:
        """Returns the Country of the Subdivision."""
        return Country[self.name[0:2]]


# noinspection PyArgumentList
Subdivision = Enum(
    'Subdivision',
    type=SubdivisionMixin,
    module=__name__,
    names=(
        'AE-AJ',
        'AE-AZ',
        'AE-DU',
        'AE-FU',
        'AE-RK',
        'AE-SH',
        'AE-UQ',
        'AF-BAL',
        'AF-BAM',
        'AF-BDG',
        'AF-BDS',
        'AF-BGL',
        'AF-FRAU',
        'AF-FYB',
        'AF-GHA',
        'AF-GHO',
        'AF-HEL',
        'AF-HER',
        'AF-JOW',
        'AF-KAB',
        'AF-KANN',
        'AF-KAP',
        'AF-KDZ',
        'AF-KNR',
        'AF-LAG',
        'AF-LOW',
        'AF-NAN',
        'AF-NIM',
        'AF-ORU',
        'AF-PAR',
        'AF-PIA',
        'AF-PKA',
        'AF-SAM',
        'AF-SAR',
        'AF-TAK',
        'AF-WAR',
        'AF-ZAB',
        'AL-BR',
        'AL-BU',
        'AL-DI',
        'AL-DL',
        'AL-DR',
        'AL-DV',
        'AL-EL',
        'AL-ER',
        'AL-FR',
        'AL-GJ',
        'AL-GR',
        'AL-HA',
        'AL-KA',
        'AL-KB',
        'AL-KC',
        'AL-KO',
        'AL-KR',
        'AL-KU',
        'AL-LA',
        'AL-LB',
        'AL-LE',
        'AL-LU',
        'AL-MK',
        'AL-MM',
        'AL-MR',
        'AL-MT',
        'AL-PG',
        'AL-PQ',
        'AL-PR',
        'AL-PU',
        'AL-SH',
        'AL-SK',
        'AL-SR',
        'AL-TE',
        'AL-TP',
        'AL-TR',
        'AL-VL',
        'AM-AG',
        'AM-AR',
        'AM-AV',
        'AM-ER',
        'AM-GR',
        'AM-KT',
        'AM-LO',
        'AM-SH',
        'AM-SU',
        'AM-TV',
        'AM-VD',
        'AO-BGO',
        'AO-BGU',
        'AO-BIE',
        'AO-CAB',
        'AO-CCU',
        'AO-CNN',
        'AO-CNO',
        'AO-CUS',
        'AO-HUA',
        'AO-HUI',
        'AO-LNO',
        'AO-LSU',
        'AO-LUA',
        'AO-MAL',
        'AO-MOX',
        'AO-NAM',
        'AO-UIG',
        'AO-ZAI',
        'AR-A',
        'AR-B',
        'AR-C',
        'AR-D',
        'AR-E',
        'AR-F',
        'AR-G',
        'AR-H',
        'AR-J',
        'AR-K',
        'AR-L',
        'AR-M',
        'AR-N',
        'AR-P',
        'AR-Q',
        'AR-R',
        'AR-S',
        'AR-T',
        'AR-U',
        'AR-V',
        'AR-W',
        'AR-X',
        'AR-Y',
        'AR-Z',
        'AT-1',
        'AT-2',
        'AT-3',
        'AT-4',
        'AT-5',
        'AT-6',
        'AT-7',
        'AT-8',
        'AT-9',
        'AU-CT',
        'AU-NS',
        'AU-NT',
        'AU-QL',
        'AU-SA',
        'AU-TS',
        'AU-VI',
        'AU-WA',
        'AZ-AB',
        'AZ-ABS',
        'AZ-AGA',
        'AZ-AGC',
        'AZ-AGM',
        'AZ-AGS',
        'AZ-AGU',
        'AZ-AST',
        'AZ-BA',
        'AZ-BAB',
        'AZ-BAL',
        'AZ-BAR',
        'AZ-BEY',
        'AZ-BIL',
        'AZ-CAB',
        'AZ-CAL',
        'AZ-CUL',
        'AZ-DAS',
        'AZ-DAV',
        'AZ-FUZ',
        'AZ-GA',
        'AZ-GAD',
        'AZ-GOR',
        'AZ-GOY',
        'AZ-HAC',
        'AZ-IMI',
        'AZ-ISM',
        'AZ-KAL',
        'AZ-KUR',
        'AZ-LA',
        'AZ-LAC',
        'AZ-LAN',
        'AZ-LER',
        'AZ-MAS',
        'AZ-MI',
        'AZ-MM',
        'AZ-NA',
        'AZ-NEF',
        'AZ-OGU',
        'AZ-ORD',
        'AZ-QAB',
        'AZ-QAX',
        'AZ-QAZ',
        'AZ-QBA',
        'AZ-QBI',
        'AZ-QOB',
        'AZ-QUS',
        'AZ-SA',
        'AZ-SAB',
        'AZ-SAD',
        'AZ-SAH',
        'AZ-SAK',
        'AZ-SAL',
        'AZ-SAR',
        'AZ-SAT',
        'AZ-SIY',
        'AZ-SKR',
        'AZ-SM',
        'AZ-SMI',
        'AZ-SMX',
        'AZ-SS',
        'AZ-SUS',
        'AZ-TAR',
        'AZ-TOV',
        'AZ-UCA',
        'AZ-XA',
        'AZ-XAC',
        'AZ-XAN',
        'AZ-XCI',
        'AZ-XIZ',
        'AZ-XVD',
        'AZ-YAR',
        'AZ-YE',
        'AZ-YEV',
        'AZ-ZAN',
        'AZ-ZAQ',
        'AZ-ZAR',
        'BA-BIH',
        'BA-SRP',
        'BD-01',
        'BD-02',
        'BD-03',
        'BD-04',
        'BD-05',
        'BD-06',
        'BD-07',
        'BD-08',
        'BD-09',
        'BD-1',
        'BD-10',
        'BD-11',
        'BD-12',
        'BD-13',
        'BD-14',
        'BD-15',
        'BD-16',
        'BD-17',
        'BD-18',
        'BD-19',
        'BD-2',
        'BD-20',
        'BD-21',
        'BD-22',
        'BD-23',
        'BD-24',
        'BD-25',
        'BD-26',
        'BD-27',
        'BD-28',
        'BD-29',
        'BD-3',
        'BD-30',
        'BD-31',
        'BD-32',
        'BD-33',
        'BD-34',
        'BD-35',
        'BD-36',
        'BD-37',
        'BD-38',
        'BD-39',
        'BD-4',
        'BD-40',
        'BD-41',
        'BD-42',
        'BD-43',
        'BD-44',
        'BD-45',
        'BD-46',
        'BD-47',
        'BD-48',
        'BD-49',
        'BD-5',
        'BD-50',
        'BD-51',
        'BD-52',
        'BD-53',
        'BD-54',
        'BD-55',
        'BD-56',
        'BD-57',
        'BD-58',
        'BD-59',
        'BD-6',
        'BD-60',
        'BD-61',
        'BD-62',
        'BD-63',
        'BD-64',
        'BE-BRU',
        'BE-VAN',
        'BE-VBR',
        'BE-VLG',
        'BE-VLI',
        'BE-VOV',
        'BE-VWV',
        'BE-WAL',
        'BE-WBR',
        'BE-WHT',
        'BE-WLG',
        'BE-WLX',
        'BE-WNA',
        'BF-BAL',
        'BF-BAM',
        'BF-BAN',
        'BF-BAZ',
        'BF-BGR',
        'BF-BLG',
        'BF-BLK',
        'BF-COM',
        'BF-GAN',
        'BF-GNA',
        'BF-GOU',
        'BF-HOU',
        'BF-IOB',
        'BF-KAD',
        'BF-KEN',
        'BF-KMD',
        'BF-KMP',
        'BF-KOP',
        'BF-KOS',
        'BF-KOT',
        'BF-KOW',
        'BF-LER',
        'BF-LOR',
        'BF-MOU',
        'BF-NAM',
        'BF-NAO',
        'BF-NAY',
        'BF-NOU',
        'BF-OUB',
        'BF-OUD',
        'BF-PAS',
        'BF-PON',
        'BF-SEN',
        'BF-SIS',
        'BF-SMT',
        'BF-SNG',
        'BF-SOM',
        'BF-SOR',
        'BF-TAP',
        'BF-TUI',
        'BF-YAG',
        'BF-YAT',
        'BF-ZIR',
        'BF-ZON',
        'BF-ZOU',
        'BG-01',
        'BG-02',
        'BG-03',
        'BG-04',
        'BG-05',
        'BG-06',
        'BG-07',
        'BG-08',
        'BG-09',
        'BG-10',
        'BG-11',
        'BG-12',
        'BG-13',
        'BG-14',
        'BG-15',
        'BG-16',
        'BG-17',
        'BG-18',
        'BG-19',
        'BG-20',
        'BG-21',
        'BG-22',
        'BG-23',
        'BG-24',
        'BG-25',
        'BG-26',
        'BG-27',
        'BG-28',
        'BH-01',
        'BH-02',
        'BH-03',
        'BH-04',
        'BH-05',
        'BH-06',
        'BH-07',
        'BH-08',
        'BH-09',
        'BH-10',
        'BH-11',
        'BH-12',
        'BI-BB',
        'BI-BJ',
        'BI-BR',
        'BI-CA',
        'BI-CI',
        'BI-GI',
        'BI-KI',
        'BI-KR',
        'BI-KY',
        'BI-MA',
        'BI-MU',
        'BI-MW',
        'BI-MY',
        'BI-NG',
        'BI-RT',
        'BI-RY',
        'BJ-AK',
        'BJ-AL',
        'BJ-AQ',
        'BJ-BO',
        'BJ-CO',
        'BJ-DO',
        'BJ-KO',
        'BJ-LI',
        'BJ-MO',
        'BJ-OU',
        'BJ-PL',
        'BJ-ZO',
        'BN-BE',
        'BN-BM',
        'BN-TE',
        'BN-TU',
        'BO-B',
        'BO-C',
        'BO-H',
        'BO-L',
        'BO-N',
        'BO-O',
        'BO-P',
        'BO-S',
        'BO-T',
        'BR-AC',
        'BR-AL',
        'BR-AM',
        'BR-AP',
        'BR-BA',
        'BR-CE',
        'BR-DF',
        'BR-ES',
        'BR-GO',
        'BR-MA',
        'BR-MG',
        'BR-MS',
        'BR-MT',
        'BR-PA',
        'BR-PB',
        'BR-PE',
        'BR-PI',
        'BR-PR',
        'BR-RJ',
        'BR-RN',
        'BR-RO',
        'BR-RR',
        'BR-RS',
        'BR-SC',
        'BR-SE',
        'BR-SP',
        'BR-TO',
        'BS-AC',
        'BS-BI',
        'BS-CI',
        'BS-EX',
        'BS-FC',
        'BS-FP',
        'BS-GH',
        'BS-GT',
        'BS-HI',
        'BS-HR',
        'BS-IN',
        'BS-KB',
        'BS-LI',
        'BS-MG',
        'BS-MH',
        'BS-NB',
        'BS-NP',
        'BS-RI',
        'BS-RS',
        'BS-SP',
        'BS-SR',
        'BT-11',
        'BT-12',
        'BT-13',
        'BT-14',
        'BT-15',
        'BT-21',
        'BT-22',
        'BT-23',
        'BT-24',
        'BT-31',
        'BT-32',
        'BT-33',
        'BT-34',
        'BT-41',
        'BT-42',
        'BT-43',
        'BT-44',
        'BT-45',
        'BT-GA',
        'BT-TY',
        'BW-CE',
        'BW-CH',
        'BW-GH',
        'BW-KG',
        'BW-KL',
        'BW-KW',
        'BW-NE',
        'BW-NG',
        'BW-SE',
        'BW-SO',
        'BY-BR',
        'BY-HO',
        'BY-HR',
        'BY-MA',
        'BY-MI',
        'BY-VI',
        'BZ-BZ',
        'BZ-CY',
        'BZ-CZL',
        'BZ-OW',
        'BZ-SC',
        'BZ-TOL',
        'CA-AB',
        'CA-BC',
        'CA-MB',
        'CA-NB',
        'CA-NL',
        'CA-NS',
        'CA-NT',
        'CA-NU',
        'CA-ON',
        'CA-PE',
        'CA-QC',
        'CA-SK',
        'CA-YT',
        'CD-BC',
        'CD-BN',
        'CD-EQ',
        'CD-KA',
        'CD-KE',
        'CD-KN',
        'CD-KW',
        'CD-MA',
        'CD-NK',
        'CD-OR',
        'CD-SK',
        'CF-AC',
        'CF-BB',
        'CF-BGF',
        'CF-BK',
        'CF-HK',
        'CF-HM',
        'CF-HS',
        'CF-KB',
        'CF-KG',
        'CF-LB',
        'CF-MB',
        'CF-MP',
        'CF-NM',
        'CF-OP',
        'CF-SE',
        'CF-UK',
        'CF-VK',
        'CG-11',
        'CG-12',
        'CG-13',
        'CG-14',
        'CG-15',
        'CG-2',
        'CG-5',
        'CG-7',
        'CG-8',
        'CG-9',
        'CG-BZV',
        'CH-AG',
        'CH-AI',
        'CH-AR',
        'CH-BE',
        'CH-BL',
        'CH-BS',
        'CH-FR',
        'CH-GE',
        'CH-GL',
        'CH-GR',
        'CH-JU',
        'CH-LU',
        'CH-NE',
        'CH-NW',
        'CH-OW',
        'CH-SG',
        'CH-SH',
        'CH-SO',
        'CH-SZ',
        'CH-TG',
        'CH-TI',
        'CH-UR',
        'CH-VD',
        'CH-VS',
        'CH-ZG',
        'CH-ZH',
        'CI-01',
        'CI-02',
        'CI-03',
        'CI-04',
        'CI-05',
        'CI-06',
        'CI-07',
        'CI-08',
        'CI-09',
        'CI-10',
        'CI-11',
        'CI-12',
        'CI-13',
        'CI-14',
        'CI-15',
        'CI-16',
        'CL-AI',
        'CL-AN',
        'CL-AR',
        'CL-AT',
        'CL-BI',
        'CL-CO',
        'CL-LI',
        'CL-LL',
        'CL-MA',
        'CL-ML',
        'CL-RM',
        'CL-TA',
        'CL-VS',
        'CM-AD',
        'CM-CE',
        'CM-EN',
        'CM-ES',
        'CM-LT',
        'CM-NO',
        'CM-NW',
        'CM-OU',
        'CM-SU',
        'CM-SW',
        'CN-11',
        'CN-12',
        'CN-13',
        'CN-14',
        'CN-15',
        'CN-21',
        'CN-22',
        'CN-23',
        'CN-31',
        'CN-32',
        'CN-33',
        'CN-34',
        'CN-35',
        'CN-36',
        'CN-37',
        'CN-41',
        'CN-42',
        'CN-43',
        'CN-44',
        'CN-45',
        'CN-46',
        'CN-50',
        'CN-51',
        'CN-52',
        'CN-53',
        'CN-54',
        'CN-61',
        'CN-62',
        'CN-63',
        'CN-64',
        'CN-65',
        'CN-71',
        'CN-91',
        'CN-92',
        'CO-AMA',
        'CO-ANT',
        'CO-ARA',
        'CO-ATL',
        'CO-BOL',
        'CO-BOY',
        'CO-CAL',
        'CO-CAQ',
        'CO-CAS',
        'CO-CAU',
        'CO-CES',
        'CO-CHO',
        'CO-COR',
        'CO-CUN',
        'CO-DC',
        'CO-GUA',
        'CO-GUV',
        'CO-HUI',
        'CO-LAG',
        'CO-MAG',
        'CO-MET',
        'CO-NAR',
        'CO-NSA',
        'CO-PUT',
        'CO-QUI',
        'CO-RIS',
        'CO-SAN',
        'CO-SAP',
        'CO-SUC',
        'CO-TOL',
        'CO-VAC',
        'CO-VAU',
        'CO-VID',
        'CR-A',
        'CR-C',
        'CR-G',
        'CR-H',
        'CR-L',
        'CR-P',
        'CR-SJ',
        'CU-01',
        'CU-02',
        'CU-03',
        'CU-04',
        'CU-05',
        'CU-06',
        'CU-07',
        'CU-08',
        'CU-09',
        'CU-10',
        'CU-11',
        'CU-12',
        'CU-13',
        'CU-14',
        'CU-99',
        'CV-B',
        'CV-BR',
        'CV-BV',
        'CV-CA',
        'CV-CR',
        'CV-CS',
        'CV-FO',
        'CV-MA',
        'CV-MO',
        'CV-PA',
        'CV-PN',
        'CV-PR',
        'CV-RG',
        'CV-S',
        'CV-SF',
        'CV-SL',
        'CV-SN',
        'CV-SV',
        'CV-TA',
        'CY-01',
        'CY-02',
        'CY-03',
        'CY-04',
        'CY-05',
        'CY-06',
        'CZ-JC',
        'CZ-JM',
        'CZ-KA',
        'CZ-KR',
        'CZ-LI',
        'CZ-MO',
        'CZ-OL',
        'CZ-PA',
        'CZ-PL',
        'CZ-PR',
        'CZ-ST',
        'CZ-US',
        'CZ-VY',
        'CZ-ZL',
        'DE-BB',
        'DE-BE',
        'DE-BW',
        'DE-BY',
        'DE-HB',
        'DE-HE',
        'DE-HH',
        'DE-MV',
        'DE-NI',
        'DE-NW',
        'DE-RP',
        'DE-SH',
        'DE-SL',
        'DE-SN',
        'DE-ST',
        'DE-TH',
        'DJ-AS',
        'DJ-DI',
        'DJ-DJ',
        'DJ-OB',
        'DJ-TA',
        'DK-015',
        'DK-020',
        'DK-025',
        'DK-030',
        'DK-035',
        'DK-040',
        'DK-042',
        'DK-050',
        'DK-055',
        'DK-060',
        'DK-065',
        'DK-070',
        'DK-076',
        'DK-080',
        'DK-101',
        'DK-147',
        'DO-01',
        'DO-02',
        'DO-03',
        'DO-04',
        'DO-05',
        'DO-06',
        'DO-07',
        'DO-08',
        'DO-09',
        'DO-10',
        'DO-11',
        'DO-12',
        'DO-13',
        'DO-14',
        'DO-15',
        'DO-16',
        'DO-17',
        'DO-18',
        'DO-19',
        'DO-20',
        'DO-21',
        'DO-22',
        'DO-23',
        'DO-24',
        'DO-25',
        'DO-26',
        'DO-27',
        'DO-28',
        'DO-29',
        'DO-30',
        'DZ-01',
        'DZ-02',
        'DZ-03',
        'DZ-04',
        'DZ-05',
        'DZ-06',
        'DZ-07',
        'DZ-08',
        'DZ-09',
        'DZ-10',
        'DZ-11',
        'DZ-12',
        'DZ-13',
        'DZ-14',
        'DZ-15',
        'DZ-16',
        'DZ-17',
        'DZ-18',
        'DZ-19',
        'DZ-20',
        'DZ-21',
        'DZ-22',
        'DZ-23',
        'DZ-24',
        'DZ-25',
        'DZ-26',
        'DZ-27',
        'DZ-28',
        'DZ-29',
        'DZ-30',
        'DZ-31',
        'DZ-32',
        'DZ-33',
        'DZ-34',
        'DZ-35',
        'DZ-36',
        'DZ-37',
        'DZ-38',
        'DZ-39',
        'DZ-40',
        'DZ-41',
        'DZ-42',
        'DZ-43',
        'DZ-44',
        'DZ-45',
        'DZ-46',
        'DZ-47',
        'DZ-48',
        'EC-A',
        'EC-B',
        'EC-C',
        'EC-D',
        'EC-E',
        'EC-F',
        'EC-G',
        'EC-H',
        'EC-I',
        'EC-L',
        'EC-M',
        'EC-N',
        'EC-O',
        'EC-P',
        'EC-R',
        'EC-S',
        'EC-T',
        'EC-U',
        'EC-W',
        'EC-X',
        'EC-Y',
        'EC-Z',
        'EE-37',
        'EE-39',
        'EE-44',
        'EE-49',
        'EE-51',
        'EE-57',
        'EE-59',
        'EE-65',
        'EE-67',
        'EE-70',
        'EE-74',
        'EE-78',
        'EE-82',
        'EE-84',
        'EE-86',
        'EG-ALX',
        'EG-ASN',
        'EG-AST',
        'EG-BA',
        'EG-BH',
        'EG-BNS',
        'EG-C',
        'EG-DK',
        'EG-DT',
        'EG-FYM',
        'EG-GH',
        'EG-GZ',
        'EG-IS',
        'EG-JS',
        'EG-KB',
        'EG-KFS',
        'EG-KN',
        'EG-MN',
        'EG-MNF',
        'EG-MT',
        'EG-PTS',
        'EG-SHG',
        'EG-SHR',
        'EG-SIN',
        'EG-SUZ',
        'EG-WAD',
        'ER-AN',
        'ER-DK',
        'ER-DU',
        'ER-GB',
        'ER-MA',
        'ER-SK',
        'ES-A',
        'ES-AB',
        'ES-AL',
        'ES-AN',
        'ES-AR',
        'ES-AV',
        'ES-B',
        'ES-BA',
        'ES-BI',
        'ES-BU',
        'ES-C',
        'ES-CA',
        'ES-CC',
        'ES-CE',
        'ES-CL',
        'ES-CM',
        'ES-CN',
        'ES-CO',
        'ES-CR',
        'ES-CS',
        'ES-CT',
        'ES-CU',
        'ES-EX',
        'ES-GA',
        'ES-GC',
        'ES-GI',
        'ES-GR',
        'ES-GU',
        'ES-H',
        'ES-HU',
        'ES-J',
        'ES-L',
        'ES-LE',
        'ES-LO',
        'ES-LU',
        'ES-M',
        'ES-MA',
        'ES-ML',
        'ES-MU',
        'ES-NA',
        'ES-O',
        'ES-OR',
        'ES-P',
        'ES-PM',
        'ES-PO',
        'ES-PV',
        'ES-S',
        'ES-SA',
        'ES-SE',
        'ES-SG',
        'ES-SO',
        'ES-SS',
        'ES-T',
        'ES-TE',
        'ES-TF',
        'ES-TO',
        'ES-V',
        'ES-VA',
        'ES-VC',
        'ES-VI',
        'ES-Z',
        'ES-ZA',
        'ET-AA',
        'ET-AF',
        'ET-AM',
        'ET-BE',
        'ET-DD',
        'ET-GA',
        'ET-HA',
        'ET-OR',
        'ET-SN',
        'ET-SO',
        'ET-TI',
        'FI-AL',
        'FI-ES',
        'FI-IS',
        'FI-LL',
        'FI-LS',
        'FI-OL',
        'FJ-C',
        'FJ-E',
        'FJ-N',
        'FJ-R',
        'FJ-W',
        'FM-KSA',
        'FM-PNI',
        'FM-TRK',
        'FM-YAP',
        'FR-01',
        'FR-02',
        'FR-03',
        'FR-04',
        'FR-05',
        'FR-06',
        'FR-07',
        'FR-08',
        'FR-09',
        'FR-10',
        'FR-11',
        'FR-12',
        'FR-13',
        'FR-14',
        'FR-15',
        'FR-16',
        'FR-17',
        'FR-18',
        'FR-19',
        'FR-21',
        'FR-22',
        'FR-23',
        'FR-24',
        'FR-25',
        'FR-26',
        'FR-27',
        'FR-28',
        'FR-29',
        'FR-2A',
        'FR-2B',
        'FR-30',
        'FR-31',
        'FR-32',
        'FR-33',
        'FR-34',
        'FR-35',
        'FR-36',
        'FR-37',
        'FR-38',
        'FR-39',
        'FR-40',
        'FR-41',
        'FR-42',
        'FR-43',
        'FR-44',
        'FR-45',
        'FR-46',
        'FR-47',
        'FR-48',
        'FR-49',
        'FR-50',
        'FR-51',
        'FR-52',
        'FR-53',
        'FR-54',
        'FR-55',
        'FR-56',
        'FR-57',
        'FR-58',
        'FR-59',
        'FR-60',
        'FR-61',
        'FR-62',
        'FR-63',
        'FR-64',
        'FR-65',
        'FR-66',
        'FR-67',
        'FR-68',
        'FR-69',
        'FR-70',
        'FR-71',
        'FR-72',
        'FR-73',
        'FR-74',
        'FR-75',
        'FR-76',
        'FR-77',
        'FR-78',
        'FR-79',
        'FR-80',
        'FR-81',
        'FR-82',
        'FR-83',
        'FR-84',
        'FR-85',
        'FR-86',
        'FR-87',
        'FR-88',
        'FR-89',
        'FR-90',
        'FR-91',
        'FR-92',
        'FR-93',
        'FR-94',
        'FR-95',
        'FR-A',
        'FR-B',
        'FR-C',
        'FR-D',
        'FR-E',
        'FR-F',
        'FR-G',
        'FR-GF',
        'FR-GP',
        'FR-H',
        'FR-I',
        'FR-J',
        'FR-K',
        'FR-L',
        'FR-M',
        'FR-MQ',
        'FR-N',
        'FR-NC',
        'FR-O',
        'FR-P',
        'FR-PF',
        'FR-PM',
        'FR-Q',
        'FR-R',
        'FR-RE',
        'FR-S',
        'FR-T',
        'FR-TF',
        'FR-U',
        'FR-V',
        'FR-WF',
        'FR-YT',
        'GA-1',
        'GA-2',
        'GA-3',
        'GA-4',
        'GA-5',
        'GA-6',
        'GA-7',
        'GA-8',
        'GA-9',
        'GB-ABD',
        'GB-ABE',
        'GB-AGB',
        'GB-AGY',
        'GB-ANS',
        'GB-ANT',
        'GB-ARD',
        'GB-ARM',
        'GB-BAS',
        'GB-BBD',
        'GB-BDF',
        'GB-BDG',
        'GB-BEN',
        'GB-BEX',
        'GB-BFS',
        'GB-BGE',
        'GB-BGW',
        'GB-BIR',
        'GB-BKM',
        'GB-BLA',
        'GB-BLY',
        'GB-BMH',
        'GB-BNB',
        'GB-BNE',
        'GB-BNH',
        'GB-BNS',
        'GB-BOL',
        'GB-BPL',
        'GB-BRC',
        'GB-BRD',
        'GB-BRY',
        'GB-BST',
        'GB-BUR',
        'GB-CAM',
        'GB-CAY',
        'GB-CGN',
        'GB-CGV',
        'GB-CHA',
        'GB-CHS',
        'GB-CKF',
        'GB-CKT',
        'GB-CLD',
        'GB-CLK',
        'GB-CLR',
        'GB-CMA',
        'GB-CMD',
        'GB-CMN',
        'GB-CON',
        'GB-COV',
        'GB-CRF',
        'GB-CRY',
        'GB-CSR',
        'GB-CWY',
        'GB-DAL',
        'GB-DBY',
        'GB-DEN',
        'GB-DER',
        'GB-DEV',
        'GB-DGN',
        'GB-DGY',
        'GB-DNC',
        'GB-DND',
        'GB-DOR',
        'GB-DOW',
        'GB-DRY',
        'GB-DUD',
        'GB-DUR',
        'GB-EAL',
        'GB-EAW',
        'GB-EAY',
        'GB-EDH',
        'GB-EDU',
        'GB-ELN',
        'GB-ELS',
        'GB-ENF',
        'GB-ENG',
        'GB-ERW',
        'GB-ERY',
        'GB-ESS',
        'GB-ESX',
        'GB-FAL',
        'GB-FER',
        'GB-FIF',
        'GB-FLN',
        'GB-GAT',
        'GB-GBN',
        'GB-GLG',
        'GB-GLS',
        'GB-GRE',
        'GB-GSY',
        'GB-GWN',
        'GB-HAL',
        'GB-HAM',
        'GB-HAV',
        'GB-HCK',
        'GB-HEF',
        'GB-HIL',
        'GB-HLD',
        'GB-HMF',
        'GB-HNS',
        'GB-HPL',
        'GB-HRT',
        'GB-HRW',
        'GB-HRY',
        'GB-IOM',
        'GB-IOS',
        'GB-IOW',
        'GB-ISL',
        'GB-IVC',
        'GB-JSY',
        'GB-KEC',
        'GB-KEN',
        'GB-KHL',
        'GB-KIR',
        'GB-KTT',
        'GB-KWL',
        'GB-LAN',
        'GB-LBH',
        'GB-LCE',
        'GB-LDS',
        'GB-LEC',
        'GB-LEW',
        'GB-LIN',
        'GB-LIV',
        'GB-LMV',
        'GB-LND',
        'GB-LRN',
        'GB-LSB',
        'GB-LUT',
        'GB-MAN',
        'GB-MDB',
        'GB-MDW',
        'GB-MFT',
        'GB-MIK',
        'GB-MLN',
        'GB-MON',
        'GB-MRT',
        'GB-MRY',
        'GB-MTY',
        'GB-MYL',
        'GB-NAY',
        'GB-NBL',
        'GB-NDN',
        'GB-NEL',
        'GB-NET',
        'GB-NFK',
        'GB-NGM',
        'GB-NIR',
        'GB-NLK',
        'GB-NLN',
        'GB-NSM',
        'GB-NTA',
        'GB-NTH',
        'GB-NTL',
        'GB-NTT',
        'GB-NTY',
        'GB-NWM',
        'GB-NWP',
        'GB-NYK',
        'GB-NYM',
        'GB-OLD',
        'GB-OMH',
        'GB-ORK',
        'GB-OXF',
        'GB-PEM',
        'GB-PKN',
        'GB-PLY',
        'GB-POL',
        'GB-POR',
        'GB-POW',
        'GB-PTE',
        'GB-RCC',
        'GB-RCH',
        'GB-RCT',
        'GB-RDB',
        'GB-RDG',
        'GB-RFW',
        'GB-RIC',
        'GB-ROT',
        'GB-RUT',
        'GB-SAW',
        'GB-SAY',
        'GB-SCB',
        'GB-SCT',
        'GB-SFK',
        'GB-SFT',
        'GB-SGC',
        'GB-SHF',
        'GB-SHN',
        'GB-SHR',
        'GB-SKP',
        'GB-SLF',
        'GB-SLG',
        'GB-SLK',
        'GB-SND',
        'GB-SOL',
        'GB-SOM',
        'GB-SOS',
        'GB-SRY',
        'GB-STB',
        'GB-STE',
        'GB-STG',
        'GB-STH',
        'GB-STN',
        'GB-STS',
        'GB-STT',
        'GB-STY',
        'GB-SWA',
        'GB-SWD',
        'GB-SWK',
        'GB-TAM',
        'GB-TFW',
        'GB-THR',
        'GB-TOB',
        'GB-TOF',
        'GB-TRF',
        'GB-TWH',
        'GB-UKM',
        'GB-VGL',
        'GB-WAR',
        'GB-WBK',
        'GB-WDU',
        'GB-WFT',
        'GB-WGN',
        'GB-WILL',
        'GB-WKF',
        'GB-WLL',
        'GB-WLN',
        'GB-WLS',
        'GB-WLV',
        'GB-WND',
        'GB-WNM',
        'GB-WOK',
        'GB-WOR',
        'GB-WRL',
        'GB-WRT',
        'GB-WRX',
        'GB-WSM',
        'GB-WSX',
        'GB-YOR',
        'GB-ZET',
        'GE-AB',
        'GE-AJ',
        'GE-GU',
        'GE-IM',
        'GE-KA',
        'GE-KK',
        'GE-MM',
        'GE-RL',
        'GE-SJ',
        'GE-SK',
        'GE-SZ',
        'GE-TB',
        'GH-AA',
        'GH-AH',
        'GH-BA',
        'GH-CP',
        'GH-EP',
        'GH-NP',
        'GH-TV',
        'GH-UE',
        'GH-UW',
        'GH-WP',
        'GM-B',
        'GM-L',
        'GM-M',
        'GM-N',
        'GM-U',
        'GM-W',
        'GN-B',
        'GN-BE',
        'GN-BF',
        'GN-BK',
        'GN-C',
        'GN-CO',
        'GN-D',
        'GN-DB',
        'GN-DI',
        'GN-DL',
        'GN-DU',
        'GN-F',
        'GN-FA',
        'GN-FO',
        'GN-FR',
        'GN-GA',
        'GN-GU',
        'GN-K',
        'GN-KA',
        'GN-KB',
        'GN-KD; 2',
        'GN-KE',
        'GN-KN',
        'GN-KO',
        'GN-KS',
        'GN-L',
        'GN-LA',
        'GN-LE',
        'GN-LO',
        'GN-M',
        'GN-MC',
        'GN-MD',
        'GN-ML',
        'GN-MM',
        'GN-N',
        'GN-NZ',
        'GN-PI',
        'GN-SI',
        'GN-TE',
        'GN-TO',
        'GN-YO',
        'GQ-AN',
        'GQ-BN',
        'GQ-BS',
        'GQ-C',
        'GQ-CS',
        'GQ-I',
        'GQ-KN',
        'GQ-LI',
        'GQ-WN',
        'GR-01',
        'GR-03',
        'GR-04',
        'GR-05',
        'GR-06',
        'GR-07',
        'GR-11',
        'GR-12',
        'GR-13',
        'GR-14',
        'GR-15',
        'GR-16',
        'GR-17',
        'GR-21',
        'GR-22',
        'GR-23',
        'GR-24',
        'GR-31',
        'GR-32',
        'GR-33',
        'GR-34',
        'GR-41',
        'GR-42',
        'GR-43',
        'GR-44',
        'GR-51',
        'GR-52',
        'GR-53',
        'GR-54',
        'GR-55',
        'GR-56',
        'GR-57',
        'GR-58',
        'GR-59',
        'GR-61',
        'GR-62',
        'GR-63',
        'GR-64',
        'GR-69',
        'GR-71',
        'GR-72',
        'GR-73',
        'GR-81',
        'GR-82',
        'GR-83',
        'GR-84',
        'GR-85',
        'GR-91',
        'GR-92',
        'GR-93',
        'GR-94',
        'GR-A1',
        'GR-I',
        'GR-II',
        'GR-III',
        'GR-IV',
        'GR-IX',
        'GR-V',
        'GR-VI',
        'GR-VII',
        'GR-VIII',
        'GR-X',
        'GR-XI',
        'GR-XII',
        'GR-XIII',
        'GT-AV',
        'GT-BV',
        'GT-CM',
        'GT-CQ',
        'GT-ES',
        'GT-GU',
        'GT-HU',
        'GT-IZ',
        'GT-JA',
        'GT-JU',
        'GT-PE',
        'GT-PR',
        'GT-QC',
        'GT-QZ',
        'GT-RE',
        'GT-SA',
        'GT-SM',
        'GT-SO',
        'GT-SR',
        'GT-SU',
        'GT-TO',
        'GT-ZA',
        'GW-BA',
        'GW-BL',
        'GW-BM',
        'GW-BS',
        'GW-CA',
        'GW-GA',
        'GW-L',
        'GW-N',
        'GW-OI',
        'GW-QU',
        'GW-S',
        'GW-TO',
        'GY-BA',
        'GY-CU',
        'GY-DE',
        'GY-EB',
        'GY-ES',
        'GY-MA',
        'GY-PM',
        'GY-PT',
        'GY-UD',
        'GY-UT',
        'HN-AT',
        'HN-CH',
        'HN-CL',
        'HN-CM',
        'HN-CP',
        'HN-CR',
        'HN-EP',
        'HN-FM',
        'HN-GD',
        'HN-IB',
        'HN-IN',
        'HN-LE',
        'HN-LP',
        'HN-OC',
        'HN-OL',
        'HN-SB',
        'HN-VA',
        'HN-YO',
        'HR-01',
        'HR-02',
        'HR-03',
        'HR-04',
        'HR-05',
        'HR-06',
        'HR-07',
        'HR-08',
        'HR-09',
        'HR-10',
        'HR-11',
        'HR-12',
        'HR-13',
        'HR-14',
        'HR-15',
        'HR-16',
        'HR-17',
        'HR-18',
        'HR-19',
        'HR-20',
        'HR-21',
        'HT-AR',
        'HT-CE',
        'HT-GA',
        'HT-ND',
        'HT-NE',
        'HT-NO',
        'HT-OU',
        'HT-SD',
        'HT-SE',
        'HU-BA',
        'HU-BC',
        'HU-BE',
        'HU-BK',
        'HU-BU',
        'HU-BZ',
        'HU-CS',
        'HU-DE',
        'HU-DU',
        'HU-EG',
        'HU-FE',
        'HU-GS',
        'HU-GY',
        'HU-HB',
        'HU-HE',
        'HU-HV',
        'HU-JN',
        'HU-KE',
        'HU-KM',
        'HU-KV',
        'HU-MI',
        'HU-NK',
        'HU-NO',
        'HU-NY',
        'HU-PE',
        'HU-PS',
        'HU-SD',
        'HU-SF',
        'HU-SH',
        'HU-SK',
        'HU-SN',
        'HU-SO',
        'HU-SS',
        'HU-ST',
        'HU-SZ',
        'HU-TB',
        'HU-TO',
        'HU-VA',
        'HU-VE',
        'HU-VM',
        'HU-ZA',
        'HU-ZE',
        'ID-AC',
        'ID-BA',
        'ID-BB',
        'ID-BE',
        'ID-BT',
        'ID-GO',
        'ID-IJ',
        'ID-JA',
        'ID-JB',
        'ID-JI',
        'ID-JK',
        'ID-JT',
        'ID-JW',
        'ID-KA',
        'ID-KB',
        'ID-KI',
        'ID-KS',
        'ID-KT',
        'ID-LA',
        'ID-MA',
        'ID-MU',
        'ID-NB',
        'ID-NT',
        'ID-NU',
        'ID-PA',
        'ID-RI',
        'ID-SA',
        'ID-SB',
        'ID-SG',
        'ID-SL',
        'ID-SM',
        'ID-SN',
        'ID-SS',
        'ID-ST',
        'ID-SU',
        'ID-YO',
        'IE-C',
        'IE-C; 2',
        'IE-CE',
        'IE-CN',
        'IE-CW',
        'IE-D',
        'IE-DL',
        'IE-G',
        'IE-KE',
        'IE-KK',
        'IE-KY',
        'IE-L',
        'IE-LD',
        'IE-LH',
        'IE-LK',
        'IE-LM',
        'IE-LS',
        'IE-M',
        'IE-MH',
        'IE-MN',
        'IE-MO',
        'IE-OY',
        'IE-RN',
        'IE-SO',
        'IE-TA',
        'IE-U',
        'IE-WD',
        'IE-WH',
        'IE-WW',
        'IE-WX',
        'IL-D',
        'IL-HA',
        'IL-JM',
        'IL-M',
        'IL-TA',
        'IL-Z',
        'IN-AN',
        'IN-AP',
        'IN-AR',
        'IN-AS',
        'IN-BR',
        'IN-CH',
        'IN-CT',
        'IN-DD',
        'IN-DL',
        'IN-DN',
        'IN-GA',
        'IN-GJ',
        'IN-HP',
        'IN-HR',
        'IN-JH',
        'IN-JK',
        'IN-KA',
        'IN-KL',
        'IN-LD',
        'IN-MH',
        'IN-ML',
        'IN-MN',
        'IN-MP',
        'IN-MZ',
        'IN-NL',
        'IN-OR',
        'IN-PB',
        'IN-PY',
        'IN-RJ',
        'IN-SK',
        'IN-TN',
        'IN-TR',
        'IN-UL',
        'IN-UP',
        'IN-WB',
        'IQ-AN',
        'IQ-AR',
        'IQ-BA',
        'IQ-BB',
        'IQ-BG',
        'IQ-DA',
        'IQ-DI',
        'IQ-DQ',
        'IQ-KA',
        'IQ-MA',
        'IQ-MU',
        'IQ-NA',
        'IQ-NI',
        'IQ-QA',
        'IQ-SD',
        'IQ-SU',
        'IQ-TS',
        'IQ-WA',
        'IR-01',
        'IR-02',
        'IR-03',
        'IR-04',
        'IR-05',
        'IR-06',
        'IR-07',
        'IR-08',
        'IR-09',
        'IR-10',
        'IR-11',
        'IR-12',
        'IR-13',
        'IR-14',
        'IR-15',
        'IR-16',
        'IR-17',
        'IR-18',
        'IR-19',
        'IR-20',
        'IR-21',
        'IR-22',
        'IR-23',
        'IR-24',
        'IR-25',
        'IR-26',
        'IR-27',
        'IR-28',
        'IS-0',
        'IS-1',
        'IS-2',
        'IS-3',
        'IS-4',
        'IS-5',
        'IS-6',
        'IS-7',
        'IS-8',
        'IT-21',
        'IT-23',
        'IT-25',
        'IT-32',
        'IT-34',
        'IT-36',
        'IT-42',
        'IT-45',
        'IT-52',
        'IT-55',
        'IT-57',
        'IT-62',
        'IT-65',
        'IT-67',
        'IT-72',
        'IT-75',
        'IT-77',
        'IT-78',
        'IT-82',
        'IT-88',
        'IT-AG',
        'IT-AL',
        'IT-AN',
        'IT-AO',
        'IT-AP',
        'IT-AQ',
        'IT-AR',
        'IT-AT',
        'IT-AV',
        'IT-BA',
        'IT-BG',
        'IT-BI',
        'IT-BL',
        'IT-BN',
        'IT-BO',
        'IT-BR',
        'IT-BS',
        'IT-BZ',
        'IT-CA',
        'IT-CB',
        'IT-CE',
        'IT-CH',
        'IT-CL',
        'IT-CN',
        'IT-CO',
        'IT-CR',
        'IT-CS',
        'IT-CT',
        'IT-CZ',
        'IT-DU',
        'IT-EN',
        'IT-FE',
        'IT-FG',
        'IT-FI',
        'IT-FO',
        'IT-FR',
        'IT-GE',
        'IT-GO',
        'IT-GR',
        'IT-IM',
        'IT-IS',
        'IT-KR',
        'IT-LC',
        'IT-LE',
        'IT-LI',
        'IT-LO',
        'IT-LT',
        'IT-LU',
        'IT-MC',
        'IT-ME',
        'IT-MI',
        'IT-MN',
        'IT-MO',
        'IT-MS',
        'IT-MT',
        'IT-NA',
        'IT-NO',
        'IT-NU',
        'IT-OR',
        'IT-PA',
        'IT-PC',
        'IT-PD',
        'IT-PE',
        'IT-PG',
        'IT-PI',
        'IT-PN',
        'IT-PO',
        'IT-PR',
        'IT-PS',
        'IT-PT',
        'IT-PV',
        'IT-PZ',
        'IT-RA',
        'IT-RC',
        'IT-RE',
        'IT-RG',
        'IT-RI',
        'IT-RM',
        'IT-RN',
        'IT-RO',
        'IT-SA',
        'IT-SI',
        'IT-SO',
        'IT-SP',
        'IT-SR',
        'IT-SS',
        'IT-SV',
        'IT-TA',
        'IT-TE',
        'IT-TN',
        'IT-TO',
        'IT-TP',
        'IT-TR',
        'IT-TS',
        'IT-TV',
        'IT-VA',
        'IT-VB',
        'IT-VC',
        'IT-VE',
        'IT-VI',
        'IT-VR',
        'IT-VT',
        'IT-VV',
        'JM-01',
        'JM-02',
        'JM-03',
        'JM-04',
        'JM-05',
        'JM-06',
        'JM-07',
        'JM-08',
        'JM-09',
        'JM-10',
        'JM-11',
        'JM-12',
        'JM-13',
        'JM-14',
        'JO-AJ',
        'JO-AM',
        'JO-AQ',
        'JO-AT',
        'JO-AZ',
        'JO-BA',
        'JO-IR',
        'JO-JA',
        'JO-KA',
        'JO-MA',
        'JO-MD',
        'JO-MN',
        'JP-01',
        'JP-02',
        'JP-03',
        'JP-04',
        'JP-05',
        'JP-06',
        'JP-07',
        'JP-08',
        'JP-09',
        'JP-10',
        'JP-11',
        'JP-12',
        'JP-13',
        'JP-14',
        'JP-15',
        'JP-16',
        'JP-17',
        'JP-18',
        'JP-19',
        'JP-20',
        'JP-21',
        'JP-22',
        'JP-23',
        'JP-24',
        'JP-25',
        'JP-26',
        'JP-27',
        'JP-28',
        'JP-29',
        'JP-30',
        'JP-31',
        'JP-32',
        'JP-33',
        'JP-34',
        'JP-35',
        'JP-36',
        'JP-37',
        'JP-38',
        'JP-39',
        'JP-40',
        'JP-41',
        'JP-42',
        'JP-43',
        'JP-44',
        'JP-45',
        'JP-46',
        'JP-47',
        'KE-110',
        'KE-200',
        'KE-300',
        'KE-400',
        'KE-500',
        'KE-600',
        'KE-700',
        'KE-900',
        'KG-B',
        'KG-C',
        'KG-GB',
        'KG-J',
        'KG-N',
        'KG-O',
        'KG-T',
        'KG-Y',
        'KH-1',
        'KH-10',
        'KH-11',
        'KH-12',
        'KH-13',
        'KH-14',
        'KH-15',
        'KH-16',
        'KH-17',
        'KH-18',
        'KH-19',
        'KH-2',
        'KH-20',
        'KH-21',
        'KH-22',
        'KH-23',
        'KH-24',
        'KH-3',
        'KH-4',
        'KH-5',
        'KH-6',
        'KH-7',
        'KH-8',
        'KH-9',
        'KI-G',
        'KI-L',
        'KI-P',
        'KM-A',
        'KM-G',
        'KM-M',
        'KP-CHA',
        'KP-HAB',
        'KP-HAN',
        'KP-HWB',
        'KP-HWN',
        'KP-KAE',
        'KP-KAN',
        'KP-NAJ',
        'KP-NAM',
        'KP-PYB',
        'KP-PYN',
        'KP-PYO',
        'KP-YAN',
        'KR-11',
        'KR-26',
        'KR-27',
        'KR-28',
        'KR-29',
        'KR-30',
        'KR-31',
        'KR-41',
        'KR-42',
        'KR-43',
        'KR-44',
        'KR-45',
        'KR-46',
        'KR-47',
        'KR-48',
        'KR-49',
        'KW-AH',
        'KW-FA',
        'KW-HA',
        'KW-JA',
        'KW-KU',
        'KZ-AKM',
        'KZ-AKT',
        'KZ-ALA',
        'KZ-ALM',
        'KZ-AST',
        'KZ-ATY',
        'KZ-KAR',
        'KZ-KUS',
        'KZ-KZY',
        'KZ-MAN',
        'KZ-PAV',
        'KZ-SEV',
        'KZ-VOS',
        'KZ-YUZ',
        'KZ-ZAP',
        'KZ-ZHA',
        'LA-AT',
        'LA-BK',
        'LA-BL',
        'LA-CH',
        'LA-HO',
        'LA-KH',
        'LA-LM',
        'LA-LP',
        'LA-OU',
        'LA-PH',
        'LA-SL',
        'LA-SV',
        'LA-VI',
        'LA-VT',
        'LA-XA',
        'LA-XE',
        'LA-XI',
        'LA-XN',
        'LB-AS',
        'LB-BA',
        'LB-BI',
        'LB-JA',
        'LB-JL',
        'LB-NA',
        'LK-1',
        'LK-11',
        'LK-12',
        'LK-13',
        'LK-2',
        'LK-21',
        'LK-22',
        'LK-23',
        'LK-3',
        'LK-31',
        'LK-32',
        'LK-33',
        'LK-4',
        'LK-41',
        'LK-42',
        'LK-43',
        'LK-44',
        'LK-45',
        'LK-5',
        'LK-51',
        'LK-52',
        'LK-53',
        'LK-6',
        'LK-61',
        'LK-62',
        'LK-7',
        'LK-71',
        'LK-72',
        'LK-8',
        'LK-81',
        'LK-82',
        'LK-9',
        'LK-91',
        'LK-92',
        'LR-BG',
        'LR-BM',
        'LR-CM',
        'LR-GB',
        'LR-GG',
        'LR-GK',
        'LR-LO',
        'LR-MG',
        'LR-MO',
        'LR-MY',
        'LR-NI',
        'LR-RI',
        'LR-SI',
        'LS-A',
        'LS-B',
        'LS-C',
        'LS-D',
        'LS-E',
        'LS-F',
        'LS-G',
        'LS-H',
        'LS-J',
        'LS-K',
        'LT-AL',
        'LT-KL',
        'LT-KU',
        'LT-MR',
        'LT-PN',
        'LT-SA',
        'LT-TA',
        'LT-TE',
        'LT-UT',
        'LT-VL',
        'LU-D',
        'LU-G',
        'LU-L',
        'LV-AI',
        'LV-AL',
        'LV-BL',
        'LV-BU',
        'LV-CE',
        'LV-DA',
        'LV-DGV',
        'LV-DO',
        'LV-GU',
        'LV-JEL',
        'LV-JK',
        'LV-JL',
        'LV-JUR',
        'LV-KR',
        'LV-KU',
        'LV-LE',
        'LV-LM',
        'LV-LPX',
        'LV-LU',
        'LV-MA',
        'LV-OG',
        'LV-PR',
        'LV-RE',
        'LV-REZ',
        'LV-RI',
        'LV-RIX',
        'LV-SA',
        'LV-TA',
        'LV-TU',
        'LV-VE',
        'LV-VEN',
        'LV-VK',
        'LV-VM',
        'LY-BA',
        'LY-BU',
        'LY-FA',
        'LY-JA',
        'LY-JG',
        'LY-JU',
        'LY-MI',
        'LY-NA',
        'LY-SF',
        'LY-TB',
        'LY-WA',
        'LY-WU',
        'LY-ZA',
        'MA-01',
        'MA-02',
        'MA-03',
        'MA-04',
        'MA-05',
        'MA-06',
        'MA-07',
        'MA-08',
        'MA-09',
        'MA-10',
        'MA-11',
        'MA-12',
        'MA-13',
        'MA-14',
        'MA-15',
        'MA-16',
        'MA-AGD',
        'MA-ASZ',
        'MA-AZI',
        'MA-BAH',
        'MA-BEM',
        'MA-BER',
        'MA-BES',
        'MA-BOD',
        'MA-BOM',
        'MA-CAS',
        'MA-CHE',
        'MA-CHI',
        'MA-ERR',
        'MA-ESI',
        'MA-ESM',
        'MA-FES',
        'MA-FIG',
        'MA-GUE',
        'MA-HAJ',
        'MA-HAO',
        'MA-HOC',
        'MA-IFR',
        'MA-JDI',
        'MA-JRA',
        'MA-KEN',
        'MA-KES',
        'MA-KHE',
        'MA-KHN',
        'MA-KHO',
        'MA-LAA',
        'MA-LAR',
        'MA-MAR',
        'MA-MEK',
        'MA-MEL',
        'MA-NAD',
        'MA-OUA',
        'MA-OUD',
        'MA-OUJ',
        'MA-RBA',
        'MA-SAF',
        'MA-SEF',
        'MA-SET',
        'MA-SIK',
        'MA-TAO',
        'MA-TAR',
        'MA-TAT',
        'MA-TAZ',
        'MA-TET',
        'MA-TIZ',
        'MA-TNG',
        'MA-TNT',
        'MD-BA',
        'MD-CA',
        'MD-CH',
        'MD-CU',
        'MD-ED',
        'MD-GA',
        'MD-LA',
        'MD-OR',
        'MD-SN',
        'MD-SO',
        'MD-TA',
        'MD-TI',
        'MD-UN',
        'MG-A',
        'MG-D',
        'MG-F',
        'MG-M',
        'MG-T',
        'MG-U',
        'MH-ALK',
        'MH-ALL',
        'MH-ARN',
        'MH-AUR',
        'MH-EBO',
        'MH-ENI',
        'MH-JAL',
        'MH-KIL',
        'MH-KWA',
        'MH-L',
        'MH-LAE',
        'MH-LIB',
        'MH-LIK',
        'MH-MAJ',
        'MH-MAL',
        'MH-MEJ',
        'MH-MIL',
        'MH-NMK',
        'MH-NMU',
        'MH-RON',
        'MH-T',
        'MH-UJA',
        'MH-UJL',
        'MH-UTI',
        'MH-WTH',
        'MH-WTJ',
        'ML-1',
        'ML-2',
        'ML-3',
        'ML-4',
        'ML-5',
        'ML-6',
        'ML-7',
        'ML-8',
        'ML-BKO',
        'MM-01',
        'MM-02',
        'MM-03',
        'MM-04',
        'MM-05',
        'MM-06',
        'MM-07',
        'MM-11',
        'MM-12',
        'MM-13',
        'MM-14',
        'MM-15',
        'MM-16',
        'MM-17',
        'MN-035',
        'MN-037',
        'MN-039',
        'MN-041',
        'MN-043',
        'MN-046',
        'MN-047',
        'MN-049',
        'MN-051',
        'MN-053',
        'MN-055',
        'MN-057',
        'MN-059',
        'MN-061',
        'MN-063',
        'MN-064',
        'MN-065',
        'MN-067',
        'MN-069',
        'MN-071',
        'MN-073',
        'MN-1',
        'MR-01',
        'MR-02',
        'MR-03',
        'MR-04',
        'MR-05',
        'MR-06',
        'MR-07',
        'MR-08',
        'MR-09',
        'MR-10',
        'MR-11',
        'MR-12',
        'MR-NKC',
        'MU-AG',
        'MU-BL',
        'MU-BR',
        'MU-CC',
        'MU-CU',
        'MU-FL',
        'MU-GP',
        'MU-MO',
        'MU-PA',
        'MU-PL',
        'MU-PU',
        'MU-PW',
        'MU-QB',
        'MU-RO',
        'MU-RR',
        'MU-SA',
        'MU-VP',
        'MV-01',
        'MV-02',
        'MV-03',
        'MV-04',
        'MV-05',
        'MV-07',
        'MV-08',
        'MV-12',
        'MV-13',
        'MV-14',
        'MV-17',
        'MV-20',
        'MV-23',
        'MV-24',
        'MV-25',
        'MV-26',
        'MV-27',
        'MV-28',
        'MV-29',
        'MV-MLE',
        'MW-BA',
        'MW-BL',
        'MW-C',
        'MW-CK',
        'MW-CR',
        'MW-CT',
        'MW-DE',
        'MW-DO',
        'MW-KR',
        'MW-KS',
        'MW-LI',
        'MW-LK',
        'MW-MC',
        'MW-MG',
        'MW-MH',
        'MW-MU',
        'MW-MW',
        'MW-MZ',
        'MW-N',
        'MW-NB',
        'MW-NI',
        'MW-NK',
        'MW-NS',
        'MW-NU',
        'MW-PH',
        'MW-RU',
        'MW-S',
        'MW-SA',
        'MW-TH',
        'MW-ZO',
        'MX-AGU',
        'MX-BCN',
        'MX-BCS',
        'MX-CAM',
        'MX-CHH',
        'MX-CHP',
        'MX-COA',
        'MX-COL',
        'MX-DIF',
        'MX-DUR',
        'MX-GRO',
        'MX-GUA',
        'MX-HID',
        'MX-JAL',
        'MX-MEX',
        'MX-MIC',
        'MX-MOR',
        'MX-NAY',
        'MX-NLE',
        'MX-OAX',
        'MX-PUE',
        'MX-QUE',
        'MX-ROO',
        'MX-SIN',
        'MX-SLP',
        'MX-SON',
        'MX-TAB',
        'MX-TAM',
        'MX-TLA',
        'MX-VER',
        'MX-YUC',
        'MX-ZAC',
        'MY-A',
        'MY-B',
        'MY-C',
        'MY-D',
        'MY-J',
        'MY-K',
        'MY-L',
        'MY-M',
        'MY-N',
        'MY-P',
        'MY-R',
        'MY-SA',
        'MY-SK',
        'MY-T',
        'MY-W',
        'MZ-A',
        'MZ-B',
        'MZ-G',
        'MZ-I',
        'MZ-L',
        'MZ-MPM',
        'MZ-N',
        'MZ-P',
        'MZ-Q',
        'MZ-S',
        'MZ-T',
        'NA-CA',
        'NA-ER',
        'NA-HA',
        'NA-KA',
        'NA-KH',
        'NA-KU',
        'NA-OD',
        'NA-OH',
        'NA-OK',
        'NA-ON',
        'NA-OS',
        'NA-OT',
        'NA-OW',
        'NE-1',
        'NE-2',
        'NE-3',
        'NE-4',
        'NE-5',
        'NE-6',
        'NE-7',
        'NE-8',
        'NG-AB',
        'NG-AD',
        'NG-AK',
        'NG-AN',
        'NG-BA',
        'NG-BE',
        'NG-BO',
        'NG-BY',
        'NG-CR',
        'NG-DE',
        'NG-EB',
        'NG-ED',
        'NG-EK',
        'NG-EN',
        'NG-FC',
        'NG-GO',
        'NG-IM',
        'NG-JI',
        'NG-KD',
        'NG-KE',
        'NG-KN',
        'NG-KO',
        'NG-KT',
        'NG-KW',
        'NG-LA',
        'NG-NA',
        'NG-NI',
        'NG-OG',
        'NG-ON',
        'NG-OS',
        'NG-OY',
        'NG-PL',
        'NG-RI',
        'NG-SO',
        'NG-TA',
        'NG-YO',
        'NG-ZA',
        'NI-AN',
        'NI-AS',
        'NI-BO',
        'NI-CA',
        'NI-CI',
        'NI-CO',
        'NI-ES',
        'NI-GR',
        'NI-JI',
        'NI-LE',
        'NI-MD',
        'NI-MN',
        'NI-MS',
        'NI-MT',
        'NI-NS',
        'NI-RI',
        'NI-SJ',
        'NL-DR',
        'NL-FL',
        'NL-FR',
        'NL-GE',
        'NL-GR',
        'NL-LI',
        'NL-NB',
        'NL-NH',
        'NL-OV',
        'NL-UT',
        'NL-ZE',
        'NL-ZH',
        'NO-01',
        'NO-02',
        'NO-03',
        'NO-04',
        'NO-05',
        'NO-06',
        'NO-07',
        'NO-08',
        'NO-09',
        'NO-10',
        'NO-11',
        'NO-12',
        'NO-14',
        'NO-15',
        'NO-16',
        'NO-17',
        'NO-18',
        'NO-19',
        'NO-20',
        'NO-21',
        'NO-22',
        'NP-1',
        'NP-2',
        'NP-3',
        'NP-4',
        'NP-5',
        'NP-BA',
        'NP-BH',
        'NP-DH',
        'NP-GA',
        'NP-JA',
        'NP-KA',
        'NP-KO',
        'NP-LU',
        'NP-MA',
        'NP-ME',
        'NP-NA',
        'NP-RA',
        'NP-SA',
        'NP-SE',
        'NZ-AUK',
        'NZ-BOP',
        'NZ-CAN',
        'NZ-GIS',
        'NZ-HKB',
        'NZ-MBH',
        'NZ-MWT',
        'NZ-N',
        'NZ-NSN',
        'NZ-NTL',
        'NZ-OTA',
        'NZ-S',
        'NZ-STL',
        'NZ-TAS',
        'NZ-TKI',
        'NZ-WGN',
        'NZ-WKO',
        'NZ-WTC',
        'OM-BA',
        'OM-DA',
        'OM-JA',
        'OM-MA',
        'OM-MU',
        'OM-SH',
        'OM-WU',
        'OM-ZA',
        'PA-0',
        'PA-1',
        'PA-2',
        'PA-3',
        'PA-4',
        'PA-5',
        'PA-6',
        'PA-7',
        'PA-8',
        'PA-9',
        'PE-AMA',
        'PE-ANC',
        'PE-APU',
        'PE-ARE',
        'PE-AYA',
        'PE-CAJ',
        'PE-CAL',
        'PE-CUS',
        'PE-HUC',
        'PE-HUV',
        'PE-ICA',
        'PE-JUN',
        'PE-LAL',
        'PE-LAM',
        'PE-LIM',
        'PE-LOR',
        'PE-MDD',
        'PE-MOQ',
        'PE-PAS',
        'PE-PIU',
        'PE-PUN',
        'PE-SAM',
        'PE-TAC',
        'PE-TUM',
        'PE-UCA',
        'PG-CPK',
        'PG-CPM',
        'PG-EBR',
        'PG-EHG',
        'PG-EPW',
        'PG-ESW',
        'PG-GPK',
        'PG-MBA',
        'PG-MPL',
        'PG-MPM',
        'PG-MRL',
        'PG-NCD',
        'PG-NIK',
        'PG-NPP',
        'PG-NSA',
        'PG-SAN',
        'PG-SHM',
        'PG-WBK',
        'PG-WHM',
        'PG-WPD',
        'PH-00',
        'PH-01',
        'PH-02',
        'PH-03',
        'PH-04',
        'PH-05',
        'PH-06',
        'PH-07',
        'PH-08',
        'PH-09',
        'PH-10',
        'PH-11',
        'PH-12',
        'PH-13',
        'PH-14',
        'PH-15',
        'PH-ABR',
        'PH-AGN',
        'PH-AGS',
        'PH-AKL',
        'PH-ALB',
        'PH-ANT',
        'PH-APA',
        'PH-AUR',
        'PH-BAN',
        'PH-BAS',
        'PH-BEN',
        'PH-BIL',
        'PH-BOH',
        'PH-BTG',
        'PH-BTN',
        'PH-BUK',
        'PH-BUL',
        'PH-CAG',
        'PH-CAM',
        'PH-CAN',
        'PH-CAP',
        'PH-CAS',
        'PH-CAT',
        'PH-CAV',
        'PH-CEB',
        'PH-COM',
        'PH-DAO',
        'PH-DAS',
        'PH-DAV',
        'PH-EAS',
        'PH-GUI',
        'PH-IFU',
        'PH-ILI',
        'PH-ILN',
        'PH-ILS',
        'PH-ISA',
        'PH-KAL',
        'PH-LAG',
        'PH-LAN',
        'PH-LAS',
        'PH-LEY',
        'PH-LUN',
        'PH-MAD',
        'PH-MAG',
        'PH-MAS',
        'PH-MDC',
        'PH-MDR',
        'PH-MOU',
        'PH-MSC',
        'PH-MSR',
        'PH-NCO',
        'PH-NEC',
        'PH-NER',
        'PH-NSA',
        'PH-NUE',
        'PH-NUV',
        'PH-PAM',
        'PH-PAN',
        'PH-PLW',
        'PH-QUE',
        'PH-QUI',
        'PH-RIZ',
        'PH-ROM',
        'PH-SAR',
        'PH-SCO',
        'PH-SIG',
        'PH-SLE',
        'PH-SLU',
        'PH-SOR',
        'PH-SUK',
        'PH-SUN',
        'PH-SUR',
        'PH-TAR',
        'PH-TAW',
        'PH-WSA',
        'PH-ZAN',
        'PH-ZAS',
        'PH-ZMB',
        'PH-ZSI',
        'PK-BA',
        'PK-IS',
        'PK-JK',
        'PK-NA',
        'PK-NW',
        'PK-PB',
        'PK-SD',
        'PK-TA',
        'PL-DS',
        'PL-KP',
        'PL-LB',
        'PL-LD',
        'PL-LU',
        'PL-MA',
        'PL-MZ',
        'PL-OP',
        'PL-PD',
        'PL-PK',
        'PL-PM',
        'PL-SK',
        'PL-SL',
        'PL-WN',
        'PL-WP',
        'PL-ZP',
        'PT-01',
        'PT-02',
        'PT-03',
        'PT-04',
        'PT-05',
        'PT-06',
        'PT-07',
        'PT-08',
        'PT-09',
        'PT-10',
        'PT-11',
        'PT-12',
        'PT-13',
        'PT-14',
        'PT-15',
        'PT-16',
        'PT-17',
        'PT-18',
        'PT-20',
        'PT-30',
        'PY-1',
        'PY-10',
        'PY-11',
        'PY-12',
        'PY-13',
        'PY-14',
        'PY-15',
        'PY-16',
        'PY-19',
        'PY-2',
        'PY-3',
        'PY-4',
        'PY-5',
        'PY-6',
        'PY-7',
        'PY-8',
        'PY-9',
        'PY-ASU',
        'QA-DA',
        'QA-GH',
        'QA-JB',
        'QA-JU',
        'QA-KH',
        'QA-MS',
        'QA-RA',
        'QA-US',
        'QA-WA',
        'RO-AB',
        'RO-AG',
        'RO-AR',
        'RO-B',
        'RO-BC',
        'RO-BH',
        'RO-BN',
        'RO-BR',
        'RO-BT',
        'RO-BV',
        'RO-BZ',
        'RO-CJ',
        'RO-CL',
        'RO-CS',
        'RO-CT',
        'RO-CV',
        'RO-DB',
        'RO-DJ',
        'RO-GJ',
        'RO-GL',
        'RO-GR',
        'RO-HD',
        'RO-HR',
        'RO-IF',
        'RO-IL',
        'RO-IS',
        'RO-MH',
        'RO-MM',
        'RO-MS',
        'RO-NT',
        'RO-OT',
        'RO-PH',
        'RO-SB',
        'RO-SJ',
        'RO-SM',
        'RO-SV',
        'RO-TL',
        'RO-TM',
        'RO-TR',
        'RO-VL',
        'RO-VN',
        'RO-VS',
        'RU-AD',
        'RU-AGB',
        'RU-AL',
        'RU-ALT',
        'RU-AMU',
        'RU-ARK',
        'RU-AST',
        'RU-BA',
        'RU-BEL',
        'RU-BRY',
        'RU-BU',
        'RU-CE',
        'RU-CHE',
        'RU-CHI',
        'RU-CHU',
        'RU-CU',
        'RU-DA',
        'RU-DU',
        'RU-EVE',
        'RU-IN',
        'RU-IRK',
        'RU-IVA',
        'RU-KAM',
        'RU-KB',
        'RU-KC',
        'RU-KDA',
        'RU-KEM',
        'RU-KGD',
        'RU-KGN',
        'RU-KHA',
        'RU-KHM',
        'RU-KIR',
        'RU-KK',
        'RU-KL',
        'RU-KLU',
        'RU-KO',
        'RU-KOP',
        'RU-KOR',
        'RU-KOS',
        'RU-KR',
        'RU-KRS',
        'RU-KYA',
        'RU-LEN',
        'RU-LIP',
        'RU-MAG',
        'RU-ME',
        'RU-MO',
        'RU-MOS',
        'RU-MOW',
        'RU-MUR',
        'RU-NEN',
        'RU-NGR',
        'RU-NIZ',
        'RU-NVS',
        'RU-OMS',
        'RU-ORE',
        'RU-ORL',
        'RU-PER',
        'RU-PNZ',
        'RU-PRI',
        'RU-PSK',
        'RU-ROS',
        'RU-RYA',
        'RU-SA',
        'RU-SAK',
        'RU-SAM',
        'RU-SAR',
        'RU-SE',
        'RU-SMO',
        'RU-SPE',
        'RU-STA',
        'RU-SVE',
        'RU-TA',
        'RU-TAM',
        'RU-TAY',
        'RU-TOM',
        'RU-TUL',
        'RU-TVE',
        'RU-TY',
        'RU-TYU',
        'RU-ULY',
        'RU-UOB',
        'RU-VGG',
        'RU-VLA',
        'RU-VLG',
        'RU-VOR',
        'RU-YAN',
        'RU-YAR',
        'RU-YEV',
        'RW-B',
        'RW-C',
        'RW-D',
        'RW-E',
        'RW-F',
        'RW-G',
        'RW-H',
        'RW-I',
        'RW-J',
        'RW-K',
        'RW-L',
        'RW-M',
        'SA-01',
        'SA-02',
        'SA-03',
        'SA-04',
        'SA-05',
        'SA-06',
        'SA-07',
        'SA-08',
        'SA-09',
        'SA-10',
        'SA-11',
        'SA-12',
        'SA-14',
        'SB-CE',
        'SB-CT',
        'SB-GU',
        'SB-IS',
        'SB-MK',
        'SB-ML',
        'SB-TE',
        'SB-WE',
        'SD-01',
        'SD-02',
        'SD-03',
        'SD-04',
        'SD-05',
        'SD-06',
        'SD-07',
        'SD-08',
        'SD-09',
        'SD-10',
        'SD-11',
        'SD-12',
        'SD-13',
        'SD-14',
        'SD-15',
        'SD-16',
        'SD-17',
        'SD-18',
        'SD-19',
        'SD-20',
        'SD-21',
        'SD-22',
        'SD-23',
        'SD-24',
        'SD-25',
        'SD-26',
        'SE-AB',
        'SE-AC',
        'SE-BD',
        'SE-C',
        'SE-D',
        'SE-E',
        'SE-F',
        'SE-G',
        'SE-H',
        'SE-I',
        'SE-K',
        'SE-M',
        'SE-N',
        'SE-O',
        'SE-S',
        'SE-T',
        'SE-U',
        'SE-W',
        'SE-X',
        'SE-Y',
        'SE-Z',
        'SH-AC',
        'SH-SH',
        'SH-TA',
        'SI-01',
        'SI-02',
        'SI-03',
        'SI-04',
        'SI-05',
        'SI-06',
        'SI-07',
        'SI-08',
        'SI-09',
        'SI-10',
        'SI-11',
        'SI-12',
        'SK-BC',
        'SK-BL',
        'SK-KI',
        'SK-NI',
        'SK-PV',
        'SK-TA',
        'SK-TC',
        'SK-ZI',
        'SL-E',
        'SL-N',
        'SL-S',
        'SL-W',
        'SN-DB',
        'SN-DK',
        'SN-FK',
        'SN-KD',
        'SN-KL',
        'SN-LG',
        'SN-SL',
        'SN-TC',
        'SN-TH',
        'SN-ZG',
        'SO-AW',
        'SO-BK',
        'SO-BN',
        'SO-BR',
        'SO-BY',
        'SO-GA',
        'SO-GE',
        'SO-HI',
        'SO-JD',
        'SO-JH',
        'SO-MU',
        'SO-NU',
        'SO-SA',
        'SO-SD',
        'SO-SH',
        'SO-SO',
        'SO-TO',
        'SO-WO',
        'SR-BR',
        'SR-CM',
        'SR-CR',
        'SR-MA',
        'SR-NI',
        'SR-PM',
        'SR-PR',
        'SR-SA',
        'SR-SI',
        'SR-WA',
        'ST-P',
        'ST-S',
        'SV-AH',
        'SV-CA',
        'SV-CH',
        'SV-CU',
        'SV-LI',
        'SV-MO',
        'SV-PA',
        'SV-SA',
        'SV-SM',
        'SV-SO',
        'SV-SS',
        'SV-SV',
        'SV-UN',
        'SV-US',
        'SY-DI',
        'SY-DR',
        'SY-DY',
        'SY-HA',
        'SY-HI',
        'SY-HL',
        'SY-HM',
        'SY-ID',
        'SY-LA',
        'SY-QU',
        'SY-RA',
        'SY-RD',
        'SY-SU',
        'SY-TA',
        'SZ-HH',
        'SZ-LU',
        'SZ-MA',
        'SZ-SH',
        'TD-BA',
        'TD-BET',
        'TD-BI',
        'TD-CB',
        'TD-GR',
        'TD-KA',
        'TD-LC',
        'TD-LO',
        'TD-LR',
        'TD-MC',
        'TD-MK',
        'TD-OD',
        'TD-SA',
        'TD-TA',
        'TG-C',
        'TG-K',
        'TG-M',
        'TG-P',
        'TG-S',
        'TH-10',
        'TH-11',
        'TH-12',
        'TH-13',
        'TH-14',
        'TH-15',
        'TH-16',
        'TH-17',
        'TH-18',
        'TH-19',
        'TH-20',
        'TH-21',
        'TH-22',
        'TH-23',
        'TH-24',
        'TH-25',
        'TH-26',
        'TH-27',
        'TH-30',
        'TH-31',
        'TH-32',
        'TH-33',
        'TH-34',
        'TH-35',
        'TH-36',
        'TH-37',
        'TH-39',
        'TH-40',
        'TH-41',
        'TH-42',
        'TH-43',
        'TH-44',
        'TH-45',
        'TH-46',
        'TH-47',
        'TH-48',
        'TH-49',
        'TH-50',
        'TH-51',
        'TH-52',
        'TH-53',
        'TH-54',
        'TH-55',
        'TH-56',
        'TH-57',
        'TH-58',
        'TH-60',
        'TH-61',
        'TH-62',
        'TH-63',
        'TH-64',
        'TH-65',
        'TH-66',
        'TH-67',
        'TH-70',
        'TH-71',
        'TH-72',
        'TH-73',
        'TH-74',
        'TH-75',
        'TH-76',
        'TH-77',
        'TH-80',
        'TH-81',
        'TH-82',
        'TH-83',
        'TH-84',
        'TH-85',
        'TH-86',
        'TH-90',
        'TH-91',
        'TH-92',
        'TH-93',
        'TH-94',
        'TH-95',
        'TH-96',
        'TH-S',
        'TJ-GB',
        'TJ-KT',
        'TJ-SU',
        'TL-AL',
        'TL-AN',
        'TL-BA',
        'TL-BO',
        'TL-CO',
        'TL-DI',
        'TL-ER',
        'TL-LA',
        'TL-LI',
        'TL-MF',
        'TL-MT',
        'TL-OE',
        'TL-VI',
        'TM-A',
        'TM-B',
        'TM-D',
        'TM-L',
        'TM-M',
        'TN-11',
        'TN-12',
        'TN-13',
        'TN-21',
        'TN-22',
        'TN-23',
        'TN-31',
        'TN-32',
        'TN-33',
        'TN-34',
        'TN-41',
        'TN-42',
        'TN-43',
        'TN-51',
        'TN-52',
        'TN-53',
        'TN-61',
        'TN-71',
        'TN-72',
        'TN-73',
        'TN-81',
        'TN-82',
        'TN-83',
        'TR-01',
        'TR-02',
        'TR-03',
        'TR-04',
        'TR-05',
        'TR-06',
        'TR-07',
        'TR-08',
        'TR-09',
        'TR-10',
        'TR-11',
        'TR-12',
        'TR-13',
        'TR-14',
        'TR-15',
        'TR-16',
        'TR-17',
        'TR-18',
        'TR-19',
        'TR-20',
        'TR-21',
        'TR-22',
        'TR-23',
        'TR-24',
        'TR-25',
        'TR-26',
        'TR-27',
        'TR-28',
        'TR-29',
        'TR-30',
        'TR-31',
        'TR-32',
        'TR-33',
        'TR-34',
        'TR-35',
        'TR-36',
        'TR-37',
        'TR-38',
        'TR-39',
        'TR-40',
        'TR-41',
        'TR-42',
        'TR-43',
        'TR-44',
        'TR-45',
        'TR-46',
        'TR-47',
        'TR-48',
        'TR-49',
        'TR-50',
        'TR-51',
        'TR-52',
        'TR-53',
        'TR-54',
        'TR-55',
        'TR-56',
        'TR-57',
        'TR-58',
        'TR-59',
        'TR-60',
        'TR-61',
        'TR-62',
        'TR-63',
        'TR-64',
        'TR-65',
        'TR-66',
        'TR-67',
        'TR-68',
        'TR-69',
        'TR-70',
        'TR-71',
        'TR-72',
        'TR-73',
        'TR-74',
        'TR-75',
        'TR-76',
        'TR-77',
        'TR-78',
        'TR-79',
        'TR-80',
        'TR-81',
        'TT-ARI',
        'TT-CHA',
        'TT-CTT',
        'TT-DMN',
        'TT-ETO',
        'TT-PED',
        'TT-POS',
        'TT-PRT',
        'TT-PTF',
        'TT-RCM',
        'TT-SFO',
        'TT-SGE',
        'TT-SIP',
        'TT-SJL',
        'TT-TUP',
        'TT-WTO',
        'TW-CHA',
        'TW-CYQ',
        'TW-HSQ',
        'TW-HUA',
        'TW-ILA',
        'TW-KEE',
        'TW-KHQ',
        'TW-MIA',
        'TW-NAN',
        'TW-PEN',
        'TW-PIF',
        'TW-TAO',
        'TW-TNQ',
        'TW-TPQ',
        'TW-TTT',
        'TW-TXQ',
        'TW-YUN',
        'TZ-01',
        'TZ-02',
        'TZ-03',
        'TZ-04',
        'TZ-05',
        'TZ-06',
        'TZ-07',
        'TZ-08',
        'TZ-09',
        'TZ-10',
        'TZ-11',
        'TZ-12',
        'TZ-13',
        'TZ-14',
        'TZ-15',
        'TZ-16',
        'TZ-17',
        'TZ-18',
        'TZ-19',
        'TZ-20',
        'TZ-21',
        'TZ-22',
        'TZ-23',
        'TZ-24',
        'TZ-25',
        'UA-05',
        'UA-07',
        'UA-09',
        'UA-12',
        'UA-14',
        'UA-18',
        'UA-21',
        'UA-23',
        'UA-26',
        'UA-30',
        'UA-32',
        'UA-35',
        'UA-40',
        'UA-43',
        'UA-46',
        'UA-48',
        'UA-51',
        'UA-53',
        'UA-56',
        'UA-59',
        'UA-61',
        'UA-63',
        'UA-65',
        'UA-68',
        'UA-71',
        'UA-74',
        'UA-77',
        'UG-AJM',
        'UG-APA',
        'UG-ARU',
        'UG-BUA',
        'UG-BUG',
        'UG-BUN',
        'UG-BUS',
        'UG-C',
        'UG-E',
        'UG-GUL',
        'UG-HOI',
        'UG-IGA',
        'UG-JIN',
        'UG-KAP',
        'UG-KAS',
        'UG-KAT',
        'UG-KBL',
        'UG-KBR',
        'UG-KIB',
        'UG-KIS',
        'UG-KIT',
        'UG-KLA',
        'UG-KLE',
        'UG-KLG',
        'UG-KLI',
        'UG-KOT',
        'UG-KUM',
        'UG-LIR',
        'UG-LUW',
        'UG-MBL',
        'UG-MBR',
        'UG-MOR',
        'UG-MOY',
        'UG-MPI',
        'UG-MSI',
        'UG-MSK',
        'UG-MUB',
        'UG-MUK',
        'UG-N',
        'UG-NAK',
        'UG-NEB',
        'UG-NTU',
        'UG-PAL',
        'UG-RAK',
        'UG-RUK',
        'UG-SEM',
        'UG-SOR',
        'UG-TOR',
        'UG-W',
        'UM-67',
        'UM-71',
        'UM-76',
        'UM-79',
        'UM-81',
        'UM-84',
        'UM-86',
        'UM-89',
        'UM-95',
        'US-AK',
        'US-AL',
        'US-AR',
        'US-AS',
        'US-AZ',
        'US-CA',
        'US-CO',
        'US-CT',
        'US-DC',
        'US-DE',
        'US-FL',
        'US-GA',
        'US-GU',
        'US-HI',
        'US-IA',
        'US-ID',
        'US-IL',
        'US-IN',
        'US-KS',
        'US-KY',
        'US-LA',
        'US-MA',
        'US-MD',
        'US-ME',
        'US-MI',
        'US-MN',
        'US-MO',
        'US-MP',
        'US-MS',
        'US-MT',
        'US-NC',
        'US-ND',
        'US-NE',
        'US-NH',
        'US-NJ',
        'US-NM',
        'US-NV',
        'US-NY',
        'US-OH',
        'US-OK',
        'US-OR',
        'US-PA',
        'US-PR',
        'US-RI',
        'US-SC',
        'US-SD',
        'US-TN',
        'US-TX',
        'US-UM',
        'US-UT',
        'US-VA',
        'US-VI',
        'US-VT',
        'US-WA',
        'US-WI',
        'US-WV',
        'US-WY',
        'UY-AR',
        'UY-CA',
        'UY-CL',
        'UY-CO',
        'UY-DU',
        'UY-FD',
        'UY-FS',
        'UY-LA',
        'UY-MA',
        'UY-MO',
        'UY-PA',
        'UY-RN',
        'UY-RO',
        'UY-RV',
        'UY-SA',
        'UY-SJ',
        'UY-SO',
        'UY-TA',
        'UY-TT',
        'UZ-AN',
        'UZ-BU',
        'UZ-FA',
        'UZ-JI',
        'UZ-NG',
        'UZ-NW',
        'UZ-QA',
        'UZ-QR',
        'UZ-SA',
        'UZ-SI',
        'UZ-SU',
        'UZ-TK',
        'UZ-TO',
        'UZ-XO',
        'VE-A',
        'VE-B',
        'VE-C',
        'VE-D',
        'VE-E',
        'VE-F',
        'VE-G',
        'VE-H',
        'VE-I',
        'VE-J',
        'VE-K',
        'VE-L',
        'VE-M',
        'VE-N',
        'VE-O',
        'VE-P',
        'VE-R',
        'VE-S',
        'VE-T',
        'VE-U',
        'VE-V',
        'VE-W',
        'VE-X',
        'VE-Y',
        'VE-Z',
        'VN-01',
        'VN-02',
        'VN-03',
        'VN-04',
        'VN-05',
        'VN-06',
        'VN-07',
        'VN-09',
        'VN-13',
        'VN-14',
        'VN-15',
        'VN-18',
        'VN-20',
        'VN-21',
        'VN-22',
        'VN-23',
        'VN-24',
        'VN-25',
        'VN-26',
        'VN-27',
        'VN-28',
        'VN-29',
        'VN-30',
        'VN-31',
        'VN-32',
        'VN-33',
        'VN-34',
        'VN-35',
        'VN-36',
        'VN-37',
        'VN-39',
        'VN-40',
        'VN-41',
        'VN-43',
        'VN-44',
        'VN-45',
        'VN-46',
        'VN-47',
        'VN-48',
        'VN-49',
        'VN-50',
        'VN-51',
        'VN-52',
        'VN-53',
        'VN-54',
        'VN-55',
        'VN-56',
        'VN-57',
        'VN-58',
        'VN-59',
        'VN-60',
        'VN-61',
        'VN-62',
        'VN-63',
        'VN-64',
        'VN-65',
        'VN-66',
        'VN-67',
        'VN-68',
        'VN-69',
        'VN-70',
        'VU-MAP',
        'VU-PAM',
        'VU-SAM',
        'VU-SEE',
        'VU-TAE',
        'VU-TOB',
        'WS-AA',
        'WS-AL',
        'WS-AT',
        'WS-FA',
        'WS-GE',
        'WS-GI',
        'WS-PA',
        'WS-SA',
        'WS-TU',
        'WS-VF',
        'WS-VS',
        'YE-AB',
        'YE-AD',
        'YE-AM',
        'YE-BA',
        'YE-DA',
        'YE-DH',
        'YE-HD',
        'YE-HJ',
        'YE-HU',
        'YE-IB',
        'YE-JA',
        'YE-LA',
        'YE-MA',
        'YE-MR',
        'YE-MW',
        'YE-SD',
        'YE-SH',
        'YE-SN',
        'YE-TA',
        'YU-CG',
        'YU-KM',
        'YU-SR',
        'YU-VO',
        'ZA-EC',
        'ZA-FS',
        'ZA-GT',
        'ZA-MP',
        'ZA-NC',
        'ZA-NL',
        'ZA-NP',
        'ZA-NW',
        'ZA-WC',
        'ZM-01',
        'ZM-02',
        'ZM-03',
        'ZM-04',
        'ZM-05',
        'ZM-06',
        'ZM-07',
        'ZM-08',
        'ZM-09',
        'ZW-BU',
        'ZW-HA',
        'ZW-MA',
        'ZW-MC',
        'ZW-ME',
        'ZW-MI',
        'ZW-MN',
        'ZW-MS',
        'ZW-MV',
        'ZW-MW'
    )
)

Subdivision.__doc__ = """
Subvidision country codes from ISO 3166-2.

Taken from `here <http://www.gefeg.com/
edifact/d03a/s3/codes/cl1h.htm>`_.
"""


class Layouts(Enum):
    """Keyboard layouts. Taken from Debian's 9
    /usr/share/X11/xkb/rules/evdev.lst.
    """
    US = 'English (US)'
    AF = 'Afghani'
    ARA = 'Arabic'
    AL = 'Albanian'
    AM = 'Armenian'
    AT = 'German (Austria)'
    AU = 'English (Australian)'
    AZ = 'Azerbaijani'
    BY = 'Belarusian'
    BE = 'Belgian'
    BD = 'Bangla'
    BA = 'Bosnian'
    BR = 'Portuguese (Brazil)'
    BG = 'Bulgarian'
    DZ = 'Berber (Algeria, Latin characters)'
    MA = 'Arabic (Morocco)'
    CM = 'English (Cameroon)'
    MM = 'Burmese'
    CA = 'French (Canada)'
    CD = 'French (Democratic Republic of the Congo)'
    CN = 'Chinese'
    HR = 'Croatian'
    CZ = 'Czech'
    DK = 'Danish'
    NL = 'Dutch'
    BT = 'Dzongkha'
    EE = 'Estonian'
    IR = 'Persian'
    IQ = 'Iraqi'
    FO = 'Faroese'
    FI = 'Finnish'
    FR = 'French'
    GH = 'English (Ghana)'
    GN = 'French (Guinea)'
    GE = 'Georgian'
    DE = 'German'
    GR = 'Greek'
    HU = 'Hungarian'
    IL = 'Hebrew'
    IT = 'Italian'
    JP = 'Japanese'
    KG = 'Kyrgyz'
    KH = 'Khmer (Cambodia)'
    KZ = 'Kazakh'
    LA = 'Lao'
    LATAM = 'Spanish (Latin American)'
    LT = 'Lithuanian'
    LV = 'Latvian'
    MAO = 'Maori'
    ME = 'Montenegrin'
    MK = 'Macedonian'
    MT = 'Maltese'
    MN = 'Mongolian'
    NO = 'Norwegian'
    PL = 'Polish'
    PT = 'Portuguese'
    RO = 'Romanian'
    RU = 'Russian'
    RS = 'Serbian'
    SI = 'Slovenian'
    SK = 'Slovak'
    ES = 'Spanish'
    SE = 'Swedish'
    CH = 'German (Switzerland)'
    SY = 'Arabic (Syria)'
    TJ = 'Tajik'
    LK = 'Sinhala (phonetic)'
    TH = 'Thai'
    TR = 'Turkish'
    TW = 'Taiwanese'
    UA = 'Ukrainian'
    GB = 'English (UK)'
    UZ = 'Uzbek'
    VN = 'Vietnamese'
    KR = 'Korean'
    IE = 'Irish'
    PK = 'Urdu (Pakistan)'
    MV = 'Dhivehi'
    ZA = 'English (South Africa)'
    EPO = 'Esperanto'
    NP = 'Nepali'
    NG = 'English (Nigeria)'
    ET = 'Amharic'
    SN = 'Wolof'
    BRAI = 'Braille'
    TM = 'Turkmen'
    ML = 'Bambara'
    TZ = 'Swahili (Tanzania)'
    TG = 'French (Togo)'
    KE = 'Swahili (Kenya)'
    BW = 'Tswana'
    PH = 'Filipino'
    MD = 'Moldavian'
    ID = 'Indonesian (Jawi)'
    MY = 'Malay (Jawi)'
    BN = 'Malay (Jawi)'
    IN = 'Indian'
    IS = 'Icelandic'
    NEC_VNDR_JP = 'Japanese (PC-98xx Series)'

    def __str__(self):
        return self.value
