import React, { useState, useEffect, useRef } from 'react';
import { createUseStyles } from 'react-jss';
import { VariableSizeList } from 'react-window';
import AutoSizer from 'react-virtualized-auto-sizer';
import { useStoreState } from 'pullstate';
import { UIStore } from '../stores/UIStore';
import { TextField } from '../components/TextField';
import { HorizontalHeading } from '../components/HorizontalHeading';
import { DIDListItem } from '../components/@Explore/DIDListItem';
import { Spinning } from '../components/Spinning';
import { withRequestAPI } from '../utils/Actions';
import { InlineDropdown } from '../components/@Explore/InlineDropdown';
const useStyles = createUseStyles({
    mainContainer: {
        height: '100%',
        display: 'flex',
        flexDirection: 'column',
        overflow: 'hidden'
    },
    searchContainer: {
        padding: '8px'
    },
    filterContainer: {
        padding: '0 16px 0 16px',
        fontSize: '9pt'
    },
    resultsContainer: {
        flex: 1
    },
    searchButton: {
        alignItems: 'center',
        padding: '4px',
        lineHeight: 0,
        cursor: 'pointer'
    },
    searchIcon: {
        color: 'var(--jp-rucio-primary-blue-color)',
        opacity: 0.5,
        '&:hover': {
            opacity: 1
        }
    },
    dropdown: {
        color: 'var(--jp-rucio-primary-blue-color)',
        cursor: 'pointer',
        marginLeft: '4px'
    },
    loading: {
        padding: '16px'
    },
    icon: {
        fontSize: '10pt',
        verticalAlign: 'middle'
    },
    iconText: {
        verticalAlign: 'middle',
        paddingLeft: '4px'
    }
});
const searchByOptions = [
    { title: 'Datasets or Containers', value: 'collection' },
    { title: 'Datasets', value: 'dataset' },
    { title: 'Containers', value: 'container' },
    { title: 'Files', value: 'file' },
    { title: 'All', value: 'all' }
];
const _Explore = props => {
    const classes = useStyles();
    const { actions } = props;
    const [searchQuery, setSearchQuery] = useState('');
    const [searchType, setSearchType] = useState('collection');
    const [searchResult, setSearchResult] = useState();
    const [didExpanded, setDidExpanded] = useState({});
    const [error, setError] = useState();
    const [lastQuery, setLastQuery] = useState('');
    const [loading, setLoading] = useState(false);
    const activeInstance = useStoreState(UIStore, s => s.activeInstance);
    const doSearch = () => {
        setLastQuery(searchQuery);
    };
    const itemsSortFunction = (a, b) => {
        if (a.type === b.type) {
            return a.did.toLowerCase() < b.did.toLowerCase() ? -1 : 1;
        }
        if (a.type === 'container' && b.type === 'dataset') {
            return -1;
        }
        if ((a.type === 'container' || a.type === 'dataset') && b.type === 'file') {
            return -1;
        }
        return 1;
    };
    useEffect(() => {
        if (!lastQuery) {
            return;
        }
        setLoading(true);
        setSearchResult(undefined);
        setDidExpanded({});
        setError(undefined);
        actions
            .searchDID(activeInstance.name, searchQuery, searchType)
            .then(items => items.sort(itemsSortFunction))
            .then(result => setSearchResult(result))
            .catch(e => {
            setSearchResult([]);
            if (e.response.status === 401) {
                setError('Authentication error. Perhaps you set an invalid credential?');
                return;
            }
            if (e.response.status === 400) {
                setError('Wildcard search is disabled.');
            }
        })
            .finally(() => setLoading(false));
    }, [lastQuery, searchType]);
    const searchButton = (React.createElement("div", { className: classes.searchButton, onClick: () => setLastQuery(searchQuery) },
        React.createElement("i", { className: `${classes.searchIcon} material-icons` }, "search")));
    const handleKeyPress = (e) => {
        if (e.key === 'Enter') {
            doSearch();
        }
    };
    const listRef = useRef();
    const toggleExpand = (index) => {
        listRef.current.resetAfterIndex(index);
        didExpanded[index] = !didExpanded[index];
        setDidExpanded(didExpanded);
    };
    const getItemHeight = (i) => (didExpanded[i] === true ? 64 : 32);
    const Row = ({ index, style }) => {
        const item = searchResult[index];
        const expanded = !!didExpanded[index];
        return (React.createElement(DIDListItem, { style: style, type: item.type, did: item.did, size: item.size, key: item.did, expand: expanded, onClick: () => toggleExpand(index) }));
    };
    return (React.createElement("div", { className: classes.mainContainer },
        React.createElement("div", { className: classes.searchContainer },
            React.createElement(TextField, { placeholder: "Enter a Data Identifier (DID)", after: searchButton, value: searchQuery, onChange: e => setSearchQuery(e.target.value), onKeyPress: handleKeyPress })),
        React.createElement("div", { className: classes.filterContainer },
            "Search",
            React.createElement(InlineDropdown, { className: classes.dropdown, options: searchByOptions, value: searchType, onItemSelected: setSearchType, optionWidth: "180px" })),
        loading && (React.createElement("div", { className: classes.loading },
            React.createElement(Spinning, { className: `${classes.icon} material-icons` }, "hourglass_top"),
            React.createElement("span", { className: classes.iconText }, "Loading..."))),
        !!searchResult && (React.createElement(React.Fragment, null,
            React.createElement(HorizontalHeading, { title: "Search Results" }),
            ((!!searchResult && searchResult.length === 0) || !!error) && (React.createElement("div", { className: classes.loading }, error || 'No results found')),
            React.createElement("div", { className: classes.resultsContainer },
                React.createElement(AutoSizer, { disableWidth: true }, ({ height }) => (React.createElement(VariableSizeList, { ref: listRef, height: height, itemCount: searchResult.length, itemSize: getItemHeight, width: "100%" }, Row))))))));
};
export const Explore = withRequestAPI(_Explore);
