"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.voiceConnectorValidator = void 0;
const _1 = require(".");
var ISO_3166_ALPHA_2 = /^A[^ABCHJKNPVY]|B[^CKPUX]|C[^BEJPQST]|D[EJKMOZ]|E[CEGHRST]|F[IJKMOR]|G[^CJKOVXZ]|H[KMNRTU]|I[DEL-OQ-T]|J[EMOP]|K[EGHIMNPRWYZ]|L[ABCIKR-VY]|M[^BIJ]|N[ACEFGILOPRUZ]|OM|P[AE-HK-NRSTWY]|QA|R[EOSUW]|S[^FPQUW]|T[^ABEIPQSUXY]|U[AGMSYZ]|V[ACEGINU]|WF|WS|YE|YT|Z[AMW]$/;
var RFC_1918 = /(^192\.168\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])$)|(^172\.([1][6-9]|[2][0-9]|[3][0-1])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])$)|(^10\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])$)/;
var IP_ADDRESS = /^(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$/;
var VALID_CIDR = /^(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\/(2[7-9])|(3[0-2])$/;
var RFC_1918_CIDR = /(^192\.168\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])(\/((2[7-9])|(3[0-2]))$))|(^172\.([1][6-9]|[2][0-9]|[3][0-1])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])(\/((2[7-9])|(3[0-2]))$)$)|(^10\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])(\/((2[7-9])|(3[0-2]))$))/;
var FQDN = /^(?!.*?_.*?)(?!(?:[\w]+?\.)?\-[\w\.\-]*?)(?![\w]+?\-\.(?:[\w\.\-]+?))(?=[\w])(?=[\w\.\-]*?\.+[\w\.\-]*?)(?![\w\.\-]{254})(?!(?:\.?[\w\-\.]*?[\w\-]{64,}\.)+?)[\w\.\-]+?(?<![\w\-\.]*?\.[\d]+?)(?<=[\w\-]{2,})(?<![\w\-]{25})$/;
function voiceConnectorValidator(props) {
    if (props.region) {
        if (!(props.region.includes('Token[AWS.Region.') ||
            props.region === 'us-east-1' ||
            props.region === 'us-west-2')) {
            throw new Error('Region must be us-east-1 or us-west-2');
        }
    }
    if (props.termination) {
        for (var country of props.termination.callingRegions) {
            if (!ISO_3166_ALPHA_2.test(country)) {
                throw new Error(`Invalid Country: ${country}`);
            }
        }
    }
    if (props.termination) {
        for (var terminationCidr of props.termination.terminationCidrs) {
            if (terminationCidr.includes('Token')) {
            }
            else if (!VALID_CIDR.test(terminationCidr)) {
                throw new Error('Termination CIDR must be a valid non-RFC1918 IPv4 CIDR block (/27-/32)');
            }
        }
    }
    if (props.termination) {
        for (var terminationCidr of props.termination.terminationCidrs) {
            if (RFC_1918_CIDR.test(terminationCidr)) {
                throw new Error('Termination CDIR must not be RFC1918 CIDR block (/27-/32)');
            }
        }
    }
    if (props.origination) {
        for (var route of props.origination) {
            if (route.host.includes('Token')) {
            }
            else if (IP_ADDRESS.test(route.host)) {
                if (RFC_1918.test(route.host)) {
                    throw new Error('Origination IP must not be RFC1918 IP Address');
                }
            }
            else if (!FQDN.test(route.host)) {
                throw new Error('Origination IP must be valid, non-RFC1918 IPv4 Address or FQDN');
            }
        }
    }
    if (props.origination) {
        for (var route of props.origination) {
            if (route.port < 1 || route.port > 65535) {
                throw new Error('Port range must be valid: 1-65535');
            }
        }
    }
    if (props.origination && props.encryption) {
        for (var route of props.origination) {
            if (route.protocol === _1.Protocol.UDP) {
                throw new Error('TCP must be used with Encryption enabled');
            }
        }
    }
    return true;
}
exports.voiceConnectorValidator = voiceConnectorValidator;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidm9pY2VDb25uZWN0b3JWYWxpZGF0b3IuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvdm9pY2VDb25uZWN0b3JWYWxpZGF0b3IudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsd0JBQWtEO0FBRWxELElBQUksZ0JBQWdCLEdBQ2xCLHFRQUFxUSxDQUFDO0FBRXhRLElBQUksUUFBUSxHQUNWLHVUQUF1VCxDQUFDO0FBRTFULElBQUksVUFBVSxHQUNaLDZGQUE2RixDQUFDO0FBRWhHLElBQUksVUFBVSxHQUNaLGdIQUFnSCxDQUFDO0FBRW5ILElBQUksYUFBYSxHQUNmLDZYQUE2WCxDQUFDO0FBRWhZLElBQUksSUFBSSxHQUNOLCtOQUErTixDQUFDO0FBRWxPLFNBQWdCLHVCQUF1QixDQUFDLEtBQTBCO0lBQ2hFLElBQUksS0FBSyxDQUFDLE1BQU0sRUFBRTtRQUNoQixJQUNFLENBQUMsQ0FDQyxLQUFLLENBQUMsTUFBTSxDQUFDLFFBQVEsQ0FBQyxtQkFBbUIsQ0FBQztZQUMxQyxLQUFLLENBQUMsTUFBTSxLQUFLLFdBQVc7WUFDNUIsS0FBSyxDQUFDLE1BQU0sS0FBSyxXQUFXLENBQzdCLEVBQ0Q7WUFDQSxNQUFNLElBQUksS0FBSyxDQUFDLHVDQUF1QyxDQUFDLENBQUM7U0FDMUQ7S0FDRjtJQUVELElBQUksS0FBSyxDQUFDLFdBQVcsRUFBRTtRQUNyQixLQUFLLElBQUksT0FBTyxJQUFJLEtBQUssQ0FBQyxXQUFXLENBQUMsY0FBYyxFQUFFO1lBQ3BELElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLEVBQUU7Z0JBQ25DLE1BQU0sSUFBSSxLQUFLLENBQUMsb0JBQW9CLE9BQU8sRUFBRSxDQUFDLENBQUM7YUFDaEQ7U0FDRjtLQUNGO0lBRUQsSUFBSSxLQUFLLENBQUMsV0FBVyxFQUFFO1FBQ3JCLEtBQUssSUFBSSxlQUFlLElBQUksS0FBSyxDQUFDLFdBQVcsQ0FBQyxnQkFBZ0IsRUFBRTtZQUM5RCxJQUFJLGVBQWUsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEVBQUU7YUFDdEM7aUJBQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLEVBQUU7Z0JBQzVDLE1BQU0sSUFBSSxLQUFLLENBQ2Isd0VBQXdFLENBQ3pFLENBQUM7YUFDSDtTQUNGO0tBQ0Y7SUFFRCxJQUFJLEtBQUssQ0FBQyxXQUFXLEVBQUU7UUFDckIsS0FBSyxJQUFJLGVBQWUsSUFBSSxLQUFLLENBQUMsV0FBVyxDQUFDLGdCQUFnQixFQUFFO1lBQzlELElBQUksYUFBYSxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsRUFBRTtnQkFDdkMsTUFBTSxJQUFJLEtBQUssQ0FDYiwyREFBMkQsQ0FDNUQsQ0FBQzthQUNIO1NBQ0Y7S0FDRjtJQUVELElBQUksS0FBSyxDQUFDLFdBQVcsRUFBRTtRQUNyQixLQUFLLElBQUksS0FBSyxJQUFJLEtBQUssQ0FBQyxXQUFXLEVBQUU7WUFDbkMsSUFBSSxLQUFLLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRTthQUNqQztpQkFBTSxJQUFJLFVBQVUsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxFQUFFO2dCQUN0QyxJQUFJLFFBQVEsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxFQUFFO29CQUM3QixNQUFNLElBQUksS0FBSyxDQUFDLCtDQUErQyxDQUFDLENBQUM7aUJBQ2xFO2FBQ0Y7aUJBQU0sSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxFQUFFO2dCQUNqQyxNQUFNLElBQUksS0FBSyxDQUNiLGdFQUFnRSxDQUNqRSxDQUFDO2FBQ0g7U0FDRjtLQUNGO0lBRUQsSUFBSSxLQUFLLENBQUMsV0FBVyxFQUFFO1FBQ3JCLEtBQUssSUFBSSxLQUFLLElBQUksS0FBSyxDQUFDLFdBQVcsRUFBRTtZQUNuQyxJQUFJLEtBQUssQ0FBQyxJQUFJLEdBQUcsQ0FBQyxJQUFJLEtBQUssQ0FBQyxJQUFJLEdBQUcsS0FBSyxFQUFFO2dCQUN4QyxNQUFNLElBQUksS0FBSyxDQUFDLG1DQUFtQyxDQUFDLENBQUM7YUFDdEQ7U0FDRjtLQUNGO0lBRUQsSUFBSSxLQUFLLENBQUMsV0FBVyxJQUFJLEtBQUssQ0FBQyxVQUFVLEVBQUU7UUFDekMsS0FBSyxJQUFJLEtBQUssSUFBSSxLQUFLLENBQUMsV0FBVyxFQUFFO1lBQ25DLElBQUksS0FBSyxDQUFDLFFBQVEsS0FBSyxXQUFRLENBQUMsR0FBRyxFQUFFO2dCQUNuQyxNQUFNLElBQUksS0FBSyxDQUFDLDBDQUEwQyxDQUFDLENBQUM7YUFDN0Q7U0FDRjtLQUNGO0lBRUQsT0FBTyxJQUFJLENBQUM7QUFDZCxDQUFDO0FBMUVELDBEQTBFQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IFByb3RvY29sLCBWb2ljZUNvbm5lY3RvclByb3BzIH0gZnJvbSAnLic7XG5cbnZhciBJU09fMzE2Nl9BTFBIQV8yID1cbiAgL15BW15BQkNISktOUFZZXXxCW15DS1BVWF18Q1teQkVKUFFTVF18RFtFSktNT1pdfEVbQ0VHSFJTVF18RltJSktNT1JdfEdbXkNKS09WWFpdfEhbS01OUlRVXXxJW0RFTC1PUS1UXXxKW0VNT1BdfEtbRUdISU1OUFJXWVpdfExbQUJDSUtSLVZZXXxNW15CSUpdfE5bQUNFRkdJTE9QUlVaXXxPTXxQW0FFLUhLLU5SU1RXWV18UUF8UltFT1NVV118U1teRlBRVVddfFRbXkFCRUlQUVNVWFldfFVbQUdNU1laXXxWW0FDRUdJTlVdfFdGfFdTfFlFfFlUfFpbQU1XXSQvO1xuXG52YXIgUkZDXzE5MTggPVxuICAvKF4xOTJcXC4xNjhcXC4oWzAtOV18WzAtOV1bMC05XXxbMC0yXVswLTVdWzAtNV0pXFwuKFswLTldfFswLTldWzAtOV18WzAtMl1bMC01XVswLTVdKSQpfCheMTcyXFwuKFsxXVs2LTldfFsyXVswLTldfFszXVswLTFdKVxcLihbMC05XXxbMC05XVswLTldfFswLTJdWzAtNV1bMC01XSlcXC4oWzAtOV18WzAtOV1bMC05XXxbMC0yXVswLTVdWzAtNV0pJCl8KF4xMFxcLihbMC05XXxbMC05XVswLTldfFswLTJdWzAtNV1bMC01XSlcXC4oWzAtOV18WzAtOV1bMC05XXxbMC0yXVswLTVdWzAtNV0pXFwuKFswLTldfFswLTldWzAtOV18WzAtMl1bMC01XVswLTVdKSQpLztcblxudmFyIElQX0FERFJFU1MgPVxuICAvXig/Oig/OjI1WzAtNV18MlswLTRdWzAtOV18WzAxXT9bMC05XVswLTldPylcXC4pezN9KD86MjVbMC01XXwyWzAtNF1bMC05XXxbMDFdP1swLTldWzAtOV0/KSQvO1xuXG52YXIgVkFMSURfQ0lEUiA9XG4gIC9eKD86KD86MjVbMC01XXwyWzAtNF1bMC05XXxbMDFdP1swLTldWzAtOV0/KVxcLil7M30oPzoyNVswLTVdfDJbMC00XVswLTldfFswMV0/WzAtOV1bMC05XT8pXFwvKDJbNy05XSl8KDNbMC0yXSkkLztcblxudmFyIFJGQ18xOTE4X0NJRFIgPVxuICAvKF4xOTJcXC4xNjhcXC4oWzAtOV18WzAtOV1bMC05XXxbMC0yXVswLTVdWzAtNV0pXFwuKFswLTldfFswLTldWzAtOV18WzAtMl1bMC01XVswLTVdKShcXC8oKDJbNy05XSl8KDNbMC0yXSkpJCkpfCheMTcyXFwuKFsxXVs2LTldfFsyXVswLTldfFszXVswLTFdKVxcLihbMC05XXxbMC05XVswLTldfFswLTJdWzAtNV1bMC01XSlcXC4oWzAtOV18WzAtOV1bMC05XXxbMC0yXVswLTVdWzAtNV0pKFxcLygoMls3LTldKXwoM1swLTJdKSkkKSQpfCheMTBcXC4oWzAtOV18WzAtOV1bMC05XXxbMC0yXVswLTVdWzAtNV0pXFwuKFswLTldfFswLTldWzAtOV18WzAtMl1bMC01XVswLTVdKVxcLihbMC05XXxbMC05XVswLTldfFswLTJdWzAtNV1bMC01XSkoXFwvKCgyWzctOV0pfCgzWzAtMl0pKSQpKS87XG5cbnZhciBGUUROID1cbiAgL14oPyEuKj9fLio/KSg/ISg/OltcXHddKz9cXC4pP1xcLVtcXHdcXC5cXC1dKj8pKD8hW1xcd10rP1xcLVxcLig/OltcXHdcXC5cXC1dKz8pKSg/PVtcXHddKSg/PVtcXHdcXC5cXC1dKj9cXC4rW1xcd1xcLlxcLV0qPykoPyFbXFx3XFwuXFwtXXsyNTR9KSg/ISg/OlxcLj9bXFx3XFwtXFwuXSo/W1xcd1xcLV17NjQsfVxcLikrPylbXFx3XFwuXFwtXSs/KD88IVtcXHdcXC1cXC5dKj9cXC5bXFxkXSs/KSg/PD1bXFx3XFwtXXsyLH0pKD88IVtcXHdcXC1dezI1fSkkLztcblxuZXhwb3J0IGZ1bmN0aW9uIHZvaWNlQ29ubmVjdG9yVmFsaWRhdG9yKHByb3BzOiBWb2ljZUNvbm5lY3RvclByb3BzKSB7XG4gIGlmIChwcm9wcy5yZWdpb24pIHtcbiAgICBpZiAoXG4gICAgICAhKFxuICAgICAgICBwcm9wcy5yZWdpb24uaW5jbHVkZXMoJ1Rva2VuW0FXUy5SZWdpb24uJykgfHxcbiAgICAgICAgcHJvcHMucmVnaW9uID09PSAndXMtZWFzdC0xJyB8fFxuICAgICAgICBwcm9wcy5yZWdpb24gPT09ICd1cy13ZXN0LTInXG4gICAgICApXG4gICAgKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1JlZ2lvbiBtdXN0IGJlIHVzLWVhc3QtMSBvciB1cy13ZXN0LTInKTtcbiAgICB9XG4gIH1cblxuICBpZiAocHJvcHMudGVybWluYXRpb24pIHtcbiAgICBmb3IgKHZhciBjb3VudHJ5IG9mIHByb3BzLnRlcm1pbmF0aW9uLmNhbGxpbmdSZWdpb25zKSB7XG4gICAgICBpZiAoIUlTT18zMTY2X0FMUEhBXzIudGVzdChjb3VudHJ5KSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYEludmFsaWQgQ291bnRyeTogJHtjb3VudHJ5fWApO1xuICAgICAgfVxuICAgIH1cbiAgfVxuXG4gIGlmIChwcm9wcy50ZXJtaW5hdGlvbikge1xuICAgIGZvciAodmFyIHRlcm1pbmF0aW9uQ2lkciBvZiBwcm9wcy50ZXJtaW5hdGlvbi50ZXJtaW5hdGlvbkNpZHJzKSB7XG4gICAgICBpZiAodGVybWluYXRpb25DaWRyLmluY2x1ZGVzKCdUb2tlbicpKSB7XG4gICAgICB9IGVsc2UgaWYgKCFWQUxJRF9DSURSLnRlc3QodGVybWluYXRpb25DaWRyKSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICAgJ1Rlcm1pbmF0aW9uIENJRFIgbXVzdCBiZSBhIHZhbGlkIG5vbi1SRkMxOTE4IElQdjQgQ0lEUiBibG9jayAoLzI3LS8zMiknLFxuICAgICAgICApO1xuICAgICAgfVxuICAgIH1cbiAgfVxuXG4gIGlmIChwcm9wcy50ZXJtaW5hdGlvbikge1xuICAgIGZvciAodmFyIHRlcm1pbmF0aW9uQ2lkciBvZiBwcm9wcy50ZXJtaW5hdGlvbi50ZXJtaW5hdGlvbkNpZHJzKSB7XG4gICAgICBpZiAoUkZDXzE5MThfQ0lEUi50ZXN0KHRlcm1pbmF0aW9uQ2lkcikpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgICdUZXJtaW5hdGlvbiBDRElSIG11c3Qgbm90IGJlIFJGQzE5MTggQ0lEUiBibG9jayAoLzI3LS8zMiknLFxuICAgICAgICApO1xuICAgICAgfVxuICAgIH1cbiAgfVxuXG4gIGlmIChwcm9wcy5vcmlnaW5hdGlvbikge1xuICAgIGZvciAodmFyIHJvdXRlIG9mIHByb3BzLm9yaWdpbmF0aW9uKSB7XG4gICAgICBpZiAocm91dGUuaG9zdC5pbmNsdWRlcygnVG9rZW4nKSkge1xuICAgICAgfSBlbHNlIGlmIChJUF9BRERSRVNTLnRlc3Qocm91dGUuaG9zdCkpIHtcbiAgICAgICAgaWYgKFJGQ18xOTE4LnRlc3Qocm91dGUuaG9zdCkpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ09yaWdpbmF0aW9uIElQIG11c3Qgbm90IGJlIFJGQzE5MTggSVAgQWRkcmVzcycpO1xuICAgICAgICB9XG4gICAgICB9IGVsc2UgaWYgKCFGUUROLnRlc3Qocm91dGUuaG9zdCkpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgICdPcmlnaW5hdGlvbiBJUCBtdXN0IGJlIHZhbGlkLCBub24tUkZDMTkxOCBJUHY0IEFkZHJlc3Mgb3IgRlFETicsXG4gICAgICAgICk7XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgaWYgKHByb3BzLm9yaWdpbmF0aW9uKSB7XG4gICAgZm9yICh2YXIgcm91dGUgb2YgcHJvcHMub3JpZ2luYXRpb24pIHtcbiAgICAgIGlmIChyb3V0ZS5wb3J0IDwgMSB8fCByb3V0ZS5wb3J0ID4gNjU1MzUpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdQb3J0IHJhbmdlIG11c3QgYmUgdmFsaWQ6IDEtNjU1MzUnKTtcbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICBpZiAocHJvcHMub3JpZ2luYXRpb24gJiYgcHJvcHMuZW5jcnlwdGlvbikge1xuICAgIGZvciAodmFyIHJvdXRlIG9mIHByb3BzLm9yaWdpbmF0aW9uKSB7XG4gICAgICBpZiAocm91dGUucHJvdG9jb2wgPT09IFByb3RvY29sLlVEUCkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1RDUCBtdXN0IGJlIHVzZWQgd2l0aCBFbmNyeXB0aW9uIGVuYWJsZWQnKTtcbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICByZXR1cm4gdHJ1ZTtcbn1cbiJdfQ==