/**
 * The set of standard attributes that can be marked as required or mutable.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html#cognito-user-pools-standard-attributes
 * @stability stable
 */
export interface StandardAttributes {
    /**
     * The user's postal address.
     *
     * @default - see the defaults under `StandardAttribute`
     * @stability stable
     */
    readonly address?: StandardAttribute;
    /**
     * The user's birthday, represented as an ISO 8601:2004 format.
     *
     * @default - see the defaults under `StandardAttribute`
     * @stability stable
     */
    readonly birthdate?: StandardAttribute;
    /**
     * The user's e-mail address, represented as an RFC 5322 [RFC5322] addr-spec.
     *
     * @default - see the defaults under `StandardAttribute`
     * @stability stable
     */
    readonly email?: StandardAttribute;
    /**
     * The surname or last name of the user.
     *
     * @default - see the defaults under `StandardAttribute`
     * @stability stable
     */
    readonly familyName?: StandardAttribute;
    /**
     * The user's gender.
     *
     * @default - see the defaults under `StandardAttribute`
     * @stability stable
     */
    readonly gender?: StandardAttribute;
    /**
     * The user's first name or give name.
     *
     * @default - see the defaults under `StandardAttribute`
     * @stability stable
     */
    readonly givenName?: StandardAttribute;
    /**
     * The user's locale, represented as a BCP47 [RFC5646] language tag.
     *
     * @default - see the defaults under `StandardAttribute`
     * @stability stable
     */
    readonly locale?: StandardAttribute;
    /**
     * The user's middle name.
     *
     * @default - see the defaults under `StandardAttribute`
     * @stability stable
     */
    readonly middleName?: StandardAttribute;
    /**
     * The user's full name in displayable form, including all name parts, titles and suffixes.
     *
     * @default - see the defaults under `StandardAttribute`
     * @stability stable
     */
    readonly fullname?: StandardAttribute;
    /**
     * The user's nickname or casual name.
     *
     * @default - see the defaults under `StandardAttribute`
     * @stability stable
     */
    readonly nickname?: StandardAttribute;
    /**
     * The user's telephone number.
     *
     * @default - see the defaults under `StandardAttribute`
     * @stability stable
     */
    readonly phoneNumber?: StandardAttribute;
    /**
     * The URL to the user's profile picture.
     *
     * @default - see the defaults under `StandardAttribute`
     * @stability stable
     */
    readonly profilePicture?: StandardAttribute;
    /**
     * The user's preffered username, different from the immutable user name.
     *
     * @default - see the defaults under `StandardAttribute`
     * @stability stable
     */
    readonly preferredUsername?: StandardAttribute;
    /**
     * The URL to the user's profile page.
     *
     * @default - see the defaults under `StandardAttribute`
     * @stability stable
     */
    readonly profilePage?: StandardAttribute;
    /**
     * The user's time zone.
     *
     * @default - see the defaults under `StandardAttribute`
     * @stability stable
     */
    readonly timezone?: StandardAttribute;
    /**
     * The time, the user's information was last updated.
     *
     * @default - see the defaults under `StandardAttribute`
     * @stability stable
     */
    readonly lastUpdateTime?: StandardAttribute;
    /**
     * The URL to the user's web page or blog.
     *
     * @default - see the defaults under `StandardAttribute`
     * @stability stable
     */
    readonly website?: StandardAttribute;
    /**
     * (deprecated) DEPRECATED.
     *
     * @default - see the defaults under `StandardAttribute`
     * @deprecated this is not a standard attribute and was incorrectly added to the CDK.
     * It is a Cognito built-in attribute and cannot be controlled as part of user pool creation.
     */
    readonly emailVerified?: StandardAttribute;
    /**
     * (deprecated) DEPRECATED.
     *
     * @default - see the defaults under `StandardAttribute`
     * @deprecated this is not a standard attribute and was incorrectly added to the CDK.
     * It is a Cognito built-in attribute and cannot be controlled as part of user pool creation.
     */
    readonly phoneNumberVerified?: StandardAttribute;
}
/**
 * Standard attribute that can be marked as required or mutable.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html#cognito-user-pools-standard-attributes
 * @stability stable
 */
export interface StandardAttribute {
    /**
     * Specifies whether the value of the attribute can be changed.
     *
     * For any user pool attribute that's mapped to an identity provider attribute, this must be set to `true`.
     * Amazon Cognito updates mapped attributes when users sign in to your application through an identity provider.
     * If an attribute is immutable, Amazon Cognito throws an error when it attempts to update the attribute.
     *
     * @default true
     * @stability stable
     */
    readonly mutable?: boolean;
    /**
     * Specifies whether the attribute is required upon user registration.
     *
     * If the attribute is required and the user does not provide a value, registration or sign-in will fail.
     *
     * @default false
     * @stability stable
     */
    readonly required?: boolean;
}
/**
 * Represents a custom attribute type.
 *
 * @stability stable
 */
export interface ICustomAttribute {
    /**
     * Bind this custom attribute type to the values as expected by CloudFormation.
     *
     * @stability stable
     */
    bind(): CustomAttributeConfig;
}
/**
 * Configuration that will be fed into CloudFormation for any custom attribute type.
 *
 * @stability stable
 */
export interface CustomAttributeConfig {
    /**
     * The data type of the custom attribute.
     *
     * @see https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_SchemaAttributeType.html#CognitoUserPools-Type-SchemaAttributeType-AttributeDataType
     * @stability stable
     */
    readonly dataType: string;
    /**
     * The constraints for a custom attribute of 'String' data type.
     *
     * @default - None.
     * @stability stable
     */
    readonly stringConstraints?: StringAttributeConstraints;
    /**
     * The constraints for a custom attribute of the 'Number' data type.
     *
     * @default - None.
     * @stability stable
     */
    readonly numberConstraints?: NumberAttributeConstraints;
    /**
     * Specifies whether the value of the attribute can be changed.
     *
     * For any user pool attribute that's mapped to an identity provider attribute, you must set this parameter to true.
     * Amazon Cognito updates mapped attributes when users sign in to your application through an identity provider.
     * If an attribute is immutable, Amazon Cognito throws an error when it attempts to update the attribute.
     *
     * @default false
     * @stability stable
     */
    readonly mutable?: boolean;
}
/**
 * Constraints that can be applied to a custom attribute of any type.
 *
 * @stability stable
 */
export interface CustomAttributeProps {
    /**
     * Specifies whether the value of the attribute can be changed.
     *
     * For any user pool attribute that's mapped to an identity provider attribute, you must set this parameter to true.
     * Amazon Cognito updates mapped attributes when users sign in to your application through an identity provider.
     * If an attribute is immutable, Amazon Cognito throws an error when it attempts to update the attribute.
     *
     * @default false
     * @stability stable
     */
    readonly mutable?: boolean;
}
/**
 * Constraints that can be applied to a custom attribute of string type.
 *
 * @stability stable
 */
export interface StringAttributeConstraints {
    /**
     * Minimum length of this attribute.
     *
     * @default 0
     * @stability stable
     */
    readonly minLen?: number;
    /**
     * Maximum length of this attribute.
     *
     * @default 2048
     * @stability stable
     */
    readonly maxLen?: number;
}
/**
 * Props for constructing a StringAttr.
 *
 * @stability stable
 */
export interface StringAttributeProps extends StringAttributeConstraints, CustomAttributeProps {
}
/**
 * The String custom attribute type.
 *
 * @stability stable
 */
export declare class StringAttribute implements ICustomAttribute {
    private readonly minLen?;
    private readonly maxLen?;
    private readonly mutable?;
    /**
     * @stability stable
     */
    constructor(props?: StringAttributeProps);
    /**
     * Bind this custom attribute type to the values as expected by CloudFormation.
     *
     * @stability stable
     */
    bind(): CustomAttributeConfig;
}
/**
 * Constraints that can be applied to a custom attribute of number type.
 *
 * @stability stable
 */
export interface NumberAttributeConstraints {
    /**
     * Minimum value of this attribute.
     *
     * @default - no minimum value
     * @stability stable
     */
    readonly min?: number;
    /**
     * Maximum value of this attribute.
     *
     * @default - no maximum value
     * @stability stable
     */
    readonly max?: number;
}
/**
 * Props for NumberAttr.
 *
 * @stability stable
 */
export interface NumberAttributeProps extends NumberAttributeConstraints, CustomAttributeProps {
}
/**
 * The Number custom attribute type.
 *
 * @stability stable
 */
export declare class NumberAttribute implements ICustomAttribute {
    private readonly min?;
    private readonly max?;
    private readonly mutable?;
    /**
     * @stability stable
     */
    constructor(props?: NumberAttributeProps);
    /**
     * Bind this custom attribute type to the values as expected by CloudFormation.
     *
     * @stability stable
     */
    bind(): CustomAttributeConfig;
}
/**
 * The Boolean custom attribute type.
 *
 * @stability stable
 */
export declare class BooleanAttribute implements ICustomAttribute {
    private readonly mutable?;
    /**
     * @stability stable
     */
    constructor(props?: CustomAttributeProps);
    /**
     * Bind this custom attribute type to the values as expected by CloudFormation.
     *
     * @stability stable
     */
    bind(): CustomAttributeConfig;
}
/**
 * The DateTime custom attribute type.
 *
 * @stability stable
 */
export declare class DateTimeAttribute implements ICustomAttribute {
    private readonly mutable?;
    /**
     * @stability stable
     */
    constructor(props?: CustomAttributeProps);
    /**
     * Bind this custom attribute type to the values as expected by CloudFormation.
     *
     * @stability stable
     */
    bind(): CustomAttributeConfig;
}
/**
 * This interface contains standard attributes recognized by Cognito from https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html including built-in attributes `email_verified` and `phone_number_verified`.
 *
 * @stability stable
 */
export interface StandardAttributesMask {
    /**
     * The user's postal address.
     *
     * @default false
     * @stability stable
     */
    readonly address?: boolean;
    /**
     * The user's birthday, represented as an ISO 8601:2004 format.
     *
     * @default false
     * @stability stable
     */
    readonly birthdate?: boolean;
    /**
     * The user's e-mail address, represented as an RFC 5322 [RFC5322] addr-spec.
     *
     * @default false
     * @stability stable
     */
    readonly email?: boolean;
    /**
     * The surname or last name of the user.
     *
     * @default false
     * @stability stable
     */
    readonly familyName?: boolean;
    /**
     * The user's gender.
     *
     * @default false
     * @stability stable
     */
    readonly gender?: boolean;
    /**
     * The user's first name or give name.
     *
     * @default false
     * @stability stable
     */
    readonly givenName?: boolean;
    /**
     * The user's locale, represented as a BCP47 [RFC5646] language tag.
     *
     * @default false
     * @stability stable
     */
    readonly locale?: boolean;
    /**
     * The user's middle name.
     *
     * @default false
     * @stability stable
     */
    readonly middleName?: boolean;
    /**
     * The user's full name in displayable form, including all name parts, titles and suffixes.
     *
     * @default false
     * @stability stable
     */
    readonly fullname?: boolean;
    /**
     * The user's nickname or casual name.
     *
     * @default false
     * @stability stable
     */
    readonly nickname?: boolean;
    /**
     * The user's telephone number.
     *
     * @default false
     * @stability stable
     */
    readonly phoneNumber?: boolean;
    /**
     * The URL to the user's profile picture.
     *
     * @default false
     * @stability stable
     */
    readonly profilePicture?: boolean;
    /**
     * The user's preffered username, different from the immutable user name.
     *
     * @default false
     * @stability stable
     */
    readonly preferredUsername?: boolean;
    /**
     * The URL to the user's profile page.
     *
     * @default false
     * @stability stable
     */
    readonly profilePage?: boolean;
    /**
     * The user's time zone.
     *
     * @default false
     * @stability stable
     */
    readonly timezone?: boolean;
    /**
     * The time, the user's information was last updated.
     *
     * @default false
     * @stability stable
     */
    readonly lastUpdateTime?: boolean;
    /**
     * The URL to the user's web page or blog.
     *
     * @default false
     * @stability stable
     */
    readonly website?: boolean;
    /**
     * Whether the email address has been verified.
     *
     * @default false
     * @stability stable
     */
    readonly emailVerified?: boolean;
    /**
     * Whether the phone number has been verified.
     *
     * @default false
     * @stability stable
     */
    readonly phoneNumberVerified?: boolean;
}
/**
 * A set of attributes, useful to set Read and Write attributes.
 *
 * @stability stable
 */
export declare class ClientAttributes {
    /**
     * The set of attributes
     */
    private attributesSet;
    /**
     * Creates a ClientAttributes with the specified attributes.
     *
     * @default - a ClientAttributes object without any attributes
     * @stability stable
     */
    constructor();
    /**
     * Creates a custom ClientAttributes with the specified attributes.
     *
     * @param attributes a list of standard attributes to add to the set.
     * @stability stable
     */
    withStandardAttributes(attributes: StandardAttributesMask): ClientAttributes;
    /**
     * Creates a custom ClientAttributes with the specified attributes.
     *
     * @param attributes a list of custom attributes to add to the set.
     * @stability stable
     */
    withCustomAttributes(...attributes: string[]): ClientAttributes;
    /**
     * The list of attributes represented by this ClientAttributes.
     *
     * @stability stable
     */
    attributes(): string[];
}
