"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MetricAggregationType = exports.AdjustmentType = exports.StepScalingAction = void 0;
const cdk = require("@aws-cdk/core");
const applicationautoscaling_generated_1 = require("./applicationautoscaling.generated");
/**
 * Define a step scaling action
 *
 * This kind of scaling policy adjusts the target capacity in configurable
 * steps. The size of the step is configurable based on the metric's distance
 * to its alarm threshold.
 *
 * This Action must be used as the target of a CloudWatch alarm to take effect.
 */
class StepScalingAction extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.adjustments = new Array();
        // Cloudformation requires either the ResourceId, ScalableDimension, and ServiceNamespace
        // properties, or the ScalingTargetId property, but not both.
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-applicationautoscaling-scalingpolicy.html
        const resource = new applicationautoscaling_generated_1.CfnScalingPolicy(this, 'Resource', {
            policyName: props.policyName || this.node.uniqueId,
            policyType: 'StepScaling',
            scalingTargetId: props.scalingTarget.scalableTargetId,
            stepScalingPolicyConfiguration: {
                adjustmentType: props.adjustmentType,
                cooldown: props.cooldown && props.cooldown.toSeconds(),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                metricAggregationType: props.metricAggregationType,
                stepAdjustments: cdk.Lazy.anyValue({ produce: () => this.adjustments }),
            },
        });
        this.scalingPolicyArn = resource.ref;
    }
    /**
     * Add an adjusment interval to the ScalingAction
     */
    addAdjustment(adjustment) {
        if (adjustment.lowerBound === undefined && adjustment.upperBound === undefined) {
            throw new Error('At least one of lowerBound or upperBound is required');
        }
        this.adjustments.push({
            metricIntervalLowerBound: adjustment.lowerBound,
            metricIntervalUpperBound: adjustment.upperBound,
            scalingAdjustment: adjustment.adjustment,
        });
    }
}
exports.StepScalingAction = StepScalingAction;
/**
 * How adjustment numbers are interpreted
 */
var AdjustmentType;
(function (AdjustmentType) {
    /**
     * Add the adjustment number to the current capacity.
     *
     * A positive number increases capacity, a negative number decreases capacity.
     */
    AdjustmentType["CHANGE_IN_CAPACITY"] = "ChangeInCapacity";
    /**
     * Add this percentage of the current capacity to itself.
     *
     * The number must be between -100 and 100; a positive number increases
     * capacity and a negative number decreases it.
     */
    AdjustmentType["PERCENT_CHANGE_IN_CAPACITY"] = "PercentChangeInCapacity";
    /**
     * Make the capacity equal to the exact number given.
     */
    AdjustmentType["EXACT_CAPACITY"] = "ExactCapacity";
})(AdjustmentType = exports.AdjustmentType || (exports.AdjustmentType = {}));
/**
 * How the scaling metric is going to be aggregated
 */
var MetricAggregationType;
(function (MetricAggregationType) {
    /**
     * Average
     */
    MetricAggregationType["AVERAGE"] = "Average";
    /**
     * Minimum
     */
    MetricAggregationType["MINIMUM"] = "Minimum";
    /**
     * Maximum
     */
    MetricAggregationType["MAXIMUM"] = "Maximum";
})(MetricAggregationType = exports.MetricAggregationType || (exports.MetricAggregationType = {}));
//# sourceMappingURL=data:application/json;base64,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