"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PredefinedMetric = exports.TargetTrackingScalingPolicy = void 0;
const cdk = require("@aws-cdk/core");
const applicationautoscaling_generated_1 = require("./applicationautoscaling.generated");
class TargetTrackingScalingPolicy extends cdk.Construct {
    constructor(scope, id, props) {
        if ((props.customMetric === undefined) === (props.predefinedMetric === undefined)) {
            throw new Error('Exactly one of \'customMetric\' or \'predefinedMetric\' must be specified.');
        }
        if (props.customMetric && !props.customMetric.toMetricConfig().metricStat) {
            throw new Error('Only direct metrics are supported for Target Tracking. Use Step Scaling or supply a Metric object.');
        }
        super(scope, id);
        const resource = new applicationautoscaling_generated_1.CfnScalingPolicy(this, 'Resource', {
            policyName: props.policyName || this.node.uniqueId,
            policyType: 'TargetTrackingScaling',
            scalingTargetId: props.scalingTarget.scalableTargetId,
            targetTrackingScalingPolicyConfiguration: {
                customizedMetricSpecification: renderCustomMetric(props.customMetric),
                disableScaleIn: props.disableScaleIn,
                predefinedMetricSpecification: props.predefinedMetric !== undefined ? {
                    predefinedMetricType: props.predefinedMetric,
                    resourceLabel: props.resourceLabel,
                } : undefined,
                scaleInCooldown: props.scaleInCooldown && props.scaleInCooldown.toSeconds(),
                scaleOutCooldown: props.scaleOutCooldown && props.scaleOutCooldown.toSeconds(),
                targetValue: props.targetValue,
            },
        });
        this.scalingPolicyArn = resource.ref;
    }
}
exports.TargetTrackingScalingPolicy = TargetTrackingScalingPolicy;
function renderCustomMetric(metric) {
    if (!metric) {
        return undefined;
    }
    const c = metric.toMetricConfig().metricStat;
    if (c.statistic.startsWith('p')) {
        throw new Error(`Cannot use statistic '${c.statistic}' for Target Tracking: only 'Average', 'Minimum', 'Maximum', 'SampleCount', and 'Sum' are supported.`);
    }
    return {
        dimensions: c.dimensions,
        metricName: c.metricName,
        namespace: c.namespace,
        statistic: c.statistic,
        unit: c.unitFilter,
    };
}
/**
 * One of the predefined autoscaling metrics
 */
var PredefinedMetric;
(function (PredefinedMetric) {
    /**
     * DYNAMODB_READ_CAPACITY_UTILIZATIO
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["DYNAMODB_READ_CAPACITY_UTILIZATION"] = "DynamoDBReadCapacityUtilization";
    /**
     * DYANMODB_WRITE_CAPACITY_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["DYANMODB_WRITE_CAPACITY_UTILIZATION"] = "DynamoDBWriteCapacityUtilization";
    /**
     * ALB_REQUEST_COUNT_PER_TARGET
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["ALB_REQUEST_COUNT_PER_TARGET"] = "ALBRequestCountPerTarget";
    /**
     * RDS_READER_AVERAGE_CPU_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["RDS_READER_AVERAGE_CPU_UTILIZATION"] = "RDSReaderAverageCPUUtilization";
    /**
     * RDS_READER_AVERAGE_DATABASE_CONNECTIONS
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["RDS_READER_AVERAGE_DATABASE_CONNECTIONS"] = "RDSReaderAverageDatabaseConnections";
    /**
     * EC2_SPOT_FLEET_REQUEST_AVERAGE_CPU_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["EC2_SPOT_FLEET_REQUEST_AVERAGE_CPU_UTILIZATION"] = "EC2SpotFleetRequestAverageCPUUtilization";
    /**
     * EC2_SPOT_FLEET_REQUEST_AVERAGE_NETWORK_IN
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["EC2_SPOT_FLEET_REQUEST_AVERAGE_NETWORK_IN"] = "EC2SpotFleetRequestAverageNetworkIn";
    /**
     * EC2_SPOT_FLEET_REQUEST_AVERAGE_NETWORK_OUT
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["EC2_SPOT_FLEET_REQUEST_AVERAGE_NETWORK_OUT"] = "EC2SpotFleetRequestAverageNetworkOut";
    /**
     * SAGEMAKER_VARIANT_INVOCATIONS_PER_INSTANCE
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["SAGEMAKER_VARIANT_INVOCATIONS_PER_INSTANCE"] = "SageMakerVariantInvocationsPerInstance";
    /**
     * ECS_SERVICE_AVERAGE_CPU_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["ECS_SERVICE_AVERAGE_CPU_UTILIZATION"] = "ECSServiceAverageCPUUtilization";
    /**
     * ECS_SERVICE_AVERAGE_MEMORY_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["ECS_SERVICE_AVERAGE_MEMORY_UTILIZATION"] = "ECSServiceAverageMemoryUtilization";
    /**
     * LAMBDA_PROVISIONED_CONCURRENCY_UTILIZATION
     * @see https://docs.aws.amazon.com/lambda/latest/dg/monitoring-metrics.html#monitoring-metrics-concurrency
     */
    PredefinedMetric["LAMBDA_PROVISIONED_CONCURRENCY_UTILIZATION"] = "LambdaProvisionedConcurrencyUtilization";
})(PredefinedMetric = exports.PredefinedMetric || (exports.PredefinedMetric = {}));
//# sourceMappingURL=data:application/json;base64,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