"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnReference = exports.ReferenceRendering = void 0;
const reference_1 = require("../reference");
const CFN_REFERENCE_SYMBOL = Symbol.for('@aws-cdk/core.CfnReference');
/**
 * An enum that allows controlling how will the created reference
 * be rendered in the resulting CloudFormation template.
 */
var ReferenceRendering;
(function (ReferenceRendering) {
    /**
     * Used for rendering a reference inside Fn::Sub expressions,
     * which mean these must resolve to "${Sth}" instead of { Ref: "Sth" }.
     */
    ReferenceRendering[ReferenceRendering["FN_SUB"] = 0] = "FN_SUB";
    /**
     * Used for rendering Fn::GetAtt with its arguments in string form
     * (as opposed to the more common arguments in array form, which we render by default).
     */
    ReferenceRendering[ReferenceRendering["GET_ATT_STRING"] = 1] = "GET_ATT_STRING";
})(ReferenceRendering = exports.ReferenceRendering || (exports.ReferenceRendering = {}));
/**
 * A Token that represents a CloudFormation reference to another resource
 *
 * If these references are used in a different stack from where they are
 * defined, appropriate CloudFormation `Export`s and `Fn::ImportValue`s will be
 * synthesized automatically instead of the regular CloudFormation references.
 *
 * Additionally, the dependency between the stacks will be recorded, and the toolkit
 * will make sure to deploy producing stack before the consuming stack.
 *
 * This magic happens in the prepare() phase, where consuming stacks will call
 * `consumeFromStack` on these Tokens and if they happen to be exported by a different
 * Stack, we'll register the dependency.
 */
class CfnReference extends reference_1.Reference {
    constructor(value, displayName, target) {
        // prepend scope path to display name
        super(value, target, displayName);
        this.replacementTokens = new Map();
        this.targetStack = stack_1.Stack.of(target);
        Object.defineProperty(this, CFN_REFERENCE_SYMBOL, { value: true });
    }
    /**
     * Check whether this is actually a Reference
     */
    static isCfnReference(x) {
        return CFN_REFERENCE_SYMBOL in x;
    }
    /**
     * Return the CfnReference for the indicated target
     *
     * Will make sure that multiple invocations for the same target and intrinsic
     * return the same CfnReference. Because CfnReferences accumulate state in
     * the prepare() phase (for the purpose of cross-stack references), it's
     * important that the state isn't lost if it's lazily created, like so:
     *
     *     Lazy.string({ produce: () => new CfnReference(...) })
     *
     */
    static for(target, attribute, refRender) {
        return CfnReference.singletonReference(target, attribute, refRender, () => {
            const cfnIntrinsic = refRender === ReferenceRendering.FN_SUB
                ? ('${' + target.logicalId + (attribute === 'Ref' ? '' : `.${attribute}`) + '}')
                : (attribute === 'Ref'
                    ? { Ref: target.logicalId }
                    : {
                        'Fn::GetAtt': refRender === ReferenceRendering.GET_ATT_STRING
                            ? `${target.logicalId}.${attribute}`
                            : [target.logicalId, attribute],
                    });
            return new CfnReference(cfnIntrinsic, attribute, target);
        });
    }
    /**
     * Return a CfnReference that references a pseudo referencd
     */
    static forPseudo(pseudoName, scope) {
        return CfnReference.singletonReference(scope, `Pseudo:${pseudoName}`, undefined, () => {
            const cfnIntrinsic = { Ref: pseudoName };
            return new CfnReference(cfnIntrinsic, pseudoName, scope);
        });
    }
    /**
     * Get or create the table.
     * Passing fnSub = true allows cloudformation-include to correctly handle Fn::Sub.
     */
    static singletonReference(target, attribKey, refRender, fresh) {
        let attribs = CfnReference.referenceTable.get(target);
        if (!attribs) {
            attribs = new Map();
            CfnReference.referenceTable.set(target, attribs);
        }
        let cacheKey = attribKey;
        switch (refRender) {
            case ReferenceRendering.FN_SUB:
                cacheKey += 'Fn::Sub';
                break;
            case ReferenceRendering.GET_ATT_STRING:
                cacheKey += 'Fn::GetAtt::String';
                break;
        }
        let ref = attribs.get(cacheKey);
        if (!ref) {
            ref = fresh();
            attribs.set(cacheKey, ref);
        }
        return ref;
    }
    resolve(context) {
        // If we have a special token for this consuming stack, resolve that. Otherwise resolve as if
        // we are in the same stack.
        const consumingStack = stack_1.Stack.of(context.scope);
        const token = this.replacementTokens.get(consumingStack);
        // if (!token && this.isCrossStackReference(consumingStack) && !context.preparing) {
        // eslint-disable-next-line max-len
        //   throw new Error(`Cross-stack reference (${context.scope.node.path} -> ${this.target.node.path}) has not been assigned a value--call prepare() first`);
        // }
        if (token) {
            return token.resolve(context);
        }
        else {
            return super.resolve(context);
        }
    }
    hasValueForStack(stack) {
        if (stack === this.targetStack) {
            return true;
        }
        return this.replacementTokens.has(stack);
    }
    assignValueForStack(stack, value) {
        if (stack === this.targetStack) {
            throw new Error('cannot assign a value for the same stack');
        }
        if (this.hasValueForStack(stack)) {
            throw new Error('Cannot assign a reference value twice to the same stack. Use hasValueForStack to check first');
        }
        this.replacementTokens.set(stack, value);
    }
    /**
     * Implementation of toString() that will use the display name
     */
    toString() {
        return token_1.Token.asString(this, {
            displayHint: `${this.target.node.id}.${this.displayName}`,
        });
    }
}
exports.CfnReference = CfnReference;
/**
 * Static table where we keep singleton CfnReference instances
 */
CfnReference.referenceTable = new Map();
const stack_1 = require("../stack");
const token_1 = require("../token");
//# sourceMappingURL=data:application/json;base64,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