# This code is part of Qiskit.
#
# (C) Copyright IBM 2021, 2022.
#
# This code is licensed under the Apache License, Version 2.0. You may
# obtain a copy of this license in the LICENSE.txt file in the root directory
# of this source tree or at http://www.apache.org/licenses/LICENSE-2.0.
#
# Any modifications or derivative works of this code must retain this
# copyright notice, and modified files need to carry a notice indicating
# that they have been altered from the originals.

"""
Fermionic Hamilton for LiH and Qubit Hamiltonian obtained from BKSF

The Fermionic Hamiltonian is generated by the following code:

    from qiskit_nature.second_q.formats.molecule_info import MoleculeInfo

    molecule = MoleculeInfo(
        # coordinates are given in Angstrom
        symbols=["Li", "H"],
        geometry=np.asarray([
            [0.0, 0.0, 0.0],
            [0.0, 0.0, 1.6],
        ]),
        multiplicity=1,  # = 2*spin + 1
        charge=0,
    )

    from qiskit_nature.second_q.drivers import PySCFDriver

    driver = PySCFDriver.from_molecule(molecule, basis="sto3g")

    from qiskit_nature.second_q.problems import ElectronicStructureProblem
    from qiskit_nature.second_q.transformers.second_q.electronic import ActiveSpaceTransformer

    transformer = ActiveSpaceTransformer(
        num_electrons=2,
        num_molecular_orbitals=3,
    )

    problem_reduced = transformer.transform(driver.run())
    second_q_ops_reduced = problem_reduced.second_q_ops()
    hamiltonian_reduced = second_q_ops_reduced[0]
"""

import numpy
from qiskit.quantum_info import PauliList, SparsePauliOp
from qiskit_nature.second_q.operators import FermionicOp


FERMIONIC_HAMILTONIAN = FermionicOp(
    {
        "+_0 -_1 +_3 -_4": 0.013063981998607477,
        "+_0 -_1 -_3 +_4": -0.013063981998607472,
        "+_0 -_1": 0.048579599520367646,
        "+_0 -_1 +_5 -_5": 0.005767502046076787,
        "+_0 -_1 +_4 -_4": 0.007484171005646165,
        "+_0 -_1 +_3 -_3": -0.04857958891220289,
        "+_0 -_1 +_2 -_2": -0.013509390402447545,
        "+_0 -_2 +_3 -_5": 0.023422673239767655,
        "+_0 -_2 -_3 +_5": -0.023422673239767655,
        "+_0 -_2 +_4 -_5": 0.019276892448524333,
        "+_0 -_2 -_4 +_5": -0.019276892448524333,
        "-_0 +_1 +_3 -_4": -0.013063981998607472,
        "-_0 +_1 -_3 +_4": 0.013063981998607473,
        "-_0 +_1": -0.04857959952036771,
        "-_0 +_1 +_5 -_5": -0.005767502046076761,
        "-_0 +_1 +_4 -_4": -0.007484171005646201,
        "-_0 +_1 +_3 -_3": 0.04857958891220291,
        "-_0 +_1 +_2 -_2": 0.013509390402447573,
        "-_0 +_2 +_3 -_5": -0.023422673239767655,
        "-_0 +_2 -_3 +_5": 0.023422673239767655,
        "-_0 +_2 +_4 -_5": -0.019276892448524333,
        "-_0 +_2 -_4 +_5": 0.019276892448524333,
        "+_1 -_2 +_3 -_5": 0.019276892448524333,
        "+_1 -_2 -_3 +_5": -0.019276892448524333,
        "+_1 -_2 +_4 -_5": 0.041276695997097185,
        "+_1 -_2 -_4 +_5": -0.04127669599709719,
        "-_1 +_2 +_3 -_5": -0.019276892448524333,
        "-_1 +_2 -_3 +_5": 0.019276892448524333,
        "-_1 +_2 +_4 -_5": -0.04127669599709719,
        "-_1 +_2 -_4 +_5": 0.041276695997097185,
        "+_3 -_4": 0.048579599520367646,
        "+_3 -_4 +_5 -_5": -0.013509390402447545,
        "-_3 +_4": -0.04857959952036771,
        "-_3 +_4 +_5 -_5": 0.013509390402447573,
        "+_5 -_5": -0.35297896520254896,
        "+_4 -_4": -0.355939542660255,
        "+_4 -_4 +_5 -_5": 0.2407146489655783,
        "+_3 -_3": -0.772581720072654,
        "+_3 -_3 +_5 -_5": 0.24674881903629914,
        "+_3 -_3 +_4 -_4": 0.2105460611420031,
        "+_2 -_2 +_3 -_4": 0.005767502046076787,
        "+_2 -_2 -_3 +_4": -0.005767502046076761,
        "+_2 -_2": -0.35297896520254896,
        "+_2 -_2 +_5 -_5": 0.3129455111594082,
        "+_2 -_2 +_4 -_4": 0.28199134496267547,
        "+_2 -_2 +_3 -_3": 0.2701714922760668,
        "+_1 -_1 +_3 -_4": 0.007484171005646165,
        "+_1 -_1 -_3 +_4": -0.007484171005646201,
        "+_1 -_1": -0.355939542660255,
        "+_1 -_1 +_5 -_5": 0.28199134496267547,
        "+_1 -_1 +_4 -_4": 0.3378822722917939,
        "+_1 -_1 +_3 -_3": 0.2236100431406106,
        "+_1 -_1 +_2 -_2": 0.2407146489655783,
        "+_0 -_0 +_3 -_4": -0.04857958891220289,
        "+_0 -_0 -_3 +_4": 0.04857958891220291,
        "+_0 -_0": -0.772581720072654,
        "+_0 -_0 +_5 -_5": 0.2701714922760668,
        "+_0 -_0 +_4 -_4": 0.2236100431406106,
        "+_0 -_0 +_3 -_3": 0.48731096863288564,
        "+_0 -_0 +_2 -_2": 0.24674881903629914,
        "+_0 -_0 +_1 -_1": 0.2105460611420031,
    },
    num_spin_orbitals=6,
    copy=False,
)


def _qubit_operator():
    pauli_list = PauliList(
        [
            "IIIIIIIIIII",
            "IIIIIIIZZZY",
            "IIIIIIIZZZZ",
            "IIIIIIXIXII",
            "IIIIIXZIZXI",
            "IIIIZYIZIYI",
            "IIIIZZYZYZI",
            "IIIIZZZIIIY",
            "IIIIZZZIIIZ",
            "IIIIZZZZZZI",
            "IIIXIIZXZIZ",
            "IIIXXZIZIZZ",
            "IIXZIZIXIZZ",
            "IIXZXIZZZIZ",
            "IIYIYIZIZIZ",
            "IIYIZZIYIZZ",
            "IIZYYZIIIZZ",
            "IIZYZIZYZIZ",
            "IIZZIYIIIYI",
            "IIZZIZYIYZI",
            "IIZZIZZIZZI",
            "IIZZZIIZIII",
            "IIZZZIXZXII",
            "IIZZZXZZZXI",
            "IXIIIIIIIXZ",
            "IXIIIIIIXZZ",
            "IXIIIIXIIZZ",
            "IXIIIXZIZIZ",
            "IXZZZIIZIXZ",
            "IXZZZIIZXZZ",
            "IXZZZIXZIZZ",
            "IXZZZXZZZIZ",
            "IYIIIYIZZII",
            "IYIIIZYZIZI",
            "IYIIZIIIYII",
            "IYIIZIIIZYI",
            "IYZZIIIZYII",
            "IYZZIIIZZYI",
            "IYZZZYIIZII",
            "IYZZZZYIIZI",
            "IZIIIZZZIIY",
            "IZIIIZZZIIZ",
            "IZIIZIIIZZY",
            "IZIIZIIIZZZ",
            "IZIIZIIZIII",
            "IZIXXIZIIZI",
            "IZIXZIZXIZI",
            "IZXZXZIIZII",
            "IZXZZZIXZII",
            "IZYIIZIYZII",
            "IZYIYZIZZII",
            "IZZYIIZYIZI",
            "IZZYYIZZIZI",
            "IZZZIIIIIII",
            "IZZZIIIZZZY",
            "IZZZIIIZZZZ",
            "IZZZZZZIIIY",
            "IZZZZZZIIIZ",
            "YIIZIIZIIZI",
            "YIIZIIZZZIZ",
            "YIIZZZIIIZZ",
            "YIZIIZIIZII",
            "YIZIIZIZIZZ",
            "YIZIZIZIZIZ",
            "YZIZIZIIZII",
            "YZIZZIZZIZI",
            "YZZIIIZIIZI",
            "YZZIZZIZZII",
            "ZIIYIIIYIII",
            "ZIIYYIIZIII",
            "ZIIZIIYZYIZ",
            "ZIIZIIZIIZI",
            "ZIIZIIZZZIY",
            "ZIIZIIZZZIZ",
            "ZIIZIXIZIXZ",
            "ZIIZZYZIZYZ",
            "ZIIZZZIIIZY",
            "ZIIZZZIIIZZ",
            "ZIIZZZXIXZZ",
            "ZIXIXIIIIII",
            "ZIXIZIIXIII",
            "ZIYZIIIYIII",
            "ZIYZYIIZIII",
            "ZIZIIYZZZYZ",
            "ZIZIIZIIZII",
            "ZIZIIZIZIZY",
            "ZIZIIZIZIZZ",
            "ZIZIIZXZXZZ",
            "ZIZIZIYIYIZ",
            "ZIZIZIZIZIY",
            "ZIZIZIZIZIZ",
            "ZIZIZXIIIXZ",
            "ZIZXXIIIIII",
            "ZIZXZIIXIII",
            "ZXIZIIZZZXI",
            "ZXIZIXIZIII",
            "ZXIZZZIIXII",
            "ZXIZZZXIIII",
            "ZXZIIZIZXII",
            "ZXZIIZXZIII",
            "ZXZIZIZIZXI",
            "ZXZIZXIIIII",
            "ZYIZIYZIIIZ",
            "ZYIZIZIIYZZ",
            "ZYIZZIYZIIZ",
            "ZYIZZIZZIYZ",
            "ZYZIIIYIIIZ",
            "ZYZIIIZIIYZ",
            "ZYZIZYZZIIZ",
            "ZYZIZZIZYZZ",
            "ZZIYYZZIIIZ",
            "ZZIYZIIYZZZ",
            "ZZIZIZIIZII",
            "ZZIZZIZZIZI",
            "ZZXIIIIXZZZ",
            "ZZXIXZZZIIZ",
            "ZZYZYZZIIIZ",
            "ZZYZZIIYZZZ",
            "ZZZIIIZIIZI",
            "ZZZIZZIZZII",
            "ZZZXIIIXZZZ",
            "ZZZXXZZZIIZ",
        ]
    )

    coeffs = numpy.array(
        [
            -0.46007434 + 0.0j,
            0.01208047 + 0.0j,
            0.02669401 + 0.0j,
            0.001633 + 0.0j,
            0.001633 + 0.0j,
            0.001633 + 0.0j,
            0.001633 + 0.0j,
            -0.01208047 + 0.0j,
            -0.14571632 + 0.0j,
            0.05263652 + 0.0j,
            0.00292783 + 0.0j,
            0.00240961 + 0.0j,
            0.00240961 + 0.0j,
            0.00515959 + 0.0j,
            0.00515959 + 0.0j,
            0.00240961 + 0.0j,
            0.00240961 + 0.0j,
            0.00292783 + 0.0j,
            0.001633 + 0.0j,
            0.001633 + 0.0j,
            0.05263652 + 0.0j,
            0.07823638 + 0.0j,
            -0.001633 + 0.0j,
            -0.001633 + 0.0j,
            0.00292783 + 0.0j,
            0.00240961 + 0.0j,
            0.00240961 + 0.0j,
            0.00515959 + 0.0j,
            0.00292783 + 0.0j,
            0.00240961 + 0.0j,
            0.00240961 + 0.0j,
            0.00515959 + 0.0j,
            0.00515959 + 0.0j,
            0.00240961 + 0.0j,
            0.00240961 + 0.0j,
            0.00292783 + 0.0j,
            0.00240961 + 0.0j,
            0.00292783 + 0.0j,
            0.00515959 + 0.0j,
            0.00240961 + 0.0j,
            0.00144188 + 0.0j,
            0.07049784 + 0.0j,
            -0.00144188 + 0.0j,
            0.06754287 + 0.0j,
            -0.16165347 + 0.0j,
            0.00240961 + 0.0j,
            0.00292783 + 0.0j,
            0.00515959 + 0.0j,
            0.00240961 + 0.0j,
            0.00240961 + 0.0j,
            0.00515959 + 0.0j,
            0.00292783 + 0.0j,
            0.00240961 + 0.0j,
            -0.16165347 + 0.0j,
            0.00337735 + 0.0j,
            0.0616872 + 0.0j,
            -0.00337735 + 0.0j,
            0.06017866 + 0.0j,
            -0.01208047 + 0.0j,
            -0.0121449 + 0.0j,
            0.00187104 + 0.0j,
            0.01208047 + 0.0j,
            0.0121449 + 0.0j,
            -0.00187104 + 0.0j,
            -0.00144188 + 0.0j,
            -0.00337735 + 0.0j,
            0.00144188 + 0.0j,
            0.00337735 + 0.0j,
            0.00292783 + 0.0j,
            0.00240961 + 0.0j,
            0.001633 + 0.0j,
            0.02669401 + 0.0j,
            0.0121449 + 0.0j,
            0.12182774 + 0.0j,
            0.001633 + 0.0j,
            0.001633 + 0.0j,
            -0.0121449 + 0.0j,
            0.05590251 + 0.0j,
            0.001633 + 0.0j,
            -0.00515959 + 0.0j,
            -0.00240961 + 0.0j,
            0.00240961 + 0.0j,
            0.00515959 + 0.0j,
            0.001633 + 0.0j,
            -0.14571632 + 0.0j,
            -0.00187104 + 0.0j,
            0.05590251 + 0.0j,
            0.001633 + 0.0j,
            0.001633 + 0.0j,
            0.00187104 + 0.0j,
            0.08447057 + 0.0j,
            0.001633 + 0.0j,
            -0.00240961 + 0.0j,
            -0.00292783 + 0.0j,
            -0.00292783 + 0.0j,
            0.00515959 + 0.0j,
            0.00240961 + 0.0j,
            -0.00240961 + 0.0j,
            -0.00240961 + 0.0j,
            0.00240961 + 0.0j,
            0.00292783 + 0.0j,
            -0.00515959 + 0.0j,
            0.00515959 + 0.0j,
            0.00240961 + 0.0j,
            0.00240961 + 0.0j,
            0.00292783 + 0.0j,
            0.00240961 + 0.0j,
            0.00292783 + 0.0j,
            0.00515959 + 0.0j,
            0.00240961 + 0.0j,
            0.00240961 + 0.0j,
            0.00292783 + 0.0j,
            0.07049784 + 0.0j,
            0.0616872 + 0.0j,
            0.00240961 + 0.0j,
            0.00515959 + 0.0j,
            0.00515959 + 0.0j,
            0.00240961 + 0.0j,
            0.06754287 + 0.0j,
            0.06017866 + 0.0j,
            0.00292783 + 0.0j,
            0.00240961 + 0.0j,
        ]
    )

    return SparsePauliOp(pauli_list, coeffs=coeffs)


QUBIT_HAMILTONIAN = _qubit_operator()
