"use strict";
var _a, _b, _c, _d, _e;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CompositeECRRepositoryAspect = exports.BatchJobDefinition = exports.StepFunctionsSageMakerTrainingJob = exports.ECSTaskDefinition = exports.ECRRepositoryAspect = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const batch = require("@aws-cdk/aws-batch-alpha");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_stepfunctions_tasks_1 = require("aws-cdk-lib/aws-stepfunctions-tasks");
const FN_SUB = 'Fn::Sub';
/**
 * Abtract aspect for ECR repository.
 *
 * You must provide the account id in props or set BSS_IMAGE_ASSET_ACCOUNT_ID in env
 */
class ECRRepositoryAspect {
    constructor(props = {}) {
        this.account = props.imageAssetAccountId ?? process.env.BSS_IMAGE_ASSET_ACCOUNT_ID;
    }
    getRepoName(imageUri) {
        const matches = /\d{12}\.dkr\.ecr\..*\/(.*):.*/g.exec(imageUri);
        if (matches) {
            return matches[1];
        }
        return undefined;
    }
    crossAccountECRPolicy(stack, repoName) {
        const id = `${stack.stackName}-${repoName}`;
        const policy = ECRRepositoryAspect._repoPolicies.get(id);
        if (policy) {
            return policy;
        }
        const newPolicy = new aws_iam_1.Policy(stack, `CrossAccountECR-${repoName}`, {
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: [
                        'ecr:BatchCheckLayerAvailability',
                        'ecr:GetDownloadUrlForLayer',
                        'ecr:BatchGetImage',
                    ],
                    resources: [
                        aws_cdk_lib_1.Arn.format({
                            account: this.account,
                            service: 'ecr',
                            resource: 'repository',
                            resourceName: repoName,
                        }, stack),
                    ],
                }),
            ],
        });
        ECRRepositoryAspect._repoPolicies.set(id, newPolicy);
        return newPolicy;
    }
}
exports.ECRRepositoryAspect = ECRRepositoryAspect;
_a = JSII_RTTI_SYMBOL_1;
ECRRepositoryAspect[_a] = { fqn: "cdk-bootstrapless-synthesizer.ECRRepositoryAspect", version: "2.2.11" };
/**
 * @internal
 */
ECRRepositoryAspect._repoPolicies = new Map();
/**
 * Process the image assets in ECS task definition
 */
class ECSTaskDefinition extends ECRRepositoryAspect {
    constructor(props = {}) {
        super(props);
    }
    hasBeReplaced(prop) {
        if (typeof prop.image === 'object' && FN_SUB in prop.image &&
            prop.image[FN_SUB].indexOf(this.account) > -1) {
            return prop.image[FN_SUB];
        }
        else if (prop.image && prop.image && prop.image.indexOf(this.account) > -1) {
            return prop.image;
        }
        return undefined;
    }
    visit(construct) {
        if (construct instanceof aws_ecs_1.TaskDefinition) {
            const containers = construct.stack.resolve(construct.node.defaultChild.containerDefinitions);
            let imageUri = undefined;
            if (containers instanceof Array) {
                for (const container of containers) {
                    if (container) {
                        imageUri = this.hasBeReplaced(container);
                        if (imageUri) {
                            break;
                        }
                    }
                }
            }
            else if (containers) {
                imageUri = this.hasBeReplaced(containers);
            }
            if (imageUri) {
                const repoName = this.getRepoName(imageUri);
                if (repoName) {
                    construct.executionRole.attachInlinePolicy(this.crossAccountECRPolicy(construct.stack, repoName));
                }
            }
        }
    }
}
exports.ECSTaskDefinition = ECSTaskDefinition;
_b = JSII_RTTI_SYMBOL_1;
ECSTaskDefinition[_b] = { fqn: "cdk-bootstrapless-synthesizer.ECSTaskDefinition", version: "2.2.11" };
/**
 * Process the image assets in SageMaker training job in Step Functions
 */
class StepFunctionsSageMakerTrainingJob extends ECRRepositoryAspect {
    constructor(props = {}) {
        super(props);
    }
    visit(construct) {
        if (construct instanceof aws_stepfunctions_tasks_1.SageMakerCreateTrainingJob) {
            const stack = aws_cdk_lib_1.Stack.of(construct);
            const state = construct.toStateJson();
            const image = stack.resolve(state.Parameters.AlgorithmSpecification.TrainingImage);
            if (FN_SUB in image) {
                const repoName = this.getRepoName(image[FN_SUB]);
                if (repoName) {
                    construct.role.attachInlinePolicy(this.crossAccountECRPolicy(stack, repoName));
                }
            }
        }
    }
}
exports.StepFunctionsSageMakerTrainingJob = StepFunctionsSageMakerTrainingJob;
_c = JSII_RTTI_SYMBOL_1;
StepFunctionsSageMakerTrainingJob[_c] = { fqn: "cdk-bootstrapless-synthesizer.StepFunctionsSageMakerTrainingJob", version: "2.2.11" };
/**
 * Process the image assets in AWS Batch job
 */
class BatchJobDefinition extends ECRRepositoryAspect {
    constructor(props = {}) {
        super(props);
        this._repoNames = [];
        this._allRolesMap = new Map();
    }
    visit(construct) {
        if (construct instanceof batch.JobDefinition) {
            const stack = construct.stack;
            this._executionRoleArn = construct.node.defaultChild
                .containerProperties.executionRoleArn;
            if (this._executionRoleArn && this._allRolesMap.get(this._executionRoleArn)) {
                this._executionRole = this._allRolesMap.get(this._executionRoleArn);
            }
            const image = construct.node.defaultChild
                .containerProperties.image;
            const image_resolved = stack.resolve(image);
            if (FN_SUB in image_resolved) {
                const repoName = this.getRepoName(image_resolved[FN_SUB]);
                if (repoName) {
                    if (this._executionRole) {
                        this._executionRole.attachInlinePolicy(this.crossAccountECRPolicy(stack, repoName));
                    }
                    else {
                        if (this._repoNames.indexOf(repoName) < 0) {
                            this._repoNames.push(repoName);
                        }
                    }
                }
            }
        }
        if (construct instanceof aws_iam_1.Role) {
            this._allRolesMap.set(construct.roleArn, construct);
            if (construct.roleArn == this._executionRoleArn) {
                const stack = construct.stack;
                this._executionRole = construct;
                while (this._repoNames.length > 0) {
                    const repoName = this._repoNames.pop();
                    if (repoName) {
                        this._executionRole.attachInlinePolicy(this.crossAccountECRPolicy(stack, repoName));
                    }
                }
            }
        }
    }
}
exports.BatchJobDefinition = BatchJobDefinition;
_d = JSII_RTTI_SYMBOL_1;
BatchJobDefinition[_d] = { fqn: "cdk-bootstrapless-synthesizer.BatchJobDefinition", version: "2.2.11" };
/**
 * Default ECR asset aspect, support using ECR assets in below services,
 *
 * - ECS task definition
 * - SageMaker training job in Step Functions
 * - AWS Batch job
 * - AWS Lambda container image
 */
class CompositeECRRepositoryAspect extends ECRRepositoryAspect {
    constructor(props = {}) {
        super(props);
        this._aspects = [
            new ECSTaskDefinition(props),
            new StepFunctionsSageMakerTrainingJob(props),
            new BatchJobDefinition(props),
        ];
    }
    visit(construct) {
        for (const _aspect of this._aspects) {
            _aspect.visit(construct);
        }
    }
}
exports.CompositeECRRepositoryAspect = CompositeECRRepositoryAspect;
_e = JSII_RTTI_SYMBOL_1;
CompositeECRRepositoryAspect[_e] = { fqn: "cdk-bootstrapless-synthesizer.CompositeECRRepositoryAspect", version: "2.2.11" };
//# sourceMappingURL=data:application/json;base64,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