"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __exportStar = (this && this.__exportStar) || function(m, exports) {
    for (var p in m) if (p !== "default" && !exports.hasOwnProperty(p)) __createBinding(exports, m, p);
};
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BootstraplessStackSynthesizer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cxschema = require("aws-cdk-lib/cloud-assembly-schema");
const cxapi = require("aws-cdk-lib/cx-api");
__exportStar(require("./aspect"), exports);
const REGION_PLACEHOLDER = '${AWS::Region}';
const ERR_MSG_CALL_BIND_FIRST = 'You must call bind() first';
/**
 * A Bootstrapless stack synthesizer that is designated to generate templates
 * that can be directly used by Cloudformation
 */
class BootstraplessStackSynthesizer extends aws_cdk_lib_1.StackSynthesizer {
    constructor(props = {}) {
        super();
        this.files = {};
        this.dockerImages = {};
        const { BSS_FILE_ASSET_BUCKET_NAME, BSS_IMAGE_ASSET_REPOSITORY_NAME, BSS_FILE_ASSET_PUBLISHING_ROLE_ARN, BSS_IMAGE_ASSET_PUBLISHING_ROLE_ARN, BSS_FILE_ASSET_PREFIX, BSS_FILE_ASSET_REGION_SET, BSS_TEMPLATE_BUCKET_NAME, BSS_IMAGE_ASSET_TAG_PREFIX, BSS_IMAGE_ASSET_REGION_SET, BSS_IMAGE_ASSET_ACCOUNT_ID, } = process.env;
        this.bucketName = props.fileAssetBucketName ?? BSS_FILE_ASSET_BUCKET_NAME;
        this.repositoryName = props.imageAssetRepositoryName ?? BSS_IMAGE_ASSET_REPOSITORY_NAME;
        this.fileAssetPublishingRoleArn = props.fileAssetPublishingRoleArn ?? BSS_FILE_ASSET_PUBLISHING_ROLE_ARN;
        this.imageAssetPublishingRoleArn = props.imageAssetPublishingRoleArn ?? BSS_IMAGE_ASSET_PUBLISHING_ROLE_ARN;
        this.fileAssetPrefix = props.fileAssetPrefix ?? BSS_FILE_ASSET_PREFIX;
        this.fileAssetRegionSet = props.fileAssetRegionSet ?? commaSplit(BSS_FILE_ASSET_REGION_SET);
        this.templateBucketName = props.templateBucketName ?? BSS_TEMPLATE_BUCKET_NAME;
        this.imageAssetTagPrefix = (props.imageAssetTagPrefix ?? BSS_IMAGE_ASSET_TAG_PREFIX) ?? '';
        this.imageAssetRegionSet = props.imageAssetRegionSet ?? commaSplit(BSS_IMAGE_ASSET_REGION_SET);
        this.imageAssetAccountId = props.imageAssetAccountId ?? BSS_IMAGE_ASSET_ACCOUNT_ID;
    }
    bind(stack) {
        if (this._stack !== undefined) {
            throw new Error('A StackSynthesizer can only be used for one Stack: create a new instance to use with a different Stack');
        }
        this._stack = stack;
        // Function to replace placeholders in the input string as much as possible
        //
        // We replace:
        // - ${AWS::AccountId}, ${AWS::Region}: only if we have the actual values available
        // - ${AWS::Partition}: never, since we never have the actual partition value.
        const specialize = (s) => {
            if (s === undefined) {
                return undefined;
            }
            return cxapi.EnvironmentPlaceholders.replace(s, {
                region: resolvedOr(stack.region, cxapi.EnvironmentPlaceholders.CURRENT_REGION),
                accountId: resolvedOr(stack.account, cxapi.EnvironmentPlaceholders.CURRENT_ACCOUNT),
                partition: cxapi.EnvironmentPlaceholders.CURRENT_PARTITION,
            });
        };
        /* eslint-disable max-len */
        this.bucketName = specialize(this.bucketName);
        this.repositoryName = specialize(this.repositoryName);
        this.fileAssetPublishingRoleArn = specialize(this.fileAssetPublishingRoleArn);
        this.imageAssetPublishingRoleArn = specialize(this.imageAssetPublishingRoleArn);
        this.fileAssetPrefix = specialize(this.fileAssetPrefix ?? '');
        /* eslint-enable max-len */
    }
    addFileAsset(asset) {
        return this._addFileAsset(asset);
    }
    _addFileAsset(asset, overrideBucketname) {
        assertNotNull(this.stack, ERR_MSG_CALL_BIND_FIRST);
        assertNotNull(this.bucketName, 'The bucketName is null');
        validateFileAssetSource(asset);
        const bucketName = overrideBucketname ?? this.bucketName;
        const objectKey = this.fileAssetPrefix + asset.sourceHash + (asset.packaging === aws_cdk_lib_1.FileAssetPackaging.ZIP_DIRECTORY ? '.zip' : '');
        const destinations = {};
        if (this.fileAssetRegionSet?.length && bucketName.includes(REGION_PLACEHOLDER)) {
            for (const region of this.fileAssetRegionSet.map(r => r.trim())) {
                if (!region) {
                    continue;
                }
                destinations[region] = {
                    bucketName: replaceAll(bucketName, REGION_PLACEHOLDER, region),
                    objectKey,
                    region,
                    assumeRoleArn: this.fileAssetPublishingRoleArn,
                };
            }
        }
        else {
            destinations[this.manifestEnvName] = {
                bucketName,
                objectKey,
                region: resolvedOr(this.stack.region, trim(head(this.fileAssetRegionSet))),
                assumeRoleArn: this.fileAssetPublishingRoleArn,
            };
        }
        // Add to manifest
        this.files[asset.sourceHash] = {
            source: {
                path: asset.fileName,
                packaging: asset.packaging,
            },
            destinations,
        };
        const { region, urlSuffix } = stackLocationOrInstrinsics(this.stack);
        const httpUrl = cfnify(`https://s3.${region}.${urlSuffix}/${bucketName}/${objectKey}`);
        const s3ObjectUrl = cfnify(`s3://${bucketName}/${objectKey}`);
        // Return CFN expression
        return {
            bucketName: cfnify(bucketName),
            objectKey,
            httpUrl,
            s3ObjectUrl,
        };
    }
    addDockerImageAsset(asset) {
        assertNotNull(this.stack, ERR_MSG_CALL_BIND_FIRST);
        assertNotNull(this.repositoryName, 'The repositoryName is null');
        validateDockerImageAssetSource(asset);
        const imageTag = this.imageAssetTagPrefix + asset.sourceHash;
        const destinations = {};
        if (this.imageAssetRegionSet?.length) {
            for (const region of this.imageAssetRegionSet.map(r => r.trim())) {
                if (!region) {
                    continue;
                }
                destinations[region] = {
                    repositoryName: this.repositoryName,
                    imageTag,
                    region,
                    assumeRoleArn: this.fileAssetPublishingRoleArn,
                };
            }
        }
        else {
            destinations[this.manifestEnvName] = {
                repositoryName: this.repositoryName,
                imageTag,
                region: resolvedOr(this.stack.region, undefined),
                assumeRoleArn: this.imageAssetPublishingRoleArn,
            };
        }
        // Add to manifest
        this.dockerImages[asset.sourceHash] = {
            source: {
                directory: asset.directoryName,
                dockerBuildArgs: asset.dockerBuildArgs,
                dockerBuildTarget: asset.dockerBuildTarget,
                dockerFile: asset.dockerFile,
            },
            destinations,
        };
        let { account, urlSuffix } = stackLocationOrInstrinsics(this.stack);
        account = this.imageAssetAccountId ?? account;
        return {
            repositoryName: cfnify(this.repositoryName),
            imageUri: cfnify(`${account}.dkr.ecr.${REGION_PLACEHOLDER}.${urlSuffix}/${this.repositoryName}:${imageTag}`),
        };
    }
    /**
     * Dumps current manifest into JSON format
     */
    dumps() {
        const manifest = {
            version: cxschema.Manifest.version(),
            files: this.files,
            dockerImages: this.dockerImages,
        };
        return JSON.stringify(manifest, undefined, 2);
    }
    /**
     * Synthesize the associated stack to the session
     */
    synthesize(session) {
        assertNotNull(this.stack, ERR_MSG_CALL_BIND_FIRST);
        this.synthesizeStackTemplate(this.stack, session);
        // Add the stack's template to the artifact manifest
        const templateManifestUrl = this.addStackTemplateToAssetManifest(session);
        const artifactId = this.writeAssetManifest(session);
        this.emitStackArtifact(this.stack, session, {
            stackTemplateAssetObjectUrl: templateManifestUrl,
            additionalDependencies: [artifactId],
        });
    }
    get stack() {
        return this._stack;
    }
    /**
     * Add the stack's template as one of the manifest assets
     *
     * This will make it get uploaded to S3 automatically by S3-assets. Return
     * the manifest URL.
     *
     * (We can't return the location returned from `addFileAsset`, as that
     * contains CloudFormation intrinsics which can't go into the manifest).
     */
    addStackTemplateToAssetManifest(_) {
        assertNotNull(this.stack, ERR_MSG_CALL_BIND_FIRST);
        const sourceHash = this.stack.templateFile;
        this._addFileAsset({
            fileName: this.stack.templateFile,
            packaging: aws_cdk_lib_1.FileAssetPackaging.FILE,
            sourceHash,
        }, this.templateBucketName);
        // We should technically return an 'https://s3.REGION.amazonaws.com[.cn]/name/hash' URL here,
        // because that is what CloudFormation expects to see.
        //
        // However, there's no way for us to actually know the UrlSuffix a priori, so we can't construct it here.
        //
        // Instead, we'll have a protocol with the CLI that we put an 's3://.../...' URL here, and the CLI
        // is going to resolve it to the correct 'https://.../' URL before it gives it to CloudFormation.
        return `s3://${this.bucketName}/${sourceHash}`;
    }
    /**
     * Write an asset manifest to the Cloud Assembly, return the artifact IDs written
     */
    writeAssetManifest(session) {
        assertNotNull(this.stack, ERR_MSG_CALL_BIND_FIRST);
        const artifactId = `${this.stack.artifactId}.assets`;
        const manifestFile = `${artifactId}.json`;
        const outPath = path.join(session.assembly.outdir, manifestFile);
        fs.writeFileSync(outPath, this.dumps());
        session.assembly.addArtifact(artifactId, {
            type: cxschema.ArtifactType.ASSET_MANIFEST,
            properties: {
                file: manifestFile,
            },
        });
        return artifactId;
    }
    get manifestEnvName() {
        assertNotNull(this.stack, ERR_MSG_CALL_BIND_FIRST);
        return [
            resolvedOr(this.stack.account, 'current_account'),
            resolvedOr(this.stack.region, 'current_region'),
        ].join('-');
    }
}
exports.BootstraplessStackSynthesizer = BootstraplessStackSynthesizer;
_a = JSII_RTTI_SYMBOL_1;
BootstraplessStackSynthesizer[_a] = { fqn: "cdk-bootstrapless-synthesizer.BootstraplessStackSynthesizer", version: "2.2.11" };
/**
 * Return the given value if resolved or fall back to a default
 */
function resolvedOr(x, def) {
    return aws_cdk_lib_1.Token.isUnresolved(x) ? def : x;
}
/**
 * A "replace-all" function that doesn't require us escaping a literal string to a regex
 */
function replaceAll(s, search, replace) {
    return s.split(search).join(replace);
}
/**
 * If the string still contains placeholders, wrap it in a Fn::Sub so they will be substituted at CFN deployment time
 *
 * (This happens to work because the placeholders we picked map directly onto CFN
 * placeholders. If they didn't we'd have to do a transformation here).
 */
function cfnify(s) {
    return s.indexOf('${') > -1 ? aws_cdk_lib_1.Fn.sub(s) : s;
}
/**
 * Return the stack locations if they're concrete, or the original CFN intrisics otherwise
 *
 * We need to return these instead of the tokenized versions of the strings,
 * since we must accept those same ${AWS::AccountId}/${AWS::Region} placeholders
 * in bucket names and role names (in order to allow environment-agnostic stacks).
 *
 * We'll wrap a single {Fn::Sub} around the final string in order to replace everything,
 * but we can't have the token system render part of the string to {Fn::Join} because
 * the CFN specification doesn't allow the {Fn::Sub} template string to be an arbitrary
 * expression--it must be a string literal.
 */
function stackLocationOrInstrinsics(stack) {
    return {
        account: resolvedOr(stack.account, '${AWS::AccountId}'),
        region: resolvedOr(stack.region, '${AWS::Region}'),
        urlSuffix: resolvedOr(stack.urlSuffix, '${AWS::URLSuffix}'),
    };
}
// function range(startIncl: number, endExcl: number) {
//     const ret = new Array<number>();
//     for (let i = startIncl; i < endExcl; i++) {
//     ret.push(i);
//     }
//     return ret;
// }
function assertNotNull(x, msg = 'Null value error') {
    if (x === null || x === undefined) {
        throw new Error(msg);
    }
}
function commaSplit(v) {
    if (v) {
        return v.split(',');
    }
    return undefined;
}
function validateFileAssetSource(asset) {
    if (!!asset.executable === !!asset.fileName) {
        throw new Error(`Exactly one of 'fileName' or 'executable' is required, got: ${JSON.stringify(asset)}`);
    }
    if (!!asset.packaging !== !!asset.fileName) {
        throw new Error(`'packaging' is expected in combination with 'fileName', got: ${JSON.stringify(asset)}`);
    }
}
function validateDockerImageAssetSource(asset) {
    if (!!asset.executable === !!asset.directoryName) {
        throw new Error(`Exactly one of 'directoryName' or 'executable' is required, got: ${JSON.stringify(asset)}`);
    }
    check('dockerBuildArgs');
    check('dockerBuildTarget');
    check('dockerFile');
    function check(key) {
        if (asset[key] && !asset.directoryName) {
            throw new Error(`'${key}' is only allowed in combination with 'directoryName', got: ${JSON.stringify(asset)}`);
        }
    }
}
function head(ss) {
    if (ss && ss.length > 0) {
        return ss[0];
    }
    return undefined;
}
function trim(s) {
    if (s) {
        return s.trim();
    }
    return undefined;
}
//# sourceMappingURL=data:application/json;base64,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