"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontOriginAccessIdentity = exports.CloudFrontDistributionForMediaStore = exports.CloudFrontDistributionForS3 = exports.CloudFrontDistributionForApiGateway = void 0;
const cloudfront = require("@aws-cdk/aws-cloudfront");
const cdk = require("@aws-cdk/core");
const cloudfront_distribution_defaults_1 = require("./cloudfront-distribution-defaults");
const utils_1 = require("./utils");
const s3_bucket_helper_1 = require("./s3-bucket-helper");
const s3_bucket_defaults_1 = require("./s3-bucket-defaults");
// Override Cfn_Nag rule: Cloudfront TLS-1.2 rule (https://github.com/stelligent/cfn_nag/issues/384)
function updateSecurityPolicy(cfDistribution) {
    utils_1.addCfnSuppressRules(cfDistribution, [
        {
            id: 'W70',
            reason: `Since the distribution uses the CloudFront domain name, CloudFront automatically sets the security policy to TLSv1 regardless of the value of MinimumProtocolVersion`
        }
    ]);
    return cfDistribution;
}
// Cloudfront function to insert the HTTP Security Headers into the response coming from the origin servers
// and before it is sent to the client
function defaultCloudfrontFunction(scope) {
    // generate a stable unique id for the cloudfront function and use it
    // both for the function name and the logical id of the function so if
    // it is changed the function will be recreated.
    // see https://github.com/aws/aws-cdk/issues/15523
    const functionId = `SetHttpSecurityHeaders${scope.node.addr}`;
    return new cloudfront.Function(scope, "SetHttpSecurityHeaders", {
        functionName: functionId,
        code: cloudfront.FunctionCode.fromInline("function handler(event) { var response = event.response; \
      var headers = response.headers; \
      headers['strict-transport-security'] = { value: 'max-age=63072000; includeSubdomains; preload'}; \
      headers['content-security-policy'] = { value: \"default-src 'none'; img-src 'self'; script-src 'self'; style-src 'self'; object-src 'none'\"}; \
      headers['x-content-type-options'] = { value: 'nosniff'}; \
      headers['x-frame-options'] = {value: 'DENY'}; \
      headers['x-xss-protection'] = {value: '1; mode=block'}; \
      return response; \
    }")
    });
}
function CloudFrontDistributionForApiGateway(scope, apiEndPoint, cloudFrontDistributionProps, httpSecurityHeaders = true, cloudFrontLoggingBucketProps) {
    const cloudfrontFunction = getCloudfrontFunction(httpSecurityHeaders, scope);
    const loggingBucket = getLoggingBucket(cloudFrontDistributionProps, scope, cloudFrontLoggingBucketProps);
    const defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForApiGatewayProps(apiEndPoint, loggingBucket, httpSecurityHeaders, cloudfrontFunction);
    const cfprops = cloudFrontDistributionProps ? utils_1.overrideProps(defaultprops, cloudFrontDistributionProps, false) : defaultprops;
    // Create the Cloudfront Distribution
    const cfDistribution = new cloudfront.Distribution(scope, 'CloudFrontDistribution', cfprops);
    updateSecurityPolicy(cfDistribution);
    return [cfDistribution, cloudfrontFunction, loggingBucket];
}
exports.CloudFrontDistributionForApiGateway = CloudFrontDistributionForApiGateway;
function CloudFrontDistributionForS3(scope, sourceBucket, cloudFrontDistributionProps, httpSecurityHeaders = true, cloudFrontLoggingBucketProps) {
    const cloudfrontFunction = getCloudfrontFunction(httpSecurityHeaders, scope);
    const loggingBucket = getLoggingBucket(cloudFrontDistributionProps, scope, cloudFrontLoggingBucketProps);
    const defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForS3Props(sourceBucket, loggingBucket, httpSecurityHeaders, cloudfrontFunction);
    const cfprops = cloudFrontDistributionProps ? utils_1.overrideProps(defaultprops, cloudFrontDistributionProps, false) : defaultprops;
    // Create the Cloudfront Distribution
    const cfDistribution = new cloudfront.Distribution(scope, 'CloudFrontDistribution', cfprops);
    updateSecurityPolicy(cfDistribution);
    // Extract the CfnBucketPolicy from the sourceBucket
    const bucketPolicy = sourceBucket.policy;
    // the lack of a bucketPolicy means the bucket was imported from outside the stack so the lack of cfn_nag suppression is not an issue
    if (bucketPolicy) {
        utils_1.addCfnSuppressRules(bucketPolicy, [
            {
                id: 'F16',
                reason: `Public website bucket policy requires a wildcard principal`
            }
        ]);
    }
    return [cfDistribution, cloudfrontFunction, loggingBucket];
}
exports.CloudFrontDistributionForS3 = CloudFrontDistributionForS3;
function CloudFrontDistributionForMediaStore(scope, mediaStoreContainer, cloudFrontDistributionProps, httpSecurityHeaders = true, cloudFrontLoggingBucketProps) {
    let originRequestPolicy;
    const loggingBucket = getLoggingBucket(cloudFrontDistributionProps, scope, cloudFrontLoggingBucketProps);
    if (cloudFrontDistributionProps
        && cloudFrontDistributionProps.defaultBehavior
        && cloudFrontDistributionProps.defaultBehavior.originRequestPolicy) {
        originRequestPolicy = cloudFrontDistributionProps.defaultBehavior.originRequestPolicy;
    }
    else {
        const originRequestPolicyProps = {
            headerBehavior: {
                behavior: 'whitelist',
                headers: [
                    'Access-Control-Allow-Origin',
                    'Access-Control-Request-Method',
                    'Access-Control-Request-Header',
                    'Origin'
                ]
            },
            queryStringBehavior: {
                behavior: 'all'
            },
            cookieBehavior: {
                behavior: 'none'
            },
            comment: 'Policy for Constructs CloudFrontDistributionForMediaStore',
            originRequestPolicyName: `${cdk.Aws.STACK_NAME}-${cdk.Aws.REGION}-CloudFrontDistributionForMediaStore`
        };
        originRequestPolicy = new cloudfront.OriginRequestPolicy(scope, 'CloudfrontOriginRequestPolicy', originRequestPolicyProps);
    }
    const cloudfrontFunction = getCloudfrontFunction(httpSecurityHeaders, scope);
    const defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontDisributionForMediaStoreProps(mediaStoreContainer, loggingBucket, originRequestPolicy, httpSecurityHeaders, cloudFrontDistributionProps === null || cloudFrontDistributionProps === void 0 ? void 0 : cloudFrontDistributionProps.customHeaders, cloudfrontFunction);
    let cfprops;
    if (cloudFrontDistributionProps) {
        cfprops = utils_1.overrideProps(defaultprops, cloudFrontDistributionProps, false);
    }
    else {
        cfprops = defaultprops;
    }
    // Create the CloudFront Distribution
    const cfDistribution = new cloudfront.Distribution(scope, 'CloudFrontDistribution', cfprops);
    updateSecurityPolicy(cfDistribution);
    return [cfDistribution, loggingBucket, originRequestPolicy, cloudfrontFunction];
}
exports.CloudFrontDistributionForMediaStore = CloudFrontDistributionForMediaStore;
function CloudFrontOriginAccessIdentity(scope, comment) {
    return new cloudfront.OriginAccessIdentity(scope, 'CloudFrontOriginAccessIdentity', {
        comment: comment ? comment : `access-identity-${cdk.Aws.REGION}-${cdk.Aws.STACK_NAME}`
    });
}
exports.CloudFrontOriginAccessIdentity = CloudFrontOriginAccessIdentity;
function getLoggingBucket(cloudFrontDistributionProps, scope, cloudFrontLoggingBucketProps) {
    const isLoggingDisabled = (cloudFrontDistributionProps === null || cloudFrontDistributionProps === void 0 ? void 0 : cloudFrontDistributionProps.enableLogging) === false;
    const userSuppliedLogBucket = cloudFrontDistributionProps === null || cloudFrontDistributionProps === void 0 ? void 0 : cloudFrontDistributionProps.logBucket;
    if (userSuppliedLogBucket && cloudFrontLoggingBucketProps) {
        throw Error('Either cloudFrontDistributionProps.logBucket or cloudFrontLoggingBucketProps can be set.');
    }
    return isLoggingDisabled
        ? undefined
        : userSuppliedLogBucket !== null && userSuppliedLogBucket !== void 0 ? userSuppliedLogBucket : s3_bucket_helper_1.createLoggingBucket(scope, 'CloudfrontLoggingBucket', cloudFrontLoggingBucketProps ? utils_1.overrideProps(s3_bucket_defaults_1.DefaultS3Props(), cloudFrontLoggingBucketProps) : s3_bucket_defaults_1.DefaultS3Props());
}
function getCloudfrontFunction(httpSecurityHeaders, scope) {
    return httpSecurityHeaders ? defaultCloudfrontFunction(scope) : undefined;
}
//# sourceMappingURL=data:application/json;base64,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