"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.createSagemakerEndpoint = exports.createSagemakerEndpointConfig = exports.createSagemakerModel = exports.deploySagemakerEndpoint = exports.BuildSagemakerEndpoint = exports.buildSagemakerNotebook = void 0;
const sagemaker = require("@aws-cdk/aws-sagemaker");
const ec2 = require("@aws-cdk/aws-ec2");
const kms_helper_1 = require("./kms-helper");
const sagemaker_defaults_1 = require("./sagemaker-defaults");
const cdk = require("@aws-cdk/core");
const utils_1 = require("./utils");
const vpc_helper_1 = require("./vpc-helper");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const vpc_defaults_1 = require("./vpc-defaults");
const security_group_helper_1 = require("./security-group-helper");
function addPermissions(_role, props) {
    var _a, _b;
    // Grant permissions to NoteBookInstance for creating and training the model
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [`arn:${core_1.Aws.PARTITION}:sagemaker:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:*`],
        actions: [
            'sagemaker:CreateTrainingJob',
            'sagemaker:DescribeTrainingJob',
            'sagemaker:CreateModel',
            'sagemaker:DescribeModel',
            'sagemaker:DeleteModel',
            'sagemaker:CreateEndpoint',
            'sagemaker:CreateEndpointConfig',
            'sagemaker:DescribeEndpoint',
            'sagemaker:DescribeEndpointConfig',
            'sagemaker:DeleteEndpoint',
            'sagemaker:DeleteEndpointConfig',
            'sagemaker:InvokeEndpoint',
        ],
    }));
    // Grant CloudWatch Logging permissions
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:/aws/sagemaker/*`],
        actions: [
            'logs:CreateLogGroup',
            'logs:CreateLogStream',
            'logs:DescribeLogStreams',
            'logs:GetLogEvents',
            'logs:PutLogEvents',
        ],
    }));
    // To place the Sagemaker endpoint in a VPC
    if (props && props.vpc) {
        _role.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:CreateNetworkInterfacePermission',
                'ec2:DeleteNetworkInterface',
                'ec2:DeleteNetworkInterfacePermission',
                'ec2:DescribeNetworkInterfaces',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
                'ec2:DescribeVpcs',
                'ec2:DescribeDhcpOptions',
                'ec2:DescribeSubnets',
                'ec2:DescribeSecurityGroups',
            ],
        }));
    }
    // To create a Sagemaker model using Bring-Your-Own-Model (BYOM) algorith image
    // The image URL is specified in the modelProps
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [`arn:${cdk.Aws.PARTITION}:ecr:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:repository/*`],
        actions: [
            'ecr:BatchCheckLayerAvailability',
            'ecr:GetDownloadUrlForLayer',
            'ecr:DescribeRepositories',
            'ecr:DescribeImages',
            'ecr:BatchGetImage',
        ],
    }));
    // Add GetAuthorizationToken (it can not be bound to resources other than *)
    _role.addToPolicy(new iam.PolicyStatement({
        resources: ['*'],
        actions: ['ecr:GetAuthorizationToken'],
    }));
    // add permission to use Elastic Inference accelerator
    if (props && props.endpointConfigProps) {
        // Get the acceleratorType, if any
        const acceleratorType = ((_a = props.endpointConfigProps) === null || _a === void 0 ? void 0 : _a.productionVariants)[0].acceleratorType;
        if (acceleratorType !== undefined) {
            _role.addToPolicy(new iam.PolicyStatement({
                resources: ['*'],
                actions: ['elastic-inference:Connect'],
            }));
        }
    }
    // add kms permissions
    _role.addToPolicy(new iam.PolicyStatement({
        // the kmsKeyId in the endpointConfigProps can be any of the following formats:
        // Key ID: 1234abcd-12ab-34cd-56ef-1234567890ab
        // Key ARN: arn:aws:kms:<region>:<accountID>:key/1234abcd-12ab-34cd-56ef-1234567890ab
        // Alias name: alias/ExampleAlias
        // Alias name ARN: arn:aws:kms:<region>:<accountID>:alias/ExampleAlias
        // the key is used to encrypt/decrypt data captured by the Sagemaker endpoint and stored in S3 bucket
        resources: [
            `arn:${cdk.Aws.PARTITION}:kms:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:key/*`,
            `arn:${cdk.Aws.PARTITION}:kms:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:alias/*`,
        ],
        actions: ['kms:Encrypt', 'kms:Decrypt', 'kms:ReEncrypt*', 'kms:GenerateDataKey*', 'kms:DescribeKey'],
    }));
    // Add S3 permissions to get Model artifact, put data capture files, etc.
    _role.addToPolicy(new iam.PolicyStatement({
        actions: ['s3:GetObject', 's3:PutObject', 's3:DeleteObject', 's3:ListBucket'],
        resources: ['arn:aws:s3:::*'],
    }));
    // Grant GetRole permissions to the Sagemaker service
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [_role.roleArn],
        actions: ['iam:GetRole'],
    }));
    // Grant PassRole permissions to the Sagemaker service
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [_role.roleArn],
        actions: ['iam:PassRole'],
        conditions: {
            StringLike: { 'iam:PassedToService': 'sagemaker.amazonaws.com' },
        },
    }));
    // Add CFN NAG uppress to allow for "Resource": "*" for ENI access in VPC,
    // ECR authorization token for custom model images, and elastic inference
    // Add CFN NAG for Complex Role because Sagmaker needs permissions to access several services
    const roleDefaultPolicy = (_b = _role.node.tryFindChild('DefaultPolicy')) === null || _b === void 0 ? void 0 : _b.node.findChild('Resource');
    utils_1.addCfnSuppressRules(roleDefaultPolicy, [
        {
            id: 'W12',
            reason: `Sagemaker needs the following minimum required permissions to access ENIs in a VPC, ECR for custom model images, and elastic inference.`,
        },
        {
            id: 'W76',
            reason: 'Complex role becuase Sagemaker needs permissions to access several services',
        }
    ]);
}
function buildSagemakerNotebook(scope, props) {
    var _a, _b, _c, _d, _e, _f, _g, _h, _j;
    // Setup the notebook properties
    let sagemakerNotebookProps;
    let vpcInstance;
    let securityGroup;
    let kmsKeyId;
    let subnetId;
    // Conditional Sagemaker Notebook creation
    if (!props.existingNotebookObj) {
        if ((((_a = props.sagemakerNotebookProps) === null || _a === void 0 ? void 0 : _a.subnetId) && ((_b = props.sagemakerNotebookProps) === null || _b === void 0 ? void 0 : _b.securityGroupIds) === undefined) ||
            (((_c = props.sagemakerNotebookProps) === null || _c === void 0 ? void 0 : _c.subnetId) === undefined && ((_d = props.sagemakerNotebookProps) === null || _d === void 0 ? void 0 : _d.securityGroupIds))) {
            throw new Error('Must define both sagemakerNotebookProps.subnetId and sagemakerNotebookProps.securityGroupIds');
        }
        addPermissions(props.role);
        if (((_e = props.sagemakerNotebookProps) === null || _e === void 0 ? void 0 : _e.kmsKeyId) === undefined) {
            kmsKeyId = kms_helper_1.buildEncryptionKey(scope).keyId;
        }
        else {
            kmsKeyId = props.sagemakerNotebookProps.kmsKeyId;
        }
        if (props.deployInsideVpc === undefined || props.deployInsideVpc) {
            if (((_f = props.sagemakerNotebookProps) === null || _f === void 0 ? void 0 : _f.subnetId) === undefined &&
                ((_g = props.sagemakerNotebookProps) === null || _g === void 0 ? void 0 : _g.securityGroupIds) === undefined) {
                vpcInstance = vpc_helper_1.buildVpc(scope, {
                    defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
                });
                securityGroup = security_group_helper_1.buildSecurityGroup(scope, 'SecurityGroup', {
                    vpc: vpcInstance,
                    allowAllOutbound: false,
                }, [], [{ peer: ec2.Peer.anyIpv4(), connection: ec2.Port.tcp(443) }]);
                subnetId = vpcInstance.privateSubnets[0].subnetId;
                sagemakerNotebookProps = sagemaker_defaults_1.DefaultSagemakerNotebookProps(props.role.roleArn, kmsKeyId, subnetId, [
                    securityGroup.securityGroupId,
                ]);
            }
            else {
                sagemakerNotebookProps = sagemaker_defaults_1.DefaultSagemakerNotebookProps(props.role.roleArn, kmsKeyId, (_h = props.sagemakerNotebookProps) === null || _h === void 0 ? void 0 : _h.subnetId, (_j = props.sagemakerNotebookProps) === null || _j === void 0 ? void 0 : _j.securityGroupIds);
            }
        }
        else {
            sagemakerNotebookProps = sagemaker_defaults_1.DefaultSagemakerNotebookProps(props.role.roleArn, kmsKeyId);
        }
        if (props.sagemakerNotebookProps) {
            sagemakerNotebookProps = utils_1.overrideProps(sagemakerNotebookProps, props.sagemakerNotebookProps);
        }
        // Create the notebook
        const sagemakerInstance = new sagemaker.CfnNotebookInstance(scope, 'SagemakerNotebook', sagemakerNotebookProps);
        if (vpcInstance) {
            return [sagemakerInstance, vpcInstance, securityGroup];
        }
        else {
            return [sagemakerInstance];
        }
    }
    else {
        // Return existing notebook object
        return [props.existingNotebookObj];
    }
}
exports.buildSagemakerNotebook = buildSagemakerNotebook;
function BuildSagemakerEndpoint(scope, props) {
    /** Conditional Sagemaker endpoint creation */
    if (!props.existingSagemakerEndpointObj) {
        if (props.modelProps) {
            /** return [endpoint, endpointConfig, model] */
            return deploySagemakerEndpoint(scope, props);
        }
        else {
            throw Error('Either existingSagemakerEndpointObj or at least modelProps is required');
        }
    }
    else {
        /** Otherwise, return [endpoint] */
        return [props.existingSagemakerEndpointObj];
    }
}
exports.BuildSagemakerEndpoint = BuildSagemakerEndpoint;
function deploySagemakerEndpoint(scope, props) {
    let model;
    let endpointConfig;
    let endpoint;
    let sagemakerRole;
    // Create Sagemaker's model, endpointConfig, and endpoint
    if (props.modelProps) {
        // Check if the client has provided executionRoleArn
        if (props.modelProps.executionRoleArn) {
            sagemakerRole = iam.Role.fromRoleArn(scope, 'SagemakerRoleCustomer', props.modelProps.executionRoleArn);
        }
        else {
            // Create the Sagemaker Role
            sagemakerRole = new iam.Role(scope, 'SagemakerRole', {
                assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
            });
            // Add required permissions
            addPermissions(sagemakerRole, props);
        }
        // Create Sagemaker Model
        model = createSagemakerModel(scope, props.modelProps, sagemakerRole, props.vpc);
        // Create Sagemaker EndpointConfig
        endpointConfig = createSagemakerEndpointConfig(scope, model.attrModelName, props.endpointConfigProps);
        // Add dependency on model
        endpointConfig.addDependsOn(model);
        // Create Sagemaker Endpoint
        endpoint = createSagemakerEndpoint(scope, endpointConfig.attrEndpointConfigName, props.endpointProps);
        // Add dependency on EndpointConfig
        endpoint.addDependsOn(endpointConfig);
        return [endpoint, endpointConfig, model];
    }
    else {
        throw Error('You need to provide at least modelProps to create Sagemaker Endpoint');
    }
}
exports.deploySagemakerEndpoint = deploySagemakerEndpoint;
function createSagemakerModel(scope, modelProps, role, vpc) {
    let finalModelProps;
    let primaryContainer;
    let vpcConfig;
    let model;
    if (vpc) {
        const modelDefaultSecurityGroup = new ec2.SecurityGroup(scope, 'ReplaceModelDefaultSecurityGroup', {
            vpc,
            allowAllOutbound: true,
        });
        // Allow https traffic from within the VPC
        modelDefaultSecurityGroup.addIngressRule(ec2.Peer.ipv4(vpc.vpcCidrBlock), ec2.Port.tcp(443));
        const cfnSecurityGroup = modelDefaultSecurityGroup.node.findChild('Resource');
        utils_1.addCfnSuppressRules(cfnSecurityGroup, [
            {
                id: 'W5',
                reason: 'Egress of 0.0.0.0/0 is default and generally considered OK',
            },
            {
                id: 'W40',
                reason: 'Egress IPProtocol of -1 is default and generally considered OK',
            }
        ]);
        // Throw an error if the VPC does not contain private or isolated subnets
        if (vpc.privateSubnets.length === 0 && vpc.isolatedSubnets.length === 0) {
            throw Error('VPC must contain private or isolated subnets to deploy the Sagemaker endpoint in a vpc');
        }
        vpcConfig = {
            // default SubnetType.PRIVATE (or ISOLATED or PUBLIC if there are no PRIVATE subnets)
            // So, private subnets will be used if provided by customer. Otherwise, use the default isolated subnets,
            subnets: vpc.selectSubnets({
                onePerAz: true,
            }).subnetIds,
            securityGroupIds: [modelDefaultSecurityGroup.securityGroupId],
        };
    }
    if (modelProps.primaryContainer) {
        // Get user provided Model's primary container
        primaryContainer = modelProps.primaryContainer;
        // Get default Model props
        finalModelProps = sagemaker_defaults_1.DefaultSagemakerModelProps(role.roleArn, primaryContainer, vpcConfig);
        // Override default model properties
        finalModelProps = utils_1.overrideProps(finalModelProps, modelProps);
        // Create the Sagemaker's Model
        model = new sagemaker.CfnModel(scope, 'SagemakerModel', finalModelProps);
        // Add dependency on the Sagemaker's role
        model.node.addDependency(role);
        return model;
    }
    else {
        throw Error('You need to provide at least primaryContainer to create Sagemaker Model');
    }
}
exports.createSagemakerModel = createSagemakerModel;
function createSagemakerEndpointConfig(scope, modelName, endpointConfigProps) {
    let finalEndpointConfigProps;
    let kmsKeyId;
    let endpointConfig;
    // Create encryption key if one is not provided
    if (endpointConfigProps && endpointConfigProps.kmsKeyId) {
        kmsKeyId = endpointConfigProps.kmsKeyId;
    }
    else {
        kmsKeyId = kms_helper_1.buildEncryptionKey(scope).keyId;
    }
    // Get default EndpointConfig props
    finalEndpointConfigProps = sagemaker_defaults_1.DefaultSagemakerEndpointConfigProps(modelName, kmsKeyId);
    // Overwrite default EndpointConfig properties
    if (endpointConfigProps) {
        finalEndpointConfigProps = utils_1.overrideProps(finalEndpointConfigProps, endpointConfigProps);
    }
    // Create the Sagemaker's EndpointConfig
    endpointConfig = new sagemaker.CfnEndpointConfig(scope, 'SagemakerEndpointConfig', finalEndpointConfigProps);
    return endpointConfig;
}
exports.createSagemakerEndpointConfig = createSagemakerEndpointConfig;
function createSagemakerEndpoint(scope, endpointConfigName, endpointProps) {
    let finalEndpointProps;
    let endpoint;
    // Get default Endpoint props
    finalEndpointProps = sagemaker_defaults_1.DefaultSagemakerEndpointProps(endpointConfigName);
    // Overwrite default Endpoint properties
    if (endpointProps) {
        finalEndpointProps = utils_1.overrideProps(finalEndpointProps, endpointProps);
    }
    // Create the Sagemaker's Endpoint
    endpoint = new sagemaker.CfnEndpoint(scope, 'SagemakerEndpoint', finalEndpointProps);
    return endpoint;
}
exports.createSagemakerEndpoint = createSagemakerEndpoint;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2FnZW1ha2VyLWhlbHBlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInNhZ2VtYWtlci1oZWxwZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7Ozs7Ozs7OztHQVdHOzs7QUFFSCxvREFBb0Q7QUFDcEQsd0NBQXdDO0FBQ3hDLDZDQUFrRDtBQUNsRCw2REFLOEI7QUFDOUIscUNBQXFDO0FBQ3JDLG1DQUE2RDtBQUM3RCw2Q0FBd0M7QUFDeEMsd0NBQXdDO0FBQ3hDLHdDQUFvQztBQUNwQyxpREFBOEQ7QUFDOUQsbUVBQTZEO0FBZTdELFNBQVMsY0FBYyxDQUFDLEtBQWUsRUFBRSxLQUFtQzs7SUFDMUUsNEVBQTRFO0lBQzVFLEtBQUssQ0FBQyxXQUFXLENBQ2YsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1FBQ3RCLFNBQVMsRUFBRSxDQUFDLE9BQU8sVUFBRyxDQUFDLFNBQVMsY0FBYyxVQUFHLENBQUMsTUFBTSxJQUFJLFVBQUcsQ0FBQyxVQUFVLElBQUksQ0FBQztRQUMvRSxPQUFPLEVBQUU7WUFDUCw2QkFBNkI7WUFDN0IsK0JBQStCO1lBQy9CLHVCQUF1QjtZQUN2Qix5QkFBeUI7WUFDekIsdUJBQXVCO1lBQ3ZCLDBCQUEwQjtZQUMxQixnQ0FBZ0M7WUFDaEMsNEJBQTRCO1lBQzVCLGtDQUFrQztZQUNsQywwQkFBMEI7WUFDMUIsZ0NBQWdDO1lBQ2hDLDBCQUEwQjtTQUMzQjtLQUNGLENBQUMsQ0FDSCxDQUFDO0lBRUYsdUNBQXVDO0lBQ3ZDLEtBQUssQ0FBQyxXQUFXLENBQ2YsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1FBQ3RCLFNBQVMsRUFBRSxDQUFDLE9BQU8sR0FBRyxDQUFDLEdBQUcsQ0FBQyxTQUFTLFNBQVMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxNQUFNLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxVQUFVLDZCQUE2QixDQUFDO1FBQy9HLE9BQU8sRUFBRTtZQUNQLHFCQUFxQjtZQUNyQixzQkFBc0I7WUFDdEIseUJBQXlCO1lBQ3pCLG1CQUFtQjtZQUNuQixtQkFBbUI7U0FDcEI7S0FDRixDQUFDLENBQ0gsQ0FBQztJQUVGLDJDQUEyQztJQUMzQyxJQUFJLEtBQUssSUFBSSxLQUFLLENBQUMsR0FBRyxFQUFFO1FBQ3RCLEtBQUssQ0FBQyxXQUFXLENBQ2YsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQ3RCLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztZQUNoQixPQUFPLEVBQUU7Z0JBQ1AsNEJBQTRCO2dCQUM1QixzQ0FBc0M7Z0JBQ3RDLDRCQUE0QjtnQkFDNUIsc0NBQXNDO2dCQUN0QywrQkFBK0I7Z0JBQy9CLDhCQUE4QjtnQkFDOUIsZ0NBQWdDO2dCQUNoQyxrQkFBa0I7Z0JBQ2xCLHlCQUF5QjtnQkFDekIscUJBQXFCO2dCQUNyQiw0QkFBNEI7YUFDN0I7U0FDRixDQUFDLENBQ0gsQ0FBQztLQUNIO0lBRUQsK0VBQStFO0lBQy9FLCtDQUErQztJQUMvQyxLQUFLLENBQUMsV0FBVyxDQUNmLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztRQUN0QixTQUFTLEVBQUUsQ0FBQyxPQUFPLEdBQUcsQ0FBQyxHQUFHLENBQUMsU0FBUyxRQUFRLEdBQUcsQ0FBQyxHQUFHLENBQUMsTUFBTSxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsVUFBVSxlQUFlLENBQUM7UUFDaEcsT0FBTyxFQUFFO1lBQ1AsaUNBQWlDO1lBQ2pDLDRCQUE0QjtZQUM1QiwwQkFBMEI7WUFDMUIsb0JBQW9CO1lBQ3BCLG1CQUFtQjtTQUNwQjtLQUNGLENBQUMsQ0FDSCxDQUFDO0lBRUYsNEVBQTRFO0lBQzVFLEtBQUssQ0FBQyxXQUFXLENBQ2YsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1FBQ3RCLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztRQUNoQixPQUFPLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQztLQUN2QyxDQUFDLENBQ0gsQ0FBQztJQUVGLHNEQUFzRDtJQUN0RCxJQUFJLEtBQUssSUFBSSxLQUFLLENBQUMsbUJBQW1CLEVBQUU7UUFDdEMsa0NBQWtDO1FBQ2xDLE1BQU0sZUFBZSxHQUFHLENBQUMsTUFBQSxLQUFLLENBQUMsbUJBQW1CLDBDQUM5QyxrQkFBOEUsQ0FBQSxDQUFDLENBQUMsQ0FBQyxDQUFDLGVBQWUsQ0FBQztRQUN0RyxJQUFJLGVBQWUsS0FBSyxTQUFTLEVBQUU7WUFDakMsS0FBSyxDQUFDLFdBQVcsQ0FDZixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7Z0JBQ3RCLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztnQkFDaEIsT0FBTyxFQUFFLENBQUMsMkJBQTJCLENBQUM7YUFDdkMsQ0FBQyxDQUNILENBQUM7U0FDSDtLQUNGO0lBRUQsc0JBQXNCO0lBQ3RCLEtBQUssQ0FBQyxXQUFXLENBQ2YsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1FBQ3RCLCtFQUErRTtRQUMvRSwrQ0FBK0M7UUFDL0MscUZBQXFGO1FBQ3JGLGlDQUFpQztRQUNqQyxzRUFBc0U7UUFDdEUscUdBQXFHO1FBQ3JHLFNBQVMsRUFBRTtZQUNULE9BQU8sR0FBRyxDQUFDLEdBQUcsQ0FBQyxTQUFTLFFBQVEsR0FBRyxDQUFDLEdBQUcsQ0FBQyxNQUFNLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxVQUFVLFFBQVE7WUFDNUUsT0FBTyxHQUFHLENBQUMsR0FBRyxDQUFDLFNBQVMsUUFBUSxHQUFHLENBQUMsR0FBRyxDQUFDLE1BQU0sSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLFVBQVUsVUFBVTtTQUMvRTtRQUNELE9BQU8sRUFBRSxDQUFDLGFBQWEsRUFBRSxhQUFhLEVBQUUsZ0JBQWdCLEVBQUUsc0JBQXNCLEVBQUUsaUJBQWlCLENBQUM7S0FDckcsQ0FBQyxDQUNILENBQUM7SUFFRix5RUFBeUU7SUFDekUsS0FBSyxDQUFDLFdBQVcsQ0FDZixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7UUFDdEIsT0FBTyxFQUFFLENBQUMsY0FBYyxFQUFFLGNBQWMsRUFBRSxpQkFBaUIsRUFBRSxlQUFlLENBQUM7UUFDN0UsU0FBUyxFQUFFLENBQUMsZ0JBQWdCLENBQUM7S0FDOUIsQ0FBQyxDQUNILENBQUM7SUFFRixxREFBcUQ7SUFDckQsS0FBSyxDQUFDLFdBQVcsQ0FDZixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7UUFDdEIsU0FBUyxFQUFFLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQztRQUMxQixPQUFPLEVBQUUsQ0FBQyxhQUFhLENBQUM7S0FDekIsQ0FBQyxDQUNILENBQUM7SUFFRixzREFBc0Q7SUFDdEQsS0FBSyxDQUFDLFdBQVcsQ0FDZixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7UUFDdEIsU0FBUyxFQUFFLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQztRQUMxQixPQUFPLEVBQUUsQ0FBQyxjQUFjLENBQUM7UUFDekIsVUFBVSxFQUFFO1lBQ1YsVUFBVSxFQUFFLEVBQUUscUJBQXFCLEVBQUUseUJBQXlCLEVBQUU7U0FDakU7S0FDRixDQUFDLENBQ0gsQ0FBQztJQUVGLDBFQUEwRTtJQUMxRSx5RUFBeUU7SUFDekUsNkZBQTZGO0lBQzdGLE1BQU0saUJBQWlCLEdBQUcsTUFBQSxLQUFLLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxlQUFlLENBQUMsMENBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQWtCLENBQUM7SUFDaEgsMkJBQW1CLENBQUMsaUJBQWlCLEVBQUU7UUFDckM7WUFDRSxFQUFFLEVBQUUsS0FBSztZQUNULE1BQU0sRUFBRSx5SUFBeUk7U0FDbEo7UUFDRDtZQUNFLEVBQUUsRUFBRSxLQUFLO1lBQ1QsTUFBTSxFQUFFLDZFQUE2RTtTQUN0RjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUM7QUFFRCxTQUFnQixzQkFBc0IsQ0FDcEMsS0FBZ0IsRUFDaEIsS0FBa0M7O0lBRWxDLGdDQUFnQztJQUNoQyxJQUFJLHNCQUFzQixDQUFDO0lBQzNCLElBQUksV0FBVyxDQUFDO0lBQ2hCLElBQUksYUFBYSxDQUFDO0lBQ2xCLElBQUksUUFBZ0IsQ0FBQztJQUNyQixJQUFJLFFBQWdCLENBQUM7SUFFckIsMENBQTBDO0lBQzFDLElBQUksQ0FBQyxLQUFLLENBQUMsbUJBQW1CLEVBQUU7UUFDOUIsSUFDRSxDQUFDLE9BQUEsS0FBSyxDQUFDLHNCQUFzQiwwQ0FBRSxRQUFRLEtBQUksT0FBQSxLQUFLLENBQUMsc0JBQXNCLDBDQUFFLGdCQUFnQixNQUFLLFNBQVMsQ0FBQztZQUN4RyxDQUFDLE9BQUEsS0FBSyxDQUFDLHNCQUFzQiwwQ0FBRSxRQUFRLE1BQUssU0FBUyxXQUFJLEtBQUssQ0FBQyxzQkFBc0IsMENBQUUsZ0JBQWdCLENBQUEsQ0FBQyxFQUN4RztZQUNBLE1BQU0sSUFBSSxLQUFLLENBQUMsOEZBQThGLENBQUMsQ0FBQztTQUNqSDtRQUVELGNBQWMsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFM0IsSUFBSSxPQUFBLEtBQUssQ0FBQyxzQkFBc0IsMENBQUUsUUFBUSxNQUFLLFNBQVMsRUFBRTtZQUN4RCxRQUFRLEdBQUcsK0JBQWtCLENBQUMsS0FBSyxDQUFDLENBQUMsS0FBSyxDQUFDO1NBQzVDO2FBQU07WUFDTCxRQUFRLEdBQUcsS0FBSyxDQUFDLHNCQUFzQixDQUFDLFFBQVEsQ0FBQztTQUNsRDtRQUVELElBQUksS0FBSyxDQUFDLGVBQWUsS0FBSyxTQUFTLElBQUksS0FBSyxDQUFDLGVBQWUsRUFBRTtZQUNoRSxJQUNFLE9BQUEsS0FBSyxDQUFDLHNCQUFzQiwwQ0FBRSxRQUFRLE1BQUssU0FBUztnQkFDcEQsT0FBQSxLQUFLLENBQUMsc0JBQXNCLDBDQUFFLGdCQUFnQixNQUFLLFNBQVMsRUFDNUQ7Z0JBQ0EsV0FBVyxHQUFHLHFCQUFRLENBQUMsS0FBSyxFQUFFO29CQUM1QixlQUFlLEVBQUUsMkNBQTRCLEVBQUU7aUJBQ2hELENBQUMsQ0FBQztnQkFDSCxhQUFhLEdBQUcsMENBQWtCLENBQ2hDLEtBQUssRUFDTCxlQUFlLEVBQ2Y7b0JBQ0UsR0FBRyxFQUFFLFdBQVc7b0JBQ2hCLGdCQUFnQixFQUFFLEtBQUs7aUJBQ3hCLEVBQ0QsRUFBRSxFQUNGLENBQUMsRUFBRSxJQUFJLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsRUFBRSxVQUFVLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxDQUM5RCxDQUFDO2dCQUVGLFFBQVEsR0FBRyxXQUFXLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQztnQkFFbEQsc0JBQXNCLEdBQUcsa0RBQTZCLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsUUFBUSxFQUFFLFFBQVEsRUFBRTtvQkFDN0YsYUFBYSxDQUFDLGVBQWU7aUJBQzlCLENBQUMsQ0FBQzthQUNKO2lCQUFNO2dCQUNMLHNCQUFzQixHQUFHLGtEQUE2QixDQUNwRCxLQUFLLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFDbEIsUUFBUSxRQUNSLEtBQUssQ0FBQyxzQkFBc0IsMENBQUUsUUFBUSxRQUN0QyxLQUFLLENBQUMsc0JBQXNCLDBDQUFFLGdCQUFnQixDQUMvQyxDQUFDO2FBQ0g7U0FDRjthQUFNO1lBQ0wsc0JBQXNCLEdBQUcsa0RBQTZCLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsUUFBUSxDQUFDLENBQUM7U0FDdEY7UUFFRCxJQUFJLEtBQUssQ0FBQyxzQkFBc0IsRUFBRTtZQUNoQyxzQkFBc0IsR0FBRyxxQkFBYSxDQUFDLHNCQUFzQixFQUFFLEtBQUssQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1NBQzlGO1FBRUQsc0JBQXNCO1FBQ3RCLE1BQU0saUJBQWlCLEdBQWtDLElBQUksU0FBUyxDQUFDLG1CQUFtQixDQUN4RixLQUFLLEVBQ0wsbUJBQW1CLEVBQ25CLHNCQUFzQixDQUN2QixDQUFDO1FBQ0YsSUFBSSxXQUFXLEVBQUU7WUFDZixPQUFPLENBQUMsaUJBQWlCLEVBQUUsV0FBVyxFQUFFLGFBQWEsQ0FBQyxDQUFDO1NBQ3hEO2FBQU07WUFDTCxPQUFPLENBQUMsaUJBQWlCLENBQUMsQ0FBQztTQUM1QjtLQUNGO1NBQU07UUFDTCxrQ0FBa0M7UUFDbEMsT0FBTyxDQUFDLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0tBQ3BDO0FBQ0gsQ0FBQztBQW5GRCx3REFtRkM7QUFlRCxTQUFnQixzQkFBc0IsQ0FDcEMsS0FBZ0IsRUFDaEIsS0FBa0M7SUFFbEMsOENBQThDO0lBQzlDLElBQUksQ0FBQyxLQUFLLENBQUMsNEJBQTRCLEVBQUU7UUFDdkMsSUFBSSxLQUFLLENBQUMsVUFBVSxFQUFFO1lBQ3BCLCtDQUErQztZQUMvQyxPQUFPLHVCQUF1QixDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQztTQUM5QzthQUFNO1lBQ0wsTUFBTSxLQUFLLENBQUMsd0VBQXdFLENBQUMsQ0FBQztTQUN2RjtLQUNGO1NBQU07UUFDTCxtQ0FBbUM7UUFDbkMsT0FBTyxDQUFDLEtBQUssQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO0tBQzdDO0FBQ0gsQ0FBQztBQWhCRCx3REFnQkM7QUFFRCxTQUFnQix1QkFBdUIsQ0FDckMsS0FBZ0IsRUFDaEIsS0FBa0M7SUFFbEMsSUFBSSxLQUF5QixDQUFDO0lBQzlCLElBQUksY0FBMkMsQ0FBQztJQUNoRCxJQUFJLFFBQStCLENBQUM7SUFDcEMsSUFBSSxhQUF1QixDQUFDO0lBRTVCLHlEQUF5RDtJQUN6RCxJQUFJLEtBQUssQ0FBQyxVQUFVLEVBQUU7UUFDcEIsb0RBQW9EO1FBQ3BELElBQUksS0FBSyxDQUFDLFVBQVUsQ0FBQyxnQkFBZ0IsRUFBRTtZQUNyQyxhQUFhLEdBQUcsR0FBRyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQ2xDLEtBQUssRUFDTCx1QkFBdUIsRUFDdkIsS0FBSyxDQUFDLFVBQVUsQ0FBQyxnQkFBZ0IsQ0FDdEIsQ0FBQztTQUNmO2FBQU07WUFDTCw0QkFBNEI7WUFDNUIsYUFBYSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsZUFBZSxFQUFFO2dCQUNuRCxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMseUJBQXlCLENBQUM7YUFDL0QsQ0FBQyxDQUFDO1lBQ0gsMkJBQTJCO1lBQzNCLGNBQWMsQ0FBQyxhQUFhLEVBQUUsS0FBSyxDQUFDLENBQUM7U0FDdEM7UUFFRCx5QkFBeUI7UUFDekIsS0FBSyxHQUFHLG9CQUFvQixDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsVUFBVSxFQUFFLGFBQWEsRUFBRSxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDaEYsa0NBQWtDO1FBQ2xDLGNBQWMsR0FBRyw2QkFBNkIsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLGFBQWEsRUFBRSxLQUFLLENBQUMsbUJBQW1CLENBQUMsQ0FBQztRQUN0RywwQkFBMEI7UUFDMUIsY0FBYyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNuQyw0QkFBNEI7UUFDNUIsUUFBUSxHQUFHLHVCQUF1QixDQUFDLEtBQUssRUFBRSxjQUFjLENBQUMsc0JBQXNCLEVBQUUsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBQ3RHLG1DQUFtQztRQUNuQyxRQUFRLENBQUMsWUFBWSxDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBRXRDLE9BQU8sQ0FBQyxRQUFRLEVBQUUsY0FBYyxFQUFFLEtBQUssQ0FBQyxDQUFDO0tBQzFDO1NBQU07UUFDTCxNQUFNLEtBQUssQ0FBQyxzRUFBc0UsQ0FBQyxDQUFDO0tBQ3JGO0FBQ0gsQ0FBQztBQTFDRCwwREEwQ0M7QUFFRCxTQUFnQixvQkFBb0IsQ0FDbEMsS0FBZ0IsRUFDaEIsVUFBbUMsRUFDbkMsSUFBYyxFQUNkLEdBQWM7SUFFZCxJQUFJLGVBQXdDLENBQUM7SUFDN0MsSUFBSSxnQkFBZ0UsQ0FBQztJQUNyRSxJQUFJLFNBQTJELENBQUM7SUFDaEUsSUFBSSxLQUF5QixDQUFDO0lBRTlCLElBQUksR0FBRyxFQUFFO1FBQ1AsTUFBTSx5QkFBeUIsR0FBRyxJQUFJLEdBQUcsQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFLGtDQUFrQyxFQUFFO1lBQ2pHLEdBQUc7WUFDSCxnQkFBZ0IsRUFBRSxJQUFJO1NBQ3ZCLENBQUMsQ0FBQztRQUVILDBDQUEwQztRQUMxQyx5QkFBeUIsQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFFN0YsTUFBTSxnQkFBZ0IsR0FBRyx5QkFBeUIsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBeUIsQ0FBQztRQUN0RywyQkFBbUIsQ0FBQyxnQkFBZ0IsRUFBRTtZQUNwQztnQkFDRSxFQUFFLEVBQUUsSUFBSTtnQkFDUixNQUFNLEVBQUUsNERBQTREO2FBQ3JFO1lBQ0Q7Z0JBQ0UsRUFBRSxFQUFFLEtBQUs7Z0JBQ1QsTUFBTSxFQUFFLGdFQUFnRTthQUN6RTtTQUNGLENBQUMsQ0FBQztRQUVILHlFQUF5RTtRQUN6RSxJQUFJLEdBQUcsQ0FBQyxjQUFjLENBQUMsTUFBTSxLQUFLLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDdkUsTUFBTSxLQUFLLENBQUMsd0ZBQXdGLENBQUMsQ0FBQztTQUN2RztRQUVELFNBQVMsR0FBRztZQUNWLHFGQUFxRjtZQUNyRix5R0FBeUc7WUFDekcsT0FBTyxFQUFFLEdBQUcsQ0FBQyxhQUFhLENBQUM7Z0JBQ3pCLFFBQVEsRUFBRSxJQUFJO2FBQ2YsQ0FBQyxDQUFDLFNBQVM7WUFDWixnQkFBZ0IsRUFBRSxDQUFDLHlCQUF5QixDQUFDLGVBQWUsQ0FBQztTQUM5RCxDQUFDO0tBQ0g7SUFFRCxJQUFJLFVBQVUsQ0FBQyxnQkFBZ0IsRUFBRTtRQUMvQiw4Q0FBOEM7UUFDOUMsZ0JBQWdCLEdBQUcsVUFBVSxDQUFDLGdCQUFrRSxDQUFDO1FBQ2pHLDBCQUEwQjtRQUMxQixlQUFlLEdBQUcsK0NBQTBCLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxnQkFBZ0IsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUN4RixvQ0FBb0M7UUFDcEMsZUFBZSxHQUFHLHFCQUFhLENBQUMsZUFBZSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBRTdELCtCQUErQjtRQUMvQixLQUFLLEdBQUcsSUFBSSxTQUFTLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxnQkFBZ0IsRUFBRSxlQUFlLENBQUMsQ0FBQztRQUN6RSx5Q0FBeUM7UUFDekMsS0FBSyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFL0IsT0FBTyxLQUFLLENBQUM7S0FDZDtTQUFNO1FBQ0wsTUFBTSxLQUFLLENBQUMseUVBQXlFLENBQUMsQ0FBQztLQUN4RjtBQUNILENBQUM7QUFoRUQsb0RBZ0VDO0FBRUQsU0FBZ0IsNkJBQTZCLENBQzNDLEtBQWdCLEVBQ2hCLFNBQWlCLEVBQ2pCLG1CQUFzRDtJQUV0RCxJQUFJLHdCQUEwRCxDQUFDO0lBQy9ELElBQUksUUFBZ0IsQ0FBQztJQUNyQixJQUFJLGNBQTJDLENBQUM7SUFFaEQsK0NBQStDO0lBQy9DLElBQUksbUJBQW1CLElBQUksbUJBQW1CLENBQUMsUUFBUSxFQUFFO1FBQ3ZELFFBQVEsR0FBRyxtQkFBbUIsQ0FBQyxRQUFRLENBQUM7S0FDekM7U0FBTTtRQUNMLFFBQVEsR0FBRywrQkFBa0IsQ0FBQyxLQUFLLENBQUMsQ0FBQyxLQUFLLENBQUM7S0FDNUM7SUFDRCxtQ0FBbUM7SUFDbkMsd0JBQXdCLEdBQUcsd0RBQW1DLENBQUMsU0FBUyxFQUFFLFFBQVEsQ0FBQyxDQUFDO0lBQ3BGLDhDQUE4QztJQUM5QyxJQUFJLG1CQUFtQixFQUFFO1FBQ3ZCLHdCQUF3QixHQUFHLHFCQUFhLENBQUMsd0JBQXdCLEVBQUUsbUJBQW1CLENBQUMsQ0FBQztLQUN6RjtJQUVELHdDQUF3QztJQUN4QyxjQUFjLEdBQUcsSUFBSSxTQUFTLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLHlCQUF5QixFQUFFLHdCQUF3QixDQUFDLENBQUM7SUFFN0csT0FBTyxjQUFjLENBQUM7QUFDeEIsQ0FBQztBQTFCRCxzRUEwQkM7QUFFRCxTQUFnQix1QkFBdUIsQ0FDckMsS0FBZ0IsRUFDaEIsa0JBQTBCLEVBQzFCLGFBQTBDO0lBRTFDLElBQUksa0JBQThDLENBQUM7SUFDbkQsSUFBSSxRQUErQixDQUFDO0lBRXBDLDZCQUE2QjtJQUM3QixrQkFBa0IsR0FBRyxrREFBNkIsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3ZFLHdDQUF3QztJQUN4QyxJQUFJLGFBQWEsRUFBRTtRQUNqQixrQkFBa0IsR0FBRyxxQkFBYSxDQUFDLGtCQUFrQixFQUFFLGFBQWEsQ0FBQyxDQUFDO0tBQ3ZFO0lBRUQsa0NBQWtDO0lBQ2xDLFFBQVEsR0FBRyxJQUFJLFNBQVMsQ0FBQyxXQUFXLENBQUMsS0FBSyxFQUFFLG1CQUFtQixFQUFFLGtCQUFrQixDQUFDLENBQUM7SUFFckYsT0FBTyxRQUFRLENBQUM7QUFDbEIsQ0FBQztBQW5CRCwwREFtQkMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqICBDb3B5cmlnaHQgMjAyMSBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqICBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLiBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlXG4gKiAgd2l0aCB0aGUgTGljZW5zZS4gQSBjb3B5IG9mIHRoZSBMaWNlbnNlIGlzIGxvY2F0ZWQgYXRcbiAqXG4gKiAgICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqICBvciBpbiB0aGUgJ2xpY2Vuc2UnIGZpbGUgYWNjb21wYW55aW5nIHRoaXMgZmlsZS4gVGhpcyBmaWxlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuICdBUyBJUycgQkFTSVMsIFdJVEhPVVQgV0FSUkFOVElFU1xuICogIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGV4cHJlc3Mgb3IgaW1wbGllZC4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zXG4gKiAgYW5kIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICovXG5cbmltcG9ydCAqIGFzIHNhZ2VtYWtlciBmcm9tICdAYXdzLWNkay9hd3Mtc2FnZW1ha2VyJztcbmltcG9ydCAqIGFzIGVjMiBmcm9tICdAYXdzLWNkay9hd3MtZWMyJztcbmltcG9ydCB7IGJ1aWxkRW5jcnlwdGlvbktleSB9IGZyb20gJy4va21zLWhlbHBlcic7XG5pbXBvcnQge1xuICBEZWZhdWx0U2FnZW1ha2VyTm90ZWJvb2tQcm9wcyxcbiAgRGVmYXVsdFNhZ2VtYWtlck1vZGVsUHJvcHMsXG4gIERlZmF1bHRTYWdlbWFrZXJFbmRwb2ludENvbmZpZ1Byb3BzLFxuICBEZWZhdWx0U2FnZW1ha2VyRW5kcG9pbnRQcm9wcyxcbn0gZnJvbSAnLi9zYWdlbWFrZXItZGVmYXVsdHMnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgb3ZlcnJpZGVQcm9wcywgYWRkQ2ZuU3VwcHJlc3NSdWxlcyB9IGZyb20gJy4vdXRpbHMnO1xuaW1wb3J0IHsgYnVpbGRWcGMgfSBmcm9tICcuL3ZwYy1oZWxwZXInO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0IHsgQXdzIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBEZWZhdWx0UHVibGljUHJpdmF0ZVZwY1Byb3BzIH0gZnJvbSAnLi92cGMtZGVmYXVsdHMnO1xuaW1wb3J0IHsgYnVpbGRTZWN1cml0eUdyb3VwIH0gZnJvbSAnLi9zZWN1cml0eS1ncm91cC1oZWxwZXInO1xuLy8gTm90ZTogVG8gZW5zdXJlIENES3YyIGNvbXBhdGliaWxpdHksIGtlZXAgdGhlIGltcG9ydCBzdGF0ZW1lbnQgZm9yIENvbnN0cnVjdCBzZXBhcmF0ZVxuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5cbmV4cG9ydCBpbnRlcmZhY2UgQnVpbGRTYWdlbWFrZXJOb3RlYm9va1Byb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc2FnZW1ha2VyTm90ZWJvb2tQcm9wcz86IHNhZ2VtYWtlci5DZm5Ob3RlYm9va0luc3RhbmNlUHJvcHMgfCBhbnk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBkZXBsb3lJbnNpZGVWcGM/OiBib29sZWFuO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGV4aXN0aW5nTm90ZWJvb2tPYmo/OiBzYWdlbWFrZXIuQ2ZuTm90ZWJvb2tJbnN0YW5jZTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHJvbGU6IGlhbS5Sb2xlO1xufVxuXG5mdW5jdGlvbiBhZGRQZXJtaXNzaW9ucyhfcm9sZTogaWFtLlJvbGUsIHByb3BzPzogQnVpbGRTYWdlbWFrZXJFbmRwb2ludFByb3BzKSB7XG4gIC8vIEdyYW50IHBlcm1pc3Npb25zIHRvIE5vdGVCb29rSW5zdGFuY2UgZm9yIGNyZWF0aW5nIGFuZCB0cmFpbmluZyB0aGUgbW9kZWxcbiAgX3JvbGUuYWRkVG9Qb2xpY3koXG4gICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgcmVzb3VyY2VzOiBbYGFybjoke0F3cy5QQVJUSVRJT059OnNhZ2VtYWtlcjoke0F3cy5SRUdJT059OiR7QXdzLkFDQ09VTlRfSUR9OipgXSxcbiAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgJ3NhZ2VtYWtlcjpDcmVhdGVUcmFpbmluZ0pvYicsXG4gICAgICAgICdzYWdlbWFrZXI6RGVzY3JpYmVUcmFpbmluZ0pvYicsXG4gICAgICAgICdzYWdlbWFrZXI6Q3JlYXRlTW9kZWwnLFxuICAgICAgICAnc2FnZW1ha2VyOkRlc2NyaWJlTW9kZWwnLFxuICAgICAgICAnc2FnZW1ha2VyOkRlbGV0ZU1vZGVsJyxcbiAgICAgICAgJ3NhZ2VtYWtlcjpDcmVhdGVFbmRwb2ludCcsXG4gICAgICAgICdzYWdlbWFrZXI6Q3JlYXRlRW5kcG9pbnRDb25maWcnLFxuICAgICAgICAnc2FnZW1ha2VyOkRlc2NyaWJlRW5kcG9pbnQnLFxuICAgICAgICAnc2FnZW1ha2VyOkRlc2NyaWJlRW5kcG9pbnRDb25maWcnLFxuICAgICAgICAnc2FnZW1ha2VyOkRlbGV0ZUVuZHBvaW50JyxcbiAgICAgICAgJ3NhZ2VtYWtlcjpEZWxldGVFbmRwb2ludENvbmZpZycsXG4gICAgICAgICdzYWdlbWFrZXI6SW52b2tlRW5kcG9pbnQnLFxuICAgICAgXSxcbiAgICB9KVxuICApO1xuXG4gIC8vIEdyYW50IENsb3VkV2F0Y2ggTG9nZ2luZyBwZXJtaXNzaW9uc1xuICBfcm9sZS5hZGRUb1BvbGljeShcbiAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICByZXNvdXJjZXM6IFtgYXJuOiR7Y2RrLkF3cy5QQVJUSVRJT059OmxvZ3M6JHtjZGsuQXdzLlJFR0lPTn06JHtjZGsuQXdzLkFDQ09VTlRfSUR9OmxvZy1ncm91cDovYXdzL3NhZ2VtYWtlci8qYF0sXG4gICAgICBhY3Rpb25zOiBbXG4gICAgICAgICdsb2dzOkNyZWF0ZUxvZ0dyb3VwJyxcbiAgICAgICAgJ2xvZ3M6Q3JlYXRlTG9nU3RyZWFtJyxcbiAgICAgICAgJ2xvZ3M6RGVzY3JpYmVMb2dTdHJlYW1zJyxcbiAgICAgICAgJ2xvZ3M6R2V0TG9nRXZlbnRzJyxcbiAgICAgICAgJ2xvZ3M6UHV0TG9nRXZlbnRzJyxcbiAgICAgIF0sXG4gICAgfSlcbiAgKTtcblxuICAvLyBUbyBwbGFjZSB0aGUgU2FnZW1ha2VyIGVuZHBvaW50IGluIGEgVlBDXG4gIGlmIChwcm9wcyAmJiBwcm9wcy52cGMpIHtcbiAgICBfcm9sZS5hZGRUb1BvbGljeShcbiAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICdlYzI6Q3JlYXRlTmV0d29ya0ludGVyZmFjZScsXG4gICAgICAgICAgJ2VjMjpDcmVhdGVOZXR3b3JrSW50ZXJmYWNlUGVybWlzc2lvbicsXG4gICAgICAgICAgJ2VjMjpEZWxldGVOZXR3b3JrSW50ZXJmYWNlJyxcbiAgICAgICAgICAnZWMyOkRlbGV0ZU5ldHdvcmtJbnRlcmZhY2VQZXJtaXNzaW9uJyxcbiAgICAgICAgICAnZWMyOkRlc2NyaWJlTmV0d29ya0ludGVyZmFjZXMnLFxuICAgICAgICAgICdlYzI6QXNzaWduUHJpdmF0ZUlwQWRkcmVzc2VzJyxcbiAgICAgICAgICAnZWMyOlVuYXNzaWduUHJpdmF0ZUlwQWRkcmVzc2VzJyxcbiAgICAgICAgICAnZWMyOkRlc2NyaWJlVnBjcycsXG4gICAgICAgICAgJ2VjMjpEZXNjcmliZURoY3BPcHRpb25zJyxcbiAgICAgICAgICAnZWMyOkRlc2NyaWJlU3VibmV0cycsXG4gICAgICAgICAgJ2VjMjpEZXNjcmliZVNlY3VyaXR5R3JvdXBzJyxcbiAgICAgICAgXSxcbiAgICAgIH0pXG4gICAgKTtcbiAgfVxuXG4gIC8vIFRvIGNyZWF0ZSBhIFNhZ2VtYWtlciBtb2RlbCB1c2luZyBCcmluZy1Zb3VyLU93bi1Nb2RlbCAoQllPTSkgYWxnb3JpdGggaW1hZ2VcbiAgLy8gVGhlIGltYWdlIFVSTCBpcyBzcGVjaWZpZWQgaW4gdGhlIG1vZGVsUHJvcHNcbiAgX3JvbGUuYWRkVG9Qb2xpY3koXG4gICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgcmVzb3VyY2VzOiBbYGFybjoke2Nkay5Bd3MuUEFSVElUSU9OfTplY3I6JHtjZGsuQXdzLlJFR0lPTn06JHtjZGsuQXdzLkFDQ09VTlRfSUR9OnJlcG9zaXRvcnkvKmBdLFxuICAgICAgYWN0aW9uczogW1xuICAgICAgICAnZWNyOkJhdGNoQ2hlY2tMYXllckF2YWlsYWJpbGl0eScsXG4gICAgICAgICdlY3I6R2V0RG93bmxvYWRVcmxGb3JMYXllcicsXG4gICAgICAgICdlY3I6RGVzY3JpYmVSZXBvc2l0b3JpZXMnLFxuICAgICAgICAnZWNyOkRlc2NyaWJlSW1hZ2VzJyxcbiAgICAgICAgJ2VjcjpCYXRjaEdldEltYWdlJyxcbiAgICAgIF0sXG4gICAgfSlcbiAgKTtcblxuICAvLyBBZGQgR2V0QXV0aG9yaXphdGlvblRva2VuIChpdCBjYW4gbm90IGJlIGJvdW5kIHRvIHJlc291cmNlcyBvdGhlciB0aGFuICopXG4gIF9yb2xlLmFkZFRvUG9saWN5KFxuICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgICBhY3Rpb25zOiBbJ2VjcjpHZXRBdXRob3JpemF0aW9uVG9rZW4nXSxcbiAgICB9KVxuICApO1xuXG4gIC8vIGFkZCBwZXJtaXNzaW9uIHRvIHVzZSBFbGFzdGljIEluZmVyZW5jZSBhY2NlbGVyYXRvclxuICBpZiAocHJvcHMgJiYgcHJvcHMuZW5kcG9pbnRDb25maWdQcm9wcykge1xuICAgIC8vIEdldCB0aGUgYWNjZWxlcmF0b3JUeXBlLCBpZiBhbnlcbiAgICBjb25zdCBhY2NlbGVyYXRvclR5cGUgPSAocHJvcHMuZW5kcG9pbnRDb25maWdQcm9wc1xuICAgICAgPy5wcm9kdWN0aW9uVmFyaWFudHMgYXMgc2FnZW1ha2VyLkNmbkVuZHBvaW50Q29uZmlnLlByb2R1Y3Rpb25WYXJpYW50UHJvcGVydHlbXSlbMF0uYWNjZWxlcmF0b3JUeXBlO1xuICAgIGlmIChhY2NlbGVyYXRvclR5cGUgIT09IHVuZGVmaW5lZCkge1xuICAgICAgX3JvbGUuYWRkVG9Qb2xpY3koXG4gICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgICAgICAgIGFjdGlvbnM6IFsnZWxhc3RpYy1pbmZlcmVuY2U6Q29ubmVjdCddLFxuICAgICAgICB9KVxuICAgICAgKTtcbiAgICB9XG4gIH1cblxuICAvLyBhZGQga21zIHBlcm1pc3Npb25zXG4gIF9yb2xlLmFkZFRvUG9saWN5KFxuICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIC8vIHRoZSBrbXNLZXlJZCBpbiB0aGUgZW5kcG9pbnRDb25maWdQcm9wcyBjYW4gYmUgYW55IG9mIHRoZSBmb2xsb3dpbmcgZm9ybWF0czpcbiAgICAgIC8vIEtleSBJRDogMTIzNGFiY2QtMTJhYi0zNGNkLTU2ZWYtMTIzNDU2Nzg5MGFiXG4gICAgICAvLyBLZXkgQVJOOiBhcm46YXdzOmttczo8cmVnaW9uPjo8YWNjb3VudElEPjprZXkvMTIzNGFiY2QtMTJhYi0zNGNkLTU2ZWYtMTIzNDU2Nzg5MGFiXG4gICAgICAvLyBBbGlhcyBuYW1lOiBhbGlhcy9FeGFtcGxlQWxpYXNcbiAgICAgIC8vIEFsaWFzIG5hbWUgQVJOOiBhcm46YXdzOmttczo8cmVnaW9uPjo8YWNjb3VudElEPjphbGlhcy9FeGFtcGxlQWxpYXNcbiAgICAgIC8vIHRoZSBrZXkgaXMgdXNlZCB0byBlbmNyeXB0L2RlY3J5cHQgZGF0YSBjYXB0dXJlZCBieSB0aGUgU2FnZW1ha2VyIGVuZHBvaW50IGFuZCBzdG9yZWQgaW4gUzMgYnVja2V0XG4gICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgYGFybjoke2Nkay5Bd3MuUEFSVElUSU9OfTprbXM6JHtjZGsuQXdzLlJFR0lPTn06JHtjZGsuQXdzLkFDQ09VTlRfSUR9OmtleS8qYCxcbiAgICAgICAgYGFybjoke2Nkay5Bd3MuUEFSVElUSU9OfTprbXM6JHtjZGsuQXdzLlJFR0lPTn06JHtjZGsuQXdzLkFDQ09VTlRfSUR9OmFsaWFzLypgLFxuICAgICAgXSxcbiAgICAgIGFjdGlvbnM6IFsna21zOkVuY3J5cHQnLCAna21zOkRlY3J5cHQnLCAna21zOlJlRW5jcnlwdConLCAna21zOkdlbmVyYXRlRGF0YUtleSonLCAna21zOkRlc2NyaWJlS2V5J10sXG4gICAgfSlcbiAgKTtcblxuICAvLyBBZGQgUzMgcGVybWlzc2lvbnMgdG8gZ2V0IE1vZGVsIGFydGlmYWN0LCBwdXQgZGF0YSBjYXB0dXJlIGZpbGVzLCBldGMuXG4gIF9yb2xlLmFkZFRvUG9saWN5KFxuICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGFjdGlvbnM6IFsnczM6R2V0T2JqZWN0JywgJ3MzOlB1dE9iamVjdCcsICdzMzpEZWxldGVPYmplY3QnLCAnczM6TGlzdEJ1Y2tldCddLFxuICAgICAgcmVzb3VyY2VzOiBbJ2Fybjphd3M6czM6OjoqJ10sXG4gICAgfSlcbiAgKTtcblxuICAvLyBHcmFudCBHZXRSb2xlIHBlcm1pc3Npb25zIHRvIHRoZSBTYWdlbWFrZXIgc2VydmljZVxuICBfcm9sZS5hZGRUb1BvbGljeShcbiAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICByZXNvdXJjZXM6IFtfcm9sZS5yb2xlQXJuXSxcbiAgICAgIGFjdGlvbnM6IFsnaWFtOkdldFJvbGUnXSxcbiAgICB9KVxuICApO1xuXG4gIC8vIEdyYW50IFBhc3NSb2xlIHBlcm1pc3Npb25zIHRvIHRoZSBTYWdlbWFrZXIgc2VydmljZVxuICBfcm9sZS5hZGRUb1BvbGljeShcbiAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICByZXNvdXJjZXM6IFtfcm9sZS5yb2xlQXJuXSxcbiAgICAgIGFjdGlvbnM6IFsnaWFtOlBhc3NSb2xlJ10sXG4gICAgICBjb25kaXRpb25zOiB7XG4gICAgICAgIFN0cmluZ0xpa2U6IHsgJ2lhbTpQYXNzZWRUb1NlcnZpY2UnOiAnc2FnZW1ha2VyLmFtYXpvbmF3cy5jb20nIH0sXG4gICAgICB9LFxuICAgIH0pXG4gICk7XG5cbiAgLy8gQWRkIENGTiBOQUcgdXBwcmVzcyB0byBhbGxvdyBmb3IgXCJSZXNvdXJjZVwiOiBcIipcIiBmb3IgRU5JIGFjY2VzcyBpbiBWUEMsXG4gIC8vIEVDUiBhdXRob3JpemF0aW9uIHRva2VuIGZvciBjdXN0b20gbW9kZWwgaW1hZ2VzLCBhbmQgZWxhc3RpYyBpbmZlcmVuY2VcbiAgLy8gQWRkIENGTiBOQUcgZm9yIENvbXBsZXggUm9sZSBiZWNhdXNlIFNhZ21ha2VyIG5lZWRzIHBlcm1pc3Npb25zIHRvIGFjY2VzcyBzZXZlcmFsIHNlcnZpY2VzXG4gIGNvbnN0IHJvbGVEZWZhdWx0UG9saWN5ID0gX3JvbGUubm9kZS50cnlGaW5kQ2hpbGQoJ0RlZmF1bHRQb2xpY3knKT8ubm9kZS5maW5kQ2hpbGQoJ1Jlc291cmNlJykgYXMgaWFtLkNmblBvbGljeTtcbiAgYWRkQ2ZuU3VwcHJlc3NSdWxlcyhyb2xlRGVmYXVsdFBvbGljeSwgW1xuICAgIHtcbiAgICAgIGlkOiAnVzEyJyxcbiAgICAgIHJlYXNvbjogYFNhZ2VtYWtlciBuZWVkcyB0aGUgZm9sbG93aW5nIG1pbmltdW0gcmVxdWlyZWQgcGVybWlzc2lvbnMgdG8gYWNjZXNzIEVOSXMgaW4gYSBWUEMsIEVDUiBmb3IgY3VzdG9tIG1vZGVsIGltYWdlcywgYW5kIGVsYXN0aWMgaW5mZXJlbmNlLmAsXG4gICAgfSxcbiAgICB7XG4gICAgICBpZDogJ1c3NicsXG4gICAgICByZWFzb246ICdDb21wbGV4IHJvbGUgYmVjdWFzZSBTYWdlbWFrZXIgbmVlZHMgcGVybWlzc2lvbnMgdG8gYWNjZXNzIHNldmVyYWwgc2VydmljZXMnLFxuICAgIH1cbiAgXSk7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBidWlsZFNhZ2VtYWtlck5vdGVib29rKFxuICBzY29wZTogQ29uc3RydWN0LFxuICBwcm9wczogQnVpbGRTYWdlbWFrZXJOb3RlYm9va1Byb3BzXG4pOiBbc2FnZW1ha2VyLkNmbk5vdGVib29rSW5zdGFuY2UsIGVjMi5JVnBjPywgZWMyLlNlY3VyaXR5R3JvdXA/XSB7XG4gIC8vIFNldHVwIHRoZSBub3RlYm9vayBwcm9wZXJ0aWVzXG4gIGxldCBzYWdlbWFrZXJOb3RlYm9va1Byb3BzO1xuICBsZXQgdnBjSW5zdGFuY2U7XG4gIGxldCBzZWN1cml0eUdyb3VwO1xuICBsZXQga21zS2V5SWQ6IHN0cmluZztcbiAgbGV0IHN1Ym5ldElkOiBzdHJpbmc7XG5cbiAgLy8gQ29uZGl0aW9uYWwgU2FnZW1ha2VyIE5vdGVib29rIGNyZWF0aW9uXG4gIGlmICghcHJvcHMuZXhpc3RpbmdOb3RlYm9va09iaikge1xuICAgIGlmIChcbiAgICAgIChwcm9wcy5zYWdlbWFrZXJOb3RlYm9va1Byb3BzPy5zdWJuZXRJZCAmJiBwcm9wcy5zYWdlbWFrZXJOb3RlYm9va1Byb3BzPy5zZWN1cml0eUdyb3VwSWRzID09PSB1bmRlZmluZWQpIHx8XG4gICAgICAocHJvcHMuc2FnZW1ha2VyTm90ZWJvb2tQcm9wcz8uc3VibmV0SWQgPT09IHVuZGVmaW5lZCAmJiBwcm9wcy5zYWdlbWFrZXJOb3RlYm9va1Byb3BzPy5zZWN1cml0eUdyb3VwSWRzKVxuICAgICkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdNdXN0IGRlZmluZSBib3RoIHNhZ2VtYWtlck5vdGVib29rUHJvcHMuc3VibmV0SWQgYW5kIHNhZ2VtYWtlck5vdGVib29rUHJvcHMuc2VjdXJpdHlHcm91cElkcycpO1xuICAgIH1cblxuICAgIGFkZFBlcm1pc3Npb25zKHByb3BzLnJvbGUpO1xuXG4gICAgaWYgKHByb3BzLnNhZ2VtYWtlck5vdGVib29rUHJvcHM/Lmttc0tleUlkID09PSB1bmRlZmluZWQpIHtcbiAgICAgIGttc0tleUlkID0gYnVpbGRFbmNyeXB0aW9uS2V5KHNjb3BlKS5rZXlJZDtcbiAgICB9IGVsc2Uge1xuICAgICAga21zS2V5SWQgPSBwcm9wcy5zYWdlbWFrZXJOb3RlYm9va1Byb3BzLmttc0tleUlkO1xuICAgIH1cblxuICAgIGlmIChwcm9wcy5kZXBsb3lJbnNpZGVWcGMgPT09IHVuZGVmaW5lZCB8fCBwcm9wcy5kZXBsb3lJbnNpZGVWcGMpIHtcbiAgICAgIGlmIChcbiAgICAgICAgcHJvcHMuc2FnZW1ha2VyTm90ZWJvb2tQcm9wcz8uc3VibmV0SWQgPT09IHVuZGVmaW5lZCAmJlxuICAgICAgICBwcm9wcy5zYWdlbWFrZXJOb3RlYm9va1Byb3BzPy5zZWN1cml0eUdyb3VwSWRzID09PSB1bmRlZmluZWRcbiAgICAgICkge1xuICAgICAgICB2cGNJbnN0YW5jZSA9IGJ1aWxkVnBjKHNjb3BlLCB7XG4gICAgICAgICAgZGVmYXVsdFZwY1Byb3BzOiBEZWZhdWx0UHVibGljUHJpdmF0ZVZwY1Byb3BzKCksXG4gICAgICAgIH0pO1xuICAgICAgICBzZWN1cml0eUdyb3VwID0gYnVpbGRTZWN1cml0eUdyb3VwKFxuICAgICAgICAgIHNjb3BlLFxuICAgICAgICAgICdTZWN1cml0eUdyb3VwJyxcbiAgICAgICAgICB7XG4gICAgICAgICAgICB2cGM6IHZwY0luc3RhbmNlLFxuICAgICAgICAgICAgYWxsb3dBbGxPdXRib3VuZDogZmFsc2UsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBbXSxcbiAgICAgICAgICBbeyBwZWVyOiBlYzIuUGVlci5hbnlJcHY0KCksIGNvbm5lY3Rpb246IGVjMi5Qb3J0LnRjcCg0NDMpIH1dXG4gICAgICAgICk7XG5cbiAgICAgICAgc3VibmV0SWQgPSB2cGNJbnN0YW5jZS5wcml2YXRlU3VibmV0c1swXS5zdWJuZXRJZDtcblxuICAgICAgICBzYWdlbWFrZXJOb3RlYm9va1Byb3BzID0gRGVmYXVsdFNhZ2VtYWtlck5vdGVib29rUHJvcHMocHJvcHMucm9sZS5yb2xlQXJuLCBrbXNLZXlJZCwgc3VibmV0SWQsIFtcbiAgICAgICAgICBzZWN1cml0eUdyb3VwLnNlY3VyaXR5R3JvdXBJZCxcbiAgICAgICAgXSk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICBzYWdlbWFrZXJOb3RlYm9va1Byb3BzID0gRGVmYXVsdFNhZ2VtYWtlck5vdGVib29rUHJvcHMoXG4gICAgICAgICAgcHJvcHMucm9sZS5yb2xlQXJuLFxuICAgICAgICAgIGttc0tleUlkLFxuICAgICAgICAgIHByb3BzLnNhZ2VtYWtlck5vdGVib29rUHJvcHM/LnN1Ym5ldElkLFxuICAgICAgICAgIHByb3BzLnNhZ2VtYWtlck5vdGVib29rUHJvcHM/LnNlY3VyaXR5R3JvdXBJZHNcbiAgICAgICAgKTtcbiAgICAgIH1cbiAgICB9IGVsc2Uge1xuICAgICAgc2FnZW1ha2VyTm90ZWJvb2tQcm9wcyA9IERlZmF1bHRTYWdlbWFrZXJOb3RlYm9va1Byb3BzKHByb3BzLnJvbGUucm9sZUFybiwga21zS2V5SWQpO1xuICAgIH1cblxuICAgIGlmIChwcm9wcy5zYWdlbWFrZXJOb3RlYm9va1Byb3BzKSB7XG4gICAgICBzYWdlbWFrZXJOb3RlYm9va1Byb3BzID0gb3ZlcnJpZGVQcm9wcyhzYWdlbWFrZXJOb3RlYm9va1Byb3BzLCBwcm9wcy5zYWdlbWFrZXJOb3RlYm9va1Byb3BzKTtcbiAgICB9XG5cbiAgICAvLyBDcmVhdGUgdGhlIG5vdGVib29rXG4gICAgY29uc3Qgc2FnZW1ha2VySW5zdGFuY2U6IHNhZ2VtYWtlci5DZm5Ob3RlYm9va0luc3RhbmNlID0gbmV3IHNhZ2VtYWtlci5DZm5Ob3RlYm9va0luc3RhbmNlKFxuICAgICAgc2NvcGUsXG4gICAgICAnU2FnZW1ha2VyTm90ZWJvb2snLFxuICAgICAgc2FnZW1ha2VyTm90ZWJvb2tQcm9wc1xuICAgICk7XG4gICAgaWYgKHZwY0luc3RhbmNlKSB7XG4gICAgICByZXR1cm4gW3NhZ2VtYWtlckluc3RhbmNlLCB2cGNJbnN0YW5jZSwgc2VjdXJpdHlHcm91cF07XG4gICAgfSBlbHNlIHtcbiAgICAgIHJldHVybiBbc2FnZW1ha2VySW5zdGFuY2VdO1xuICAgIH1cbiAgfSBlbHNlIHtcbiAgICAvLyBSZXR1cm4gZXhpc3Rpbmcgbm90ZWJvb2sgb2JqZWN0XG4gICAgcmV0dXJuIFtwcm9wcy5leGlzdGluZ05vdGVib29rT2JqXTtcbiAgfVxufVxuXG5leHBvcnQgaW50ZXJmYWNlIEJ1aWxkU2FnZW1ha2VyRW5kcG9pbnRQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBleGlzdGluZ1NhZ2VtYWtlckVuZHBvaW50T2JqPzogc2FnZW1ha2VyLkNmbkVuZHBvaW50O1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBtb2RlbFByb3BzPzogc2FnZW1ha2VyLkNmbk1vZGVsUHJvcHMgfCBhbnk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZW5kcG9pbnRDb25maWdQcm9wcz86IHNhZ2VtYWtlci5DZm5FbmRwb2ludENvbmZpZ1Byb3BzO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBlbmRwb2ludFByb3BzPzogc2FnZW1ha2VyLkNmbkVuZHBvaW50UHJvcHM7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB2cGM/OiBlYzIuSVZwYztcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIEJ1aWxkU2FnZW1ha2VyRW5kcG9pbnQoXG4gIHNjb3BlOiBDb25zdHJ1Y3QsXG4gIHByb3BzOiBCdWlsZFNhZ2VtYWtlckVuZHBvaW50UHJvcHNcbik6IFtzYWdlbWFrZXIuQ2ZuRW5kcG9pbnQsIHNhZ2VtYWtlci5DZm5FbmRwb2ludENvbmZpZz8sIHNhZ2VtYWtlci5DZm5Nb2RlbD9dIHtcbiAgLyoqIENvbmRpdGlvbmFsIFNhZ2VtYWtlciBlbmRwb2ludCBjcmVhdGlvbiAqL1xuICBpZiAoIXByb3BzLmV4aXN0aW5nU2FnZW1ha2VyRW5kcG9pbnRPYmopIHtcbiAgICBpZiAocHJvcHMubW9kZWxQcm9wcykge1xuICAgICAgLyoqIHJldHVybiBbZW5kcG9pbnQsIGVuZHBvaW50Q29uZmlnLCBtb2RlbF0gKi9cbiAgICAgIHJldHVybiBkZXBsb3lTYWdlbWFrZXJFbmRwb2ludChzY29wZSwgcHJvcHMpO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aHJvdyBFcnJvcignRWl0aGVyIGV4aXN0aW5nU2FnZW1ha2VyRW5kcG9pbnRPYmogb3IgYXQgbGVhc3QgbW9kZWxQcm9wcyBpcyByZXF1aXJlZCcpO1xuICAgIH1cbiAgfSBlbHNlIHtcbiAgICAvKiogT3RoZXJ3aXNlLCByZXR1cm4gW2VuZHBvaW50XSAqL1xuICAgIHJldHVybiBbcHJvcHMuZXhpc3RpbmdTYWdlbWFrZXJFbmRwb2ludE9ial07XG4gIH1cbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGRlcGxveVNhZ2VtYWtlckVuZHBvaW50KFxuICBzY29wZTogQ29uc3RydWN0LFxuICBwcm9wczogQnVpbGRTYWdlbWFrZXJFbmRwb2ludFByb3BzXG4pOiBbc2FnZW1ha2VyLkNmbkVuZHBvaW50LCBzYWdlbWFrZXIuQ2ZuRW5kcG9pbnRDb25maWc/LCBzYWdlbWFrZXIuQ2ZuTW9kZWw/XSB7XG4gIGxldCBtb2RlbDogc2FnZW1ha2VyLkNmbk1vZGVsO1xuICBsZXQgZW5kcG9pbnRDb25maWc6IHNhZ2VtYWtlci5DZm5FbmRwb2ludENvbmZpZztcbiAgbGV0IGVuZHBvaW50OiBzYWdlbWFrZXIuQ2ZuRW5kcG9pbnQ7XG4gIGxldCBzYWdlbWFrZXJSb2xlOiBpYW0uUm9sZTtcblxuICAvLyBDcmVhdGUgU2FnZW1ha2VyJ3MgbW9kZWwsIGVuZHBvaW50Q29uZmlnLCBhbmQgZW5kcG9pbnRcbiAgaWYgKHByb3BzLm1vZGVsUHJvcHMpIHtcbiAgICAvLyBDaGVjayBpZiB0aGUgY2xpZW50IGhhcyBwcm92aWRlZCBleGVjdXRpb25Sb2xlQXJuXG4gICAgaWYgKHByb3BzLm1vZGVsUHJvcHMuZXhlY3V0aW9uUm9sZUFybikge1xuICAgICAgc2FnZW1ha2VyUm9sZSA9IGlhbS5Sb2xlLmZyb21Sb2xlQXJuKFxuICAgICAgICBzY29wZSxcbiAgICAgICAgJ1NhZ2VtYWtlclJvbGVDdXN0b21lcicsXG4gICAgICAgIHByb3BzLm1vZGVsUHJvcHMuZXhlY3V0aW9uUm9sZUFyblxuICAgICAgKSBhcyBpYW0uUm9sZTtcbiAgICB9IGVsc2Uge1xuICAgICAgLy8gQ3JlYXRlIHRoZSBTYWdlbWFrZXIgUm9sZVxuICAgICAgc2FnZW1ha2VyUm9sZSA9IG5ldyBpYW0uUm9sZShzY29wZSwgJ1NhZ2VtYWtlclJvbGUnLCB7XG4gICAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdzYWdlbWFrZXIuYW1hem9uYXdzLmNvbScpLFxuICAgICAgfSk7XG4gICAgICAvLyBBZGQgcmVxdWlyZWQgcGVybWlzc2lvbnNcbiAgICAgIGFkZFBlcm1pc3Npb25zKHNhZ2VtYWtlclJvbGUsIHByb3BzKTtcbiAgICB9XG5cbiAgICAvLyBDcmVhdGUgU2FnZW1ha2VyIE1vZGVsXG4gICAgbW9kZWwgPSBjcmVhdGVTYWdlbWFrZXJNb2RlbChzY29wZSwgcHJvcHMubW9kZWxQcm9wcywgc2FnZW1ha2VyUm9sZSwgcHJvcHMudnBjKTtcbiAgICAvLyBDcmVhdGUgU2FnZW1ha2VyIEVuZHBvaW50Q29uZmlnXG4gICAgZW5kcG9pbnRDb25maWcgPSBjcmVhdGVTYWdlbWFrZXJFbmRwb2ludENvbmZpZyhzY29wZSwgbW9kZWwuYXR0ck1vZGVsTmFtZSwgcHJvcHMuZW5kcG9pbnRDb25maWdQcm9wcyk7XG4gICAgLy8gQWRkIGRlcGVuZGVuY3kgb24gbW9kZWxcbiAgICBlbmRwb2ludENvbmZpZy5hZGREZXBlbmRzT24obW9kZWwpO1xuICAgIC8vIENyZWF0ZSBTYWdlbWFrZXIgRW5kcG9pbnRcbiAgICBlbmRwb2ludCA9IGNyZWF0ZVNhZ2VtYWtlckVuZHBvaW50KHNjb3BlLCBlbmRwb2ludENvbmZpZy5hdHRyRW5kcG9pbnRDb25maWdOYW1lLCBwcm9wcy5lbmRwb2ludFByb3BzKTtcbiAgICAvLyBBZGQgZGVwZW5kZW5jeSBvbiBFbmRwb2ludENvbmZpZ1xuICAgIGVuZHBvaW50LmFkZERlcGVuZHNPbihlbmRwb2ludENvbmZpZyk7XG5cbiAgICByZXR1cm4gW2VuZHBvaW50LCBlbmRwb2ludENvbmZpZywgbW9kZWxdO1xuICB9IGVsc2Uge1xuICAgIHRocm93IEVycm9yKCdZb3UgbmVlZCB0byBwcm92aWRlIGF0IGxlYXN0IG1vZGVsUHJvcHMgdG8gY3JlYXRlIFNhZ2VtYWtlciBFbmRwb2ludCcpO1xuICB9XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBjcmVhdGVTYWdlbWFrZXJNb2RlbChcbiAgc2NvcGU6IENvbnN0cnVjdCxcbiAgbW9kZWxQcm9wczogc2FnZW1ha2VyLkNmbk1vZGVsUHJvcHMsXG4gIHJvbGU6IGlhbS5Sb2xlLFxuICB2cGM/OiBlYzIuSVZwY1xuKTogc2FnZW1ha2VyLkNmbk1vZGVsIHtcbiAgbGV0IGZpbmFsTW9kZWxQcm9wczogc2FnZW1ha2VyLkNmbk1vZGVsUHJvcHM7XG4gIGxldCBwcmltYXJ5Q29udGFpbmVyOiBzYWdlbWFrZXIuQ2ZuTW9kZWwuQ29udGFpbmVyRGVmaW5pdGlvblByb3BlcnR5O1xuICBsZXQgdnBjQ29uZmlnOiBzYWdlbWFrZXIuQ2ZuTW9kZWwuVnBjQ29uZmlnUHJvcGVydHkgfCB1bmRlZmluZWQ7XG4gIGxldCBtb2RlbDogc2FnZW1ha2VyLkNmbk1vZGVsO1xuXG4gIGlmICh2cGMpIHtcbiAgICBjb25zdCBtb2RlbERlZmF1bHRTZWN1cml0eUdyb3VwID0gbmV3IGVjMi5TZWN1cml0eUdyb3VwKHNjb3BlLCAnUmVwbGFjZU1vZGVsRGVmYXVsdFNlY3VyaXR5R3JvdXAnLCB7XG4gICAgICB2cGMsXG4gICAgICBhbGxvd0FsbE91dGJvdW5kOiB0cnVlLFxuICAgIH0pO1xuXG4gICAgLy8gQWxsb3cgaHR0cHMgdHJhZmZpYyBmcm9tIHdpdGhpbiB0aGUgVlBDXG4gICAgbW9kZWxEZWZhdWx0U2VjdXJpdHlHcm91cC5hZGRJbmdyZXNzUnVsZShlYzIuUGVlci5pcHY0KHZwYy52cGNDaWRyQmxvY2spLCBlYzIuUG9ydC50Y3AoNDQzKSk7XG5cbiAgICBjb25zdCBjZm5TZWN1cml0eUdyb3VwID0gbW9kZWxEZWZhdWx0U2VjdXJpdHlHcm91cC5ub2RlLmZpbmRDaGlsZCgnUmVzb3VyY2UnKSBhcyBlYzIuQ2ZuU2VjdXJpdHlHcm91cDtcbiAgICBhZGRDZm5TdXBwcmVzc1J1bGVzKGNmblNlY3VyaXR5R3JvdXAsIFtcbiAgICAgIHtcbiAgICAgICAgaWQ6ICdXNScsXG4gICAgICAgIHJlYXNvbjogJ0VncmVzcyBvZiAwLjAuMC4wLzAgaXMgZGVmYXVsdCBhbmQgZ2VuZXJhbGx5IGNvbnNpZGVyZWQgT0snLFxuICAgICAgfSxcbiAgICAgIHtcbiAgICAgICAgaWQ6ICdXNDAnLFxuICAgICAgICByZWFzb246ICdFZ3Jlc3MgSVBQcm90b2NvbCBvZiAtMSBpcyBkZWZhdWx0IGFuZCBnZW5lcmFsbHkgY29uc2lkZXJlZCBPSycsXG4gICAgICB9XG4gICAgXSk7XG5cbiAgICAvLyBUaHJvdyBhbiBlcnJvciBpZiB0aGUgVlBDIGRvZXMgbm90IGNvbnRhaW4gcHJpdmF0ZSBvciBpc29sYXRlZCBzdWJuZXRzXG4gICAgaWYgKHZwYy5wcml2YXRlU3VibmV0cy5sZW5ndGggPT09IDAgJiYgdnBjLmlzb2xhdGVkU3VibmV0cy5sZW5ndGggPT09IDApIHtcbiAgICAgIHRocm93IEVycm9yKCdWUEMgbXVzdCBjb250YWluIHByaXZhdGUgb3IgaXNvbGF0ZWQgc3VibmV0cyB0byBkZXBsb3kgdGhlIFNhZ2VtYWtlciBlbmRwb2ludCBpbiBhIHZwYycpO1xuICAgIH1cblxuICAgIHZwY0NvbmZpZyA9IHtcbiAgICAgIC8vIGRlZmF1bHQgU3VibmV0VHlwZS5QUklWQVRFIChvciBJU09MQVRFRCBvciBQVUJMSUMgaWYgdGhlcmUgYXJlIG5vIFBSSVZBVEUgc3VibmV0cylcbiAgICAgIC8vIFNvLCBwcml2YXRlIHN1Ym5ldHMgd2lsbCBiZSB1c2VkIGlmIHByb3ZpZGVkIGJ5IGN1c3RvbWVyLiBPdGhlcndpc2UsIHVzZSB0aGUgZGVmYXVsdCBpc29sYXRlZCBzdWJuZXRzLFxuICAgICAgc3VibmV0czogdnBjLnNlbGVjdFN1Ym5ldHMoe1xuICAgICAgICBvbmVQZXJBejogdHJ1ZSxcbiAgICAgIH0pLnN1Ym5ldElkcyxcbiAgICAgIHNlY3VyaXR5R3JvdXBJZHM6IFttb2RlbERlZmF1bHRTZWN1cml0eUdyb3VwLnNlY3VyaXR5R3JvdXBJZF0sXG4gICAgfTtcbiAgfVxuXG4gIGlmIChtb2RlbFByb3BzLnByaW1hcnlDb250YWluZXIpIHtcbiAgICAvLyBHZXQgdXNlciBwcm92aWRlZCBNb2RlbCdzIHByaW1hcnkgY29udGFpbmVyXG4gICAgcHJpbWFyeUNvbnRhaW5lciA9IG1vZGVsUHJvcHMucHJpbWFyeUNvbnRhaW5lciBhcyBzYWdlbWFrZXIuQ2ZuTW9kZWwuQ29udGFpbmVyRGVmaW5pdGlvblByb3BlcnR5O1xuICAgIC8vIEdldCBkZWZhdWx0IE1vZGVsIHByb3BzXG4gICAgZmluYWxNb2RlbFByb3BzID0gRGVmYXVsdFNhZ2VtYWtlck1vZGVsUHJvcHMocm9sZS5yb2xlQXJuLCBwcmltYXJ5Q29udGFpbmVyLCB2cGNDb25maWcpO1xuICAgIC8vIE92ZXJyaWRlIGRlZmF1bHQgbW9kZWwgcHJvcGVydGllc1xuICAgIGZpbmFsTW9kZWxQcm9wcyA9IG92ZXJyaWRlUHJvcHMoZmluYWxNb2RlbFByb3BzLCBtb2RlbFByb3BzKTtcblxuICAgIC8vIENyZWF0ZSB0aGUgU2FnZW1ha2VyJ3MgTW9kZWxcbiAgICBtb2RlbCA9IG5ldyBzYWdlbWFrZXIuQ2ZuTW9kZWwoc2NvcGUsICdTYWdlbWFrZXJNb2RlbCcsIGZpbmFsTW9kZWxQcm9wcyk7XG4gICAgLy8gQWRkIGRlcGVuZGVuY3kgb24gdGhlIFNhZ2VtYWtlcidzIHJvbGVcbiAgICBtb2RlbC5ub2RlLmFkZERlcGVuZGVuY3kocm9sZSk7XG5cbiAgICByZXR1cm4gbW9kZWw7XG4gIH0gZWxzZSB7XG4gICAgdGhyb3cgRXJyb3IoJ1lvdSBuZWVkIHRvIHByb3ZpZGUgYXQgbGVhc3QgcHJpbWFyeUNvbnRhaW5lciB0byBjcmVhdGUgU2FnZW1ha2VyIE1vZGVsJyk7XG4gIH1cbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGNyZWF0ZVNhZ2VtYWtlckVuZHBvaW50Q29uZmlnKFxuICBzY29wZTogQ29uc3RydWN0LFxuICBtb2RlbE5hbWU6IHN0cmluZyxcbiAgZW5kcG9pbnRDb25maWdQcm9wcz86IHNhZ2VtYWtlci5DZm5FbmRwb2ludENvbmZpZ1Byb3BzXG4pOiBzYWdlbWFrZXIuQ2ZuRW5kcG9pbnRDb25maWcge1xuICBsZXQgZmluYWxFbmRwb2ludENvbmZpZ1Byb3BzOiBzYWdlbWFrZXIuQ2ZuRW5kcG9pbnRDb25maWdQcm9wcztcbiAgbGV0IGttc0tleUlkOiBzdHJpbmc7XG4gIGxldCBlbmRwb2ludENvbmZpZzogc2FnZW1ha2VyLkNmbkVuZHBvaW50Q29uZmlnO1xuXG4gIC8vIENyZWF0ZSBlbmNyeXB0aW9uIGtleSBpZiBvbmUgaXMgbm90IHByb3ZpZGVkXG4gIGlmIChlbmRwb2ludENvbmZpZ1Byb3BzICYmIGVuZHBvaW50Q29uZmlnUHJvcHMua21zS2V5SWQpIHtcbiAgICBrbXNLZXlJZCA9IGVuZHBvaW50Q29uZmlnUHJvcHMua21zS2V5SWQ7XG4gIH0gZWxzZSB7XG4gICAga21zS2V5SWQgPSBidWlsZEVuY3J5cHRpb25LZXkoc2NvcGUpLmtleUlkO1xuICB9XG4gIC8vIEdldCBkZWZhdWx0IEVuZHBvaW50Q29uZmlnIHByb3BzXG4gIGZpbmFsRW5kcG9pbnRDb25maWdQcm9wcyA9IERlZmF1bHRTYWdlbWFrZXJFbmRwb2ludENvbmZpZ1Byb3BzKG1vZGVsTmFtZSwga21zS2V5SWQpO1xuICAvLyBPdmVyd3JpdGUgZGVmYXVsdCBFbmRwb2ludENvbmZpZyBwcm9wZXJ0aWVzXG4gIGlmIChlbmRwb2ludENvbmZpZ1Byb3BzKSB7XG4gICAgZmluYWxFbmRwb2ludENvbmZpZ1Byb3BzID0gb3ZlcnJpZGVQcm9wcyhmaW5hbEVuZHBvaW50Q29uZmlnUHJvcHMsIGVuZHBvaW50Q29uZmlnUHJvcHMpO1xuICB9XG5cbiAgLy8gQ3JlYXRlIHRoZSBTYWdlbWFrZXIncyBFbmRwb2ludENvbmZpZ1xuICBlbmRwb2ludENvbmZpZyA9IG5ldyBzYWdlbWFrZXIuQ2ZuRW5kcG9pbnRDb25maWcoc2NvcGUsICdTYWdlbWFrZXJFbmRwb2ludENvbmZpZycsIGZpbmFsRW5kcG9pbnRDb25maWdQcm9wcyk7XG5cbiAgcmV0dXJuIGVuZHBvaW50Q29uZmlnO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gY3JlYXRlU2FnZW1ha2VyRW5kcG9pbnQoXG4gIHNjb3BlOiBDb25zdHJ1Y3QsXG4gIGVuZHBvaW50Q29uZmlnTmFtZTogc3RyaW5nLFxuICBlbmRwb2ludFByb3BzPzogc2FnZW1ha2VyLkNmbkVuZHBvaW50UHJvcHNcbik6IHNhZ2VtYWtlci5DZm5FbmRwb2ludCB7XG4gIGxldCBmaW5hbEVuZHBvaW50UHJvcHM6IHNhZ2VtYWtlci5DZm5FbmRwb2ludFByb3BzO1xuICBsZXQgZW5kcG9pbnQ6IHNhZ2VtYWtlci5DZm5FbmRwb2ludDtcblxuICAvLyBHZXQgZGVmYXVsdCBFbmRwb2ludCBwcm9wc1xuICBmaW5hbEVuZHBvaW50UHJvcHMgPSBEZWZhdWx0U2FnZW1ha2VyRW5kcG9pbnRQcm9wcyhlbmRwb2ludENvbmZpZ05hbWUpO1xuICAvLyBPdmVyd3JpdGUgZGVmYXVsdCBFbmRwb2ludCBwcm9wZXJ0aWVzXG4gIGlmIChlbmRwb2ludFByb3BzKSB7XG4gICAgZmluYWxFbmRwb2ludFByb3BzID0gb3ZlcnJpZGVQcm9wcyhmaW5hbEVuZHBvaW50UHJvcHMsIGVuZHBvaW50UHJvcHMpO1xuICB9XG5cbiAgLy8gQ3JlYXRlIHRoZSBTYWdlbWFrZXIncyBFbmRwb2ludFxuICBlbmRwb2ludCA9IG5ldyBzYWdlbWFrZXIuQ2ZuRW5kcG9pbnQoc2NvcGUsICdTYWdlbWFrZXJFbmRwb2ludCcsIGZpbmFsRW5kcG9pbnRQcm9wcyk7XG5cbiAgcmV0dXJuIGVuZHBvaW50O1xufVxuIl19