"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const elb = require("@aws-cdk/aws-elasticloadbalancingv2");
const acm = require("@aws-cdk/aws-certificatemanager");
const lambda = require("@aws-cdk/aws-lambda");
const defaults = require("../index");
require("@aws-cdk/assert/jest");
test('Test ObtainAlb with existing ALB', () => {
    const stack = new core_1.Stack();
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    const existingLoadBalancer = new elb.ApplicationLoadBalancer(stack, 'load-balancer', {
        vpc,
        internetFacing: true,
        loadBalancerName: 'unique-name'
    });
    defaults.ObtainAlb(stack, 'test', vpc, true, existingLoadBalancer);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Name: "unique-name",
    });
});
test('Test ObtainAlb for new ALB with provided props', () => {
    const stack = new core_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    defaults.ObtainAlb(stack, 'test', vpc, true, undefined, {
        loadBalancerName: 'new-loadbalancer',
        vpc,
        internetFacing: true
    });
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Name: "new-loadbalancer",
        Scheme: "internet-facing",
    });
});
test('Test ObtainAlb for new ALB with default props', () => {
    const stack = new core_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    defaults.ObtainAlb(stack, 'test', vpc, false);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: "internal",
    });
});
test('Test ObtainAlb for new ALB with default props', () => {
    const stack = new core_1.Stack();
    const testFunction = new lambda.Function(stack, 'test-function', {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: "index.handler",
    });
    defaults.CreateLambdaTargetGroup(stack, 'test-target', testFunction);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: "lambda"
    });
});
test('Test ObtainAlb for new ALB with custom props', () => {
    const stack = new core_1.Stack();
    const testFunction = new lambda.Function(stack, 'test-function', {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: "index.handler",
    });
    defaults.CreateLambdaTargetGroup(stack, 'test-target', testFunction, {
        targetGroupName: 'test-target-group'
    });
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: "lambda",
        Name: 'test-target-group'
    });
});
test('Test Add Target without ruleProps', () => {
    const stack = new core_1.Stack();
    const testFunction = new lambda.Function(stack, 'test-function', {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: "index.handler",
    });
    const targetGroup = defaults.CreateLambdaTargetGroup(stack, 'test-target', testFunction, {
        targetGroupName: 'test-target-group'
    });
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    const existingLoadBalancer = new elb.ApplicationLoadBalancer(stack, 'load-balancer', {
        vpc,
        internetFacing: true,
        loadBalancerName: 'unique-name'
    });
    const testListener = new elb.ApplicationListener(stack, 'test-listener', {
        loadBalancer: existingLoadBalancer,
        protocol: elb.ApplicationProtocol.HTTP
    });
    defaults.AddTarget(stack, targetGroup, testListener);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
        DefaultActions: [
            {
                TargetGroupArn: {
                    Ref: "testtargettgB2EE41CA"
                },
                Type: "forward"
            }
        ],
    });
});
test('Test Add Target with ruleProps', () => {
    const stack = new core_1.Stack();
    const testFunction = new lambda.Function(stack, 'test-function', {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: "index.handler",
    });
    const targetGroup = defaults.CreateLambdaTargetGroup(stack, 'test-target', testFunction, {
        targetGroupName: 'test-target-group'
    });
    const secondTargetGroup = defaults.CreateLambdaTargetGroup(stack, 'second-target', testFunction, {
        targetGroupName: 'second-target-group'
    });
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    const existingLoadBalancer = new elb.ApplicationLoadBalancer(stack, 'load-balancer', {
        vpc,
        internetFacing: true,
        loadBalancerName: 'unique-name'
    });
    const testListener = new elb.ApplicationListener(stack, 'test-listener', {
        loadBalancer: existingLoadBalancer,
        protocol: elb.ApplicationProtocol.HTTP
    });
    // The first target is default and can't have rules, so
    // we need to add 2 targets
    defaults.AddTarget(stack, targetGroup, testListener);
    defaults.AddTarget(stack, secondTargetGroup, testListener, {
        conditions: [elb.ListenerCondition.pathPatterns(["*admin*"])],
        priority: 10
    });
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::ListenerRule', {
        Actions: [
            {
                TargetGroupArn: {
                    Ref: "secondtargettg0CE37E1F"
                },
                Type: "forward"
            }
        ],
        Conditions: [
            {
                Field: "path-pattern",
                PathPatternConfig: {
                    Values: [
                        "*admin*"
                    ]
                }
            }
        ]
    });
});
test('Test AddListener with defaults', () => {
    const stack = new core_1.Stack();
    const testFunction = new lambda.Function(stack, 'test-function', {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: "index.handler",
    });
    const targetGroup = defaults.CreateLambdaTargetGroup(stack, 'test-target', testFunction, {
        targetGroupName: 'test-target-group'
    });
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    const existingLoadBalancer = new elb.ApplicationLoadBalancer(stack, 'load-balancer', {
        vpc,
        internetFacing: true,
        loadBalancerName: 'unique-name'
    });
    const cert = acm.Certificate.fromCertificateArn(stack, 'not-really-a-cert', "arn:aws:acm:us-east-1:123456789012:certificate/85c52dc8-1b37-4afd-a7aa-f03aac2db0cc");
    defaults.AddListener(stack, existingLoadBalancer, targetGroup, {
        certificates: [cert],
    });
    // This should create 2 listeners, HTTPS plus redirect of HTTP
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS',
    });
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP',
    });
});
test('Test AddListener with no cert for an HTTPS listener', () => {
    const stack = new core_1.Stack();
    const testFunction = new lambda.Function(stack, 'test-function', {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: "index.handler",
    });
    const targetGroup = defaults.CreateLambdaTargetGroup(stack, 'test-target', testFunction, {
        targetGroupName: 'test-target-group'
    });
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    const existingLoadBalancer = new elb.ApplicationLoadBalancer(stack, 'load-balancer', {
        vpc,
        internetFacing: true,
        loadBalancerName: 'unique-name'
    });
    const app = () => {
        defaults.AddListener(stack, existingLoadBalancer, targetGroup, {});
    };
    expect(app).toThrowError();
});
test('Test AddListener error for HTTP with a cert', () => {
    const stack = new core_1.Stack();
    const testFunction = new lambda.Function(stack, 'test-function', {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: "index.handler",
    });
    const targetGroup = defaults.CreateLambdaTargetGroup(stack, 'test-target', testFunction, {
        targetGroupName: 'test-target-group'
    });
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    const existingLoadBalancer = new elb.ApplicationLoadBalancer(stack, 'load-balancer', {
        vpc,
        internetFacing: true,
        loadBalancerName: 'unique-name'
    });
    const cert = acm.Certificate.fromCertificateArn(stack, 'not-really-a-cert', "arn:aws:acm:us-east-1:123456789012:certificate/85c52dc8-1b37-4afd-a7aa-f03aac2db0cc");
    const app = () => {
        defaults.AddListener(stack, existingLoadBalancer, targetGroup, {
            certificates: [cert],
            protocol: elb.ApplicationProtocol.HTTP,
        });
    };
    expect(app).toThrowError();
});
test('Test AddListener for HTTP Listener', () => {
    const stack = new core_1.Stack();
    const testFunction = new lambda.Function(stack, 'test-function', {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: "index.handler",
    });
    const targetGroup = defaults.CreateLambdaTargetGroup(stack, 'test-target', testFunction, {
        targetGroupName: 'test-target-group'
    });
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    const existingLoadBalancer = new elb.ApplicationLoadBalancer(stack, 'load-balancer', {
        vpc,
        internetFacing: true,
        loadBalancerName: 'unique-name'
    });
    defaults.AddListener(stack, existingLoadBalancer, targetGroup, {
        protocol: elb.ApplicationProtocol.HTTP,
    });
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP',
    });
    expect(stack).not.toHaveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS',
    });
});
test('Test with custom logging bucket props', () => {
    // Creating ALB logging requires a region and account (but
    // these can be fake in unit tests)
    const stack = new core_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    const testName = 'test-name';
    defaults.ObtainAlb(stack, 'test', vpc, false, undefined, undefined, true, { bucketName: testName });
    expect(stack).toHaveResourceLike('AWS::S3::Bucket', {
        BucketName: testName
    });
});
test('Test with no logging', () => {
    const stack = new core_1.Stack();
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    defaults.ObtainAlb(stack, 'test', vpc, false, undefined, undefined, false);
    expect(stack).not.toHaveResourceLike('AWS::S3::Bucket', {});
});
//# sourceMappingURL=data:application/json;base64,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