"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lambda = require("@aws-cdk/aws-lambda");
const api = require("@aws-cdk/aws-apigateway");
const iam = require("@aws-cdk/aws-iam");
const defaults = require("../index");
require("@aws-cdk/assert/jest");
function deployRegionalApiGateway(stack) {
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    return defaults.RegionalLambdaRestApi(stack, fn);
}
function setupRestApi(stack, apiProps) {
    const [restApi] = defaults.GlobalRestApi(stack, apiProps);
    // Setup the API Gateway resource
    const apiGatewayResource = restApi.root.addResource('api-gateway-resource');
    // Setup the API Gateway Integration
    const apiGatewayIntegration = new api.AwsIntegration({
        service: "sqs",
        integrationHttpMethod: "POST",
        options: {
            passthroughBehavior: api.PassthroughBehavior.NEVER,
            requestParameters: {
                "integration.request.header.Content-Type": "'application/x-www-form-urlencoded'"
            },
            requestTemplates: {
                "application/x-www-form-urlencoded": "Action=SendMessage&MessageBody=$util.urlEncode(\"$input.body\")&MessageAttribute.1.Name=queryParam1&MessageAttribute.1.Value.StringValue=$input.params(\"query_param_1\")&MessageAttribute.1.Value.DataType=String"
            },
            integrationResponses: [
                {
                    statusCode: "200",
                    responseTemplates: {
                        "text/html": "Success"
                    }
                },
                {
                    statusCode: "500",
                    responseTemplates: {
                        "text/html": "Error"
                    },
                    selectionPattern: "500"
                }
            ]
        },
        path: '11112222' + "/" + 'thisqueuequeueName'
    });
    // Setup the API Gateway method(s)
    apiGatewayResource.addMethod('POST', apiGatewayIntegration, {
        requestParameters: {
            "method.request.querystring.query_param_1": true
        },
        methodResponses: [
            {
                statusCode: "200",
                responseParameters: {
                    "method.response.header.Content-Type": true
                }
            },
            {
                statusCode: "500",
                responseParameters: {
                    "method.response.header.Content-Type": true
                },
            }
        ]
    });
}
test('Test override for RegionalApiGateway', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    defaults.RegionalLambdaRestApi(stack, fn, {
        handler: fn,
        description: 'Hello World'
    });
    expect(stack).toHaveResource('AWS::ApiGateway::RestApi', {
        Type: "AWS::ApiGateway::RestApi",
        Properties: {
            Description: "Hello World",
            EndpointConfiguration: {
                Types: [
                    "REGIONAL"
                ]
            },
            Name: "LambdaRestApi"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
test('Test override for GlobalApiGateway', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    defaults.GlobalLambdaRestApi(stack, fn, {
        handler: fn,
        restApiName: "HelloWorld"
    });
    expect(stack).toHaveResource('AWS::ApiGateway::RestApi', {
        Type: "AWS::ApiGateway::RestApi",
        Properties: {
            EndpointConfiguration: {
                Types: [
                    "EDGE"
                ]
            },
            Name: "HelloWorld"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
test('Test ApiGateway::Account resource for RegionalApiGateway', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    defaults.RegionalLambdaRestApi(stack, fn);
    expect(stack).toHaveResource('AWS::ApiGateway::Account', {
        CloudWatchRoleArn: {
            "Fn::GetAtt": [
                "LambdaRestApiCloudWatchRoleF339D4E6",
                "Arn"
            ]
        }
    });
});
test('Test ApiGateway::Account resource for GlobalApiGateway', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    defaults.GlobalLambdaRestApi(stack, fn);
    expect(stack).toHaveResource('AWS::ApiGateway::Account', {
        CloudWatchRoleArn: {
            "Fn::GetAtt": [
                "LambdaRestApiCloudWatchRoleF339D4E6",
                "Arn"
            ]
        }
    });
});
test('Test default RestApi deployment w/ ApiGatewayProps', () => {
    const stack = new core_1.Stack();
    setupRestApi(stack, {
        restApiName: "customRestApi"
    });
    expect(stack).toHaveResource('AWS::ApiGateway::RestApi', {
        Name: "customRestApi"
    });
});
test('Test default RestApi deployment w/ cloudWatchRole set to false', () => {
    const stack = new core_1.Stack();
    setupRestApi(stack, {
        cloudWatchRole: false
    });
    expect(stack).not.toHaveResourceLike("AWS::ApiGateway::Account", {});
});
test('Test default RestApi deployment for Cloudwatch loggroup', () => {
    const stack = new core_1.Stack();
    deployRegionalApiGateway(stack);
    expect(stack).toHaveResource('AWS::Logs::LogGroup', {
        UpdateReplacePolicy: "Retain",
        DeletionPolicy: "Retain"
    }, assert_1.ResourcePart.CompleteDefinition);
    expect(stack).toHaveResource('AWS::ApiGateway::Stage', {
        AccessLogSetting: {
            DestinationArn: {
                "Fn::GetAtt": [
                    "ApiAccessLogGroupCEA70788",
                    "Arn"
                ]
            },
            Format: "{\"requestId\":\"$context.requestId\",\"ip\":\"$context.identity.sourceIp\",\"user\":\"$context.identity.user\",\"caller\":\"$context.identity.caller\",\"requestTime\":\"$context.requestTime\",\"httpMethod\":\"$context.httpMethod\",\"resourcePath\":\"$context.resourcePath\",\"status\":\"$context.status\",\"protocol\":\"$context.protocol\",\"responseLength\":\"$context.responseLength\"}",
        },
    });
});
test('Test addMethodToApiResource with action', () => {
    const stack = new core_1.Stack();
    const [restApi] = defaults.GlobalRestApi(stack);
    // Setup the API Gateway role
    const apiGatewayRole = new iam.Role(stack, 'api-gateway-role', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
    });
    // Setup the API Gateway resource
    const apiGatewayResource = restApi.root.addResource('api-gateway-resource');
    const getRequestTemplate = "{}";
    // Add Method
    defaults.addProxyMethodToApiResource({
        action: "Query",
        service: "dynamodb",
        apiResource: apiGatewayResource,
        apiGatewayRole,
        apiMethod: "GET",
        requestTemplate: getRequestTemplate
    });
    // Add Method
    defaults.addProxyMethodToApiResource({
        path: '11112222' + "/" + 'thisqueuequeueName',
        service: "sqs",
        apiResource: apiGatewayResource,
        apiGatewayRole,
        apiMethod: "PUT",
        requestTemplate: getRequestTemplate
    });
    // Error scenario: missing action and path
    try {
        defaults.addProxyMethodToApiResource({
            service: "sqs",
            apiResource: apiGatewayResource,
            apiGatewayRole,
            apiMethod: "DELETE",
            requestTemplate: getRequestTemplate
        });
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
test('Test default RestApi w/ request model and validator', () => {
    const stack = new core_1.Stack();
    const [restApi] = defaults.GlobalRestApi(stack);
    // Setup the API Gateway role
    const apiGatewayRole = new iam.Role(stack, 'api-gateway-role', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
    });
    // Setup the API Gateway resource
    const apiGatewayResource = restApi.root.addResource('api-gateway-resource');
    const validator = restApi.addRequestValidator('default-validator', {
        requestValidatorName: 'default-validator',
        validateRequestBody: true
    });
    defaults.addProxyMethodToApiResource({
        service: "kinesis",
        action: "PutRecord",
        apiGatewayRole,
        apiMethod: 'POST',
        apiResource: apiGatewayResource,
        requestTemplate: "{}",
        contentType: "'x-amz-json-1.1'",
        requestValidator: validator,
        requestModel: { "application/json": api.Model.EMPTY_MODEL }
    });
    expect(stack).toHaveResource('AWS::ApiGateway::RequestValidator', {
        Name: "default-validator",
        ValidateRequestBody: true
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Method', {
        RequestModels: { "application/json": "Empty" }
    });
});
// -----------------------------------------------------------------------
// Test for Regional ApiGateway Creation
// -----------------------------------------------------------------------
test('Test for RegionalRestApiGateway', () => {
    const stack = new core_1.Stack();
    const [regionalApi] = defaults.RegionalRestApi(stack, {
        restApiName: "HelloWorld-RegionalApi"
    });
    // Setup the API Gateway role
    const apiGatewayRole = new iam.Role(stack, 'api-gateway-role', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
    });
    // Setup the API Gateway resource
    const apiGatewayResource = regionalApi.root.addResource('hello');
    defaults.addProxyMethodToApiResource({
        service: 'iotdata',
        path: 'hello',
        apiGatewayRole,
        apiMethod: 'POST',
        apiResource: apiGatewayResource,
        requestTemplate: "$input.json('$')"
    });
    expect(stack).toHaveResource('AWS::ApiGateway::RestApi', {
        Type: "AWS::ApiGateway::RestApi",
        Properties: {
            EndpointConfiguration: {
                Types: [
                    "REGIONAL"
                ]
            },
            Name: "HelloWorld-RegionalApi"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// -----------------------------------------------------------------------
// Tests for exception while overriding restApiProps using endPointTypes
// -----------------------------------------------------------------------
test('Test for Exception while overriding restApiProps using endPointTypes', () => {
    const stack = new core_1.Stack();
    try {
        defaults.RegionalRestApi(stack, {
            endpointTypes: [api.EndpointType.REGIONAL]
        });
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
// -----------------------------------------------------------------------
// Tests for exception while overriding LambdaRestApiProps using endPointTypes
// -----------------------------------------------------------------------
test('Test for Exception while overriding LambdaRestApiProps using endPointTypes', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    try {
        defaults.GlobalLambdaRestApi(stack, fn, {
            handler: fn,
            endpointTypes: [api.EndpointType.REGIONAL]
        });
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
// -----------------------------------------------------------------------
// Test for Integration Request Props Override
// -----------------------------------------------------------------------
test('Test for Integration Request Props Override', () => {
    const stack = new core_1.Stack();
    const [regionalApi] = defaults.RegionalRestApi(stack);
    // Setup the API Gateway role
    const apiGatewayRole = new iam.Role(stack, 'api-gateway-role', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
    });
    // Setup the API Gateway resource
    const apiGatewayResource = regionalApi.root.addResource('hello');
    const integReqParams = { 'integration.request.path.topic-level-1': "'method.request.path.topic-level-1'" };
    const integResp = [
        {
            statusCode: "200",
            selectionPattern: "2\\d{2}",
            responseTemplates: {
                "application/json": "$input.json('$')"
            }
        }
    ];
    // Override the default Integration Request Props
    const integrationReqProps = {
        subdomain: 'abcdefgh12345',
        options: {
            requestParameters: integReqParams,
            integrationResponses: integResp,
            passthroughBehavior: api.PassthroughBehavior.WHEN_NO_MATCH
        }
    };
    defaults.addProxyMethodToApiResource({
        service: 'iotdata',
        path: 'hello',
        apiGatewayRole,
        apiMethod: 'POST',
        apiResource: apiGatewayResource,
        requestTemplate: "$input.json('$')",
        awsIntegrationProps: integrationReqProps
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        AuthorizationType: "AWS_IAM",
        Integration: {
            IntegrationHttpMethod: "POST",
            IntegrationResponses: [
                {
                    ResponseTemplates: {
                        "application/json": "$input.json('$')"
                    },
                    SelectionPattern: "2\\d{2}",
                    StatusCode: "200"
                }
            ],
            PassthroughBehavior: "WHEN_NO_MATCH",
            RequestParameters: {
                "integration.request.header.Content-Type": "'application/json'",
                "integration.request.path.topic-level-1": "'method.request.path.topic-level-1'",
            },
            RequestTemplates: {
                "application/json": "$input.json('$')"
            },
            Type: "AWS",
            Uri: {
                "Fn::Join": [
                    "",
                    [
                        "arn:",
                        {
                            Ref: "AWS::Partition"
                        },
                        ":apigateway:",
                        {
                            Ref: "AWS::Region"
                        },
                        `:abcdefgh12345.iotdata:path/hello`
                    ]
                ]
            }
        },
        MethodResponses: [
            {
                StatusCode: "200",
                ResponseParameters: {
                    "method.response.header.Content-Type": true
                }
            }
        ]
    });
});
// -----------------------------------------------------------------------
// Test for Method Options Override
// -----------------------------------------------------------------------
test('Test for Method Request Props Override', () => {
    const stack = new core_1.Stack();
    const [globalApi] = defaults.GlobalRestApi(stack);
    // Setup the API Gateway role
    const apiGatewayRole = new iam.Role(stack, 'api-gateway-role', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
    });
    // Setup the API Gateway resource
    const apiGatewayResource = globalApi.root.addResource('hello');
    const methodReqParams = { 'method.request.path.topic-level-1': true };
    const methodResp = [
        {
            statusCode: "403"
        }
    ];
    const resourceMethodOptions = {
        requestParameters: methodReqParams,
        methodResponses: methodResp,
    };
    defaults.addProxyMethodToApiResource({
        service: 'iotdata',
        path: 'hello',
        apiGatewayRole,
        apiMethod: 'POST',
        apiResource: apiGatewayResource,
        requestTemplate: "$input.json('$')",
        methodOptions: resourceMethodOptions
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        AuthorizationType: "AWS_IAM",
        Integration: {
            IntegrationHttpMethod: "POST",
            IntegrationResponses: [
                {
                    StatusCode: "200"
                },
                {
                    StatusCode: "500",
                    ResponseTemplates: {
                        "text/html": "Error"
                    },
                    SelectionPattern: "500"
                }
            ],
            PassthroughBehavior: "NEVER",
            RequestParameters: {
                "integration.request.header.Content-Type": "'application/json'",
            },
            RequestTemplates: {
                "application/json": "$input.json('$')"
            },
            Type: "AWS",
            Uri: {
                "Fn::Join": [
                    "",
                    [
                        "arn:",
                        {
                            Ref: "AWS::Partition"
                        },
                        ":apigateway:",
                        {
                            Ref: "AWS::Region"
                        },
                        `:iotdata:path/hello`
                    ]
                ]
            }
        },
        MethodResponses: [
            {
                StatusCode: "403"
            }
        ],
        RequestParameters: {
            "method.request.path.topic-level-1": true
        }
    });
});
// -----------------------------------------------------------------------
// Test for ApiKey Creation of RestApi
// -----------------------------------------------------------------------
test('Test for ApiKey creation using restApiProps', () => {
    const stack = new core_1.Stack();
    const [globalRestApi] = defaults.GlobalRestApi(stack, {
        defaultMethodOptions: {
            apiKeyRequired: true
        }
    });
    // Setup the API Gateway role
    const apiGatewayRole = new iam.Role(stack, 'api-gateway-role', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
    });
    // Setup the API Gateway resource
    const apiGatewayResource = globalRestApi.root.addResource('hello');
    defaults.addProxyMethodToApiResource({
        service: 'iotdata',
        path: 'hello',
        apiGatewayRole,
        apiMethod: 'POST',
        apiResource: apiGatewayResource,
        requestTemplate: "$input.json('$')"
    });
    // Assertion to check for ApiKey
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        ApiKeyRequired: true
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::ApiKey", {
        Enabled: true
    });
    // Assertion to check for UsagePlan Api Key Mapping
    expect(stack).toHaveResourceLike("AWS::ApiGateway::UsagePlanKey", {
        KeyType: "API_KEY"
    });
});
// -----------------------------------------------------------------------
// Test for ApiKey Creation of LambdaRestApi
// -----------------------------------------------------------------------
test('Test for ApiKey creation using lambdaApiProps', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    defaults.RegionalLambdaRestApi(stack, fn, {
        handler: fn,
        defaultMethodOptions: {
            apiKeyRequired: true
        }
    });
    // Assertion to check for ApiKey
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        ApiKeyRequired: true
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::ApiKey", {
        Enabled: true
    });
    // Assertion to check for UsagePlan Api Key Mapping
    expect(stack).toHaveResourceLike("AWS::ApiGateway::UsagePlanKey", {
        KeyType: "API_KEY"
    });
});
//# sourceMappingURL=data:application/json;base64,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