"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const defaults = require("..");
const _fieldSchema = [{
        name: "id",
        type: "int",
        comment: "Identifier for the record"
    }, {
        name: "name",
        type: "string",
        comment: "The name of the record"
    }, {
        name: "type",
        type: "string",
        comment: "The type of the record"
    }, {
        name: "numericvalue",
        type: "int",
        comment: "Some value associated with the record"
    }];
// --------------------------------------------------------------
// Create database by passing props
// --------------------------------------------------------------
test('create default CfnTable with default props', () => {
    const stack = new core_1.Stack();
    defaults.DefaultGlueTable(stack, {
        catalogId: 'fakecatalogfortest',
        databaseName: 'fakedatabase',
        tableInput: {
            parameters: {
                classification: 'json'
            },
            storageDescriptor: {
                parameters: {
                    endpointUrl: `https://kinesis.${core_1.Aws.REGION}.amazonaws.com`,
                    streamName: 'testStream',
                    typeOfData: 'kinesis'
                }
            }
        }
    });
    expect(stack).toHaveResourceLike('AWS::Glue::Table', {
        Type: "AWS::Glue::Table",
        Properties: {
            CatalogId: "fakecatalogfortest",
            DatabaseName: "fakedatabase",
            TableInput: {
                Parameters: {
                    classification: "json"
                },
                StorageDescriptor: {
                    Parameters: {
                        endpointUrl: {
                            "Fn::Join": [
                                "",
                                [
                                    "https://kinesis.",
                                    {
                                        Ref: "AWS::Region"
                                    },
                                    ".amazonaws.com"
                                ]
                            ]
                        },
                        streamName: "testStream",
                        typeOfData: "kinesis"
                    }
                }
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Create database by passing no props to database (default database)
// --------------------------------------------------------------
test('Create table', () => {
    const stack = new core_1.Stack();
    defaults.createGlueTable(stack, defaults.createGlueDatabase(stack), undefined, _fieldSchema, 'kinesis', {
        STREAM_NAME: 'testStream'
    });
    expect(stack).toHaveResourceLike('AWS::Glue::Database', {
        Type: "AWS::Glue::Database",
        Properties: {
            CatalogId: {
                Ref: "AWS::AccountId"
            },
            DatabaseInput: {
                Description: "An AWS Glue database generated by AWS Solutions Construct"
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
    expect(stack).toHaveResourceLike('AWS::Glue::Table', {
        Properties: {
            CatalogId: {
                Ref: "AWS::AccountId"
            },
            DatabaseName: {
                Ref: "GlueDatabase"
            },
            TableInput: {
                Parameters: {
                    classification: "json"
                },
                StorageDescriptor: {
                    Columns: [
                        {
                            Comment: "Identifier for the record",
                            Name: "id",
                            Type: "int"
                        },
                        {
                            Comment: "The name of the record",
                            Name: "name",
                            Type: "string"
                        },
                        {
                            Comment: "The type of the record",
                            Name: "type",
                            Type: "string"
                        },
                        {
                            Comment: "Some value associated with the record",
                            Name: "numericvalue",
                            Type: "int"
                        }
                    ],
                    Compressed: false,
                    InputFormat: "org.apache.hadoop.mapred.TextInputFormat",
                    Location: "testStream",
                    NumberOfBuckets: -1,
                    OutputFormat: "org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat",
                    Parameters: {
                        endpointUrl: {
                            "Fn::Join": [
                                "",
                                [
                                    "https://kinesis.",
                                    {
                                        Ref: "AWS::Region"
                                    },
                                    ".amazonaws.com"
                                ]
                            ]
                        },
                        streamName: "testStream",
                        typeOfData: "kinesis"
                    },
                    SerdeInfo: {
                        Parameters: {
                            paths: "id,name,type,numericvalue"
                        },
                        SerializationLibrary: "org.openx.data.jsonserde.JsonSerDe"
                    }
                },
                TableType: "EXTERNAL_TABLE"
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Pass an unsupported source type, it should throw an error
// --------------------------------------------------------------
test('error condition', () => {
    const stack = new core_1.Stack();
    try {
        const _database = defaults.DefaultGlueDatabase(stack, defaults.DefaultGlueDatabaseProps());
        defaults.DefaultGlueTable(_database, defaults.DefaultGlueTableProps(_database, _fieldSchema, 'SomeSource', { STREAM_NAME: 'somefakestream' }));
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
//# sourceMappingURL=data:application/json;base64,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