"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const dynamodb = require("@aws-cdk/aws-dynamodb");
const glue = require("@aws-cdk/aws-glue");
const iam = require("@aws-cdk/aws-iam");
const kinesis = require("@aws-cdk/aws-kinesis");
const lambda = require("@aws-cdk/aws-lambda");
const mediastore = require("@aws-cdk/aws-mediastore");
const s3 = require("@aws-cdk/aws-s3");
const sns = require("@aws-cdk/aws-sns");
const sqs = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const defaults = require("../");
const mediastore_defaults_1 = require("../lib/mediastore-defaults");
const sagemaker_helper_1 = require("../lib/sagemaker-helper");
const test_helper_1 = require("./test-helper");
test('Test with valid props', () => {
    const props = {};
    defaults.CheckProps(props);
});
test('Test fail DynamoDB table check', () => {
    const stack = new core_1.Stack();
    const props = {
        existingTableObj: new dynamodb.Table(stack, 'placeholder', defaults.DefaultTableProps),
        dynamoTableProps: defaults.DefaultTableProps,
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide existingTableObj or dynamoTableProps, but not both.\n');
});
test("Test fail Lambda function check", () => {
    const stack = new core_1.Stack();
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: "index.handler",
        },
        existingLambdaObj: new lambda.Function(stack, "placeholder", {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: "index.handler",
        }),
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError("Error - Either provide lambdaFunctionProps or existingLambdaObj, but not both.\n");
});
test("Test fail SQS Queue check", () => {
    const stack = new core_1.Stack();
    const props = {
        queueProps: {},
        existingQueueObj: new sqs.Queue(stack, 'placeholder', {}),
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide queueProps or existingQueueObj, but not both.\n');
});
test('Test fail Dead Letter Queue check', () => {
    const props = {
        deployDeadLetterQueue: false,
        deadLetterQueueProps: {},
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - If deployDeadLetterQueue is false then deadLetterQueueProps cannot be specified.\n');
});
test('Test fail Dead Letter Queue check with queueProps fifo set to true and undefined deadLetterQueueProps', () => {
    const props = {
        queueProps: { fifo: true },
        deadLetterQueueProps: {},
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - If you specify a fifo: true in either queueProps or deadLetterQueueProps, you must also set fifo: true in the other props object. \
    Fifo must match for the Queue and the Dead Letter Queue.\n');
});
test('Test fail Dead Letter Queue check with queueProps fifo set to true and deadLetterQueueProps fifo set to false', () => {
    const props = {
        queueProps: { fifo: true },
        deadLetterQueueProps: { fifo: false },
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - If you specify a fifo: true in either queueProps or deadLetterQueueProps, you must also set fifo: true in the other props object. \
    Fifo must match for the Queue and the Dead Letter Queue.\n');
});
test('Test fail Dead Letter Queue check with queueProps fifo set to false and deadLetterQueueProps fifo set to true', () => {
    const props = {
        deadLetterQueueProps: { fifo: true },
        queueProps: { fifo: false },
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - If you specify a fifo: true in either queueProps or deadLetterQueueProps, you must also set fifo: true in the other props object. \
    Fifo must match for the Queue and the Dead Letter Queue.\n');
});
test('Test fail Dead Letter Queue check with deadLetterQueueProps fifo set to true', () => {
    const props = {
        deadLetterQueueProps: { fifo: true },
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    expect(app).toThrowError('Error - If you specify a fifo: true in either queueProps or deadLetterQueueProps, you must also set fifo: true in the other props object. \
    Fifo must match for the Queue and the Dead Letter Queue.\n');
});
test('Test fail Dead Letter Queue check with queueProps fifo set to false', () => {
    const props = {
        queueProps: { fifo: false },
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    expect(app).toThrowError('Error - If you specify a fifo: true in either queueProps or deadLetterQueueProps, you must also set fifo: true in the other props object. \
    Fifo must match for the Queue and the Dead Letter Queue.\n');
});
test("Test fail MediaStore container check", () => {
    const stack = new core_1.Stack();
    const mediaStoreContainer = new mediastore.CfnContainer(stack, "placeholder", mediastore_defaults_1.MediaStoreContainerProps());
    const props = {
        mediaStoreContainerProps: mediastore_defaults_1.MediaStoreContainerProps(),
        existingMediaStoreContainerObj: mediaStoreContainer,
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError("Error - Either provide mediaStoreContainerProps or existingMediaStoreContainerObj, but not both.\n");
});
test('Test fail Kinesis stream check', () => {
    const stack = new core_1.Stack();
    const stream = new kinesis.Stream(stack, 'placeholder', {});
    const props = {
        existingStreamObj: stream,
        kinesisStreamProps: {}
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide existingStreamObj or kinesisStreamProps, but not both.\n');
});
test('Test fail S3 check', () => {
    const stack = new core_1.Stack();
    const props = {
        existingBucketObj: test_helper_1.CreateScrapBucket(stack, {}),
        bucketProps: {},
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide bucketProps or existingBucketObj, but not both.\n');
});
test('Test fail SNS topic check', () => {
    const stack = new core_1.Stack();
    const props = {
        topicProps: {},
        existingTopicObj: new sns.Topic(stack, 'placeholder', {})
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide topicProps or existingTopicObj, but not both.\n');
});
test('Test fail SNS topic check with bad topic attribute name', () => {
    const stack = new core_1.Stack();
    const props = {
        topicProps: {},
        existingTopicObj: new sns.Topic(stack, 'placeholder', {})
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide topicProps or existingTopicObj, but not both.\n');
});
test('Test fail Glue job check', () => {
    const stack = new core_1.Stack();
    const _jobRole = new iam.Role(stack, 'CustomETLJobRole', {
        assumedBy: new iam.ServicePrincipal('glue.amazonaws.com')
    });
    const jobProps = defaults.DefaultGlueJobProps(_jobRole, {
        command: {
            name: 'glueetl',
            pythonVersion: '3',
            scriptLocation: new s3.Bucket(stack, 'ScriptBucket').bucketArn,
        },
        role: new iam.Role(stack, 'JobRole', {
            assumedBy: new iam.ServicePrincipal('glue.amazonaws.com')
        }).roleArn
    }, 'testETLJob', {});
    const job = new glue.CfnJob(stack, 'placeholder', jobProps);
    const props = {
        glueJobProps: jobProps,
        existingGlueJob: job
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide glueJobProps or existingGlueJob, but not both.\n');
});
test('Test fail SageMaker endpoint check', () => {
    const stack = new core_1.Stack();
    // Build Sagemaker Inference Endpoint
    const modelProps = {
        primaryContainer: {
            image: "<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest",
            modelDataUrl: "s3://<bucket-name>/<prefix>/model.tar.gz",
        },
    };
    const [endpoint] = sagemaker_helper_1.BuildSagemakerEndpoint(stack, { modelProps });
    const props = {
        existingSagemakerEndpointObj: endpoint,
        endpointProps: {
            endpointConfigName: 'placeholder'
        }
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide endpointProps or existingSagemakerEndpointObj, but not both.\n');
});
test('Test fail Secret check', () => {
    const stack = new core_1.Stack();
    const props = {
        secretProps: {},
        existingSecretObj: defaults.buildSecretsManagerSecret(stack, 'secret', {}),
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide secretProps or existingSecretObj, but not both.\n');
});
test('Test fail encryption key check', () => {
    const stack = new core_1.Stack();
    const key = defaults.buildEncryptionKey(stack, {
        enableKeyRotation: false
    });
    const props = {
        encryptionKey: key,
        encryptionKeyProps: {},
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide encryptionKey or encryptionKeyProps, but not both.\n');
});
test('Test fail Vpc check with deployVpc', () => {
    const stack = new core_1.Stack();
    const props = {
        deployVpc: true,
        existingVpc: defaults.buildVpc(stack, {
            defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
        }),
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
test('Test fail Vpc check with vpcProps', () => {
    const stack = new core_1.Stack();
    const props = {
        vpcProps: defaults.DefaultPublicPrivateVpcProps(),
        existingVpc: defaults.buildVpc(stack, {
            defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
        }),
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
test('Test fail multiple failures message', () => {
    const stack = new core_1.Stack();
    const props = {
        secretProps: {},
        existingSecretObj: defaults.buildSecretsManagerSecret(stack, 'secret', {}),
        topicProps: {},
        existingTopicObj: new sns.Topic(stack, 'placeholder', {})
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide topicProps or existingTopicObj, but not both.\n' +
        'Error - Either provide secretProps or existingSecretObj, but not both.\n');
});
//# sourceMappingURL=data:application/json;base64,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