"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const defaults = require("../");
const kinesis = require("@aws-cdk/aws-kinesis");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Test minimal deployment with no properties
// --------------------------------------------------------------
test('Test minimal deployment with no properties', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    defaults.buildKinesisStream(stack, {});
    expect(stack).toHaveResourceLike('AWS::Kinesis::Stream', {
        Type: "AWS::Kinesis::Stream",
        Properties: {
            StreamEncryption: {
                EncryptionType: "KMS"
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Test deployment w/ custom properties
// --------------------------------------------------------------
test('Test deployment w/ custom properties', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper setup
    const encKey = defaults.buildEncryptionKey(stack);
    // Helper declaration
    defaults.buildKinesisStream(stack, {
        kinesisStreamProps: {
            streamName: 'myCustomKinesisStream',
            encryption: kinesis.StreamEncryption.KMS,
            encryptionKey: encKey
        }
    });
    expect(stack).toHaveResource('AWS::Kinesis::Stream', {
        Name: 'myCustomKinesisStream'
    });
    // Assertion 3
    expect(stack).toHaveResource('AWS::KMS::Key', {
        EnableKeyRotation: true
    });
});
// --------------------------------------------------------------
// Test deployment w/ existing stream
// --------------------------------------------------------------
test('Test deployment w/ existing stream', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper setup
    const stream = new kinesis.Stream(stack, 'existing-stream', {
        shardCount: 2,
        retentionPeriod: core_1.Duration.days(3)
    });
    // Helper declaration
    defaults.buildKinesisStream(stack, {
        existingStreamObj: stream,
        // These props will be ignored as an existing stream was provided
        kinesisStreamProps: {
            shardCount: 1,
            retentionPeriod: core_1.Duration.days(1)
        }
    });
    expect(stack).toHaveResource('AWS::Kinesis::Stream', {
        ShardCount: 2,
        RetentionPeriodHours: 72
    });
});
test('Count Kinesis CW Alarms', () => {
    const stack = new core_1.Stack();
    const cwList = defaults.buildKinesisStreamCWAlarms(stack);
    expect(cwList.length).toEqual(2);
});
//# sourceMappingURL=data:application/json;base64,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