"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const defaults = require("../");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Test deployment with no properties using AWS Managed KMS Key
// --------------------------------------------------------------
test('Test deployment with no properties using AWS Managed KMS Key', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    defaults.buildTopic(stack, {});
    expect(stack).toHaveResource("AWS::SNS::Topic", {
        KmsMasterKeyId: {
            "Fn::Join": [
                "",
                [
                    "arn:",
                    {
                        Ref: "AWS::Partition"
                    },
                    ":kms:",
                    {
                        Ref: "AWS::Region"
                    },
                    ":",
                    {
                        Ref: "AWS::AccountId"
                    },
                    ":alias/aws/sns"
                ]
            ]
        }
    });
});
// --------------------------------------------------------------
// Test deployment without imported encryption key
// --------------------------------------------------------------
test('Test deployment without imported encryption key', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    defaults.buildTopic(stack, {
        topicProps: {
            topicName: "custom-topic"
        },
        enableEncryptionWithCustomerManagedKey: true
    });
    expect(stack).toHaveResource("AWS::SNS::Topic", {
        TopicName: "custom-topic"
    });
    // Assertion 3
    expect(stack).toHaveResource("AWS::KMS::Key", {
        EnableKeyRotation: true
    });
});
// --------------------------------------------------------------
// Test deployment w/ imported encryption key
// --------------------------------------------------------------
test('Test deployment w/ imported encryption key', () => {
    // Stack
    const stack = new core_1.Stack();
    // Generate KMS Key
    const key = defaults.buildEncryptionKey(stack);
    // Helper declaration
    defaults.buildTopic(stack, {
        topicProps: {
            topicName: "custom-topic"
        },
        enableEncryptionWithCustomerManagedKey: true,
        encryptionKey: key
    });
    expect(stack).toHaveResource("AWS::SNS::Topic", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "EncryptionKey1B843E66",
                "Arn"
            ]
        },
        TopicName: "custom-topic"
    });
});
test('Check SNS Topic policy', () => {
    const stack = new core_1.Stack();
    defaults.buildTopic(stack, {});
    assert_1.expect(stack).to(assert_1.haveResource("AWS::SNS::TopicPolicy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "SNS:Publish",
                        "SNS:RemovePermission",
                        "SNS:SetTopicAttributes",
                        "SNS:DeleteTopic",
                        "SNS:ListSubscriptionsByTopic",
                        "SNS:GetTopicAttributes",
                        "SNS:Receive",
                        "SNS:AddPermission",
                        "SNS:Subscribe"
                    ],
                    Condition: {
                        StringEquals: {
                            "AWS:SourceOwner": {
                                Ref: "AWS::AccountId"
                            }
                        }
                    },
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    {
                                        Ref: "AWS::Partition"
                                    },
                                    ":iam::",
                                    {
                                        Ref: "AWS::AccountId"
                                    },
                                    ":root"
                                ]
                            ]
                        }
                    },
                    Resource: {
                        Ref: "SnsTopic2C1570A4"
                    },
                    Sid: "TopicOwnerOnlyAccess"
                },
                {
                    Action: [
                        "SNS:Publish",
                        "SNS:RemovePermission",
                        "SNS:SetTopicAttributes",
                        "SNS:DeleteTopic",
                        "SNS:ListSubscriptionsByTopic",
                        "SNS:GetTopicAttributes",
                        "SNS:Receive",
                        "SNS:AddPermission",
                        "SNS:Subscribe"
                    ],
                    Condition: {
                        Bool: {
                            "aws:SecureTransport": "false"
                        }
                    },
                    Effect: "Deny",
                    Principal: {
                        AWS: "*"
                    },
                    Resource: {
                        Ref: "SnsTopic2C1570A4"
                    },
                    Sid: "HttpsOnly"
                }
            ],
            Version: "2012-10-17"
        },
    }));
});
//# sourceMappingURL=data:application/json;base64,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