# -*- coding: utf-8 -*-
"""
    sanic_dispatcher
    ~~~~

    :copyright: (c) 2017 by Ashley Sommer (based on DispatcherMiddleware in the Werkzeug Project).
    :license: MIT, see LICENSE for more details.
"""
from collections import defaultdict
from inspect import isawaitable
from io import BytesIO
from warnings import warn
try:
    from setuptools.extern import packaging
except ImportError:
    from pkg_resources.extern import packaging

from sanic import Sanic, __version__ as sanic_version
from sanic.exceptions import URLBuildError
from sanic.response import HTTPResponse, BaseHTTPResponse
from sanic.server import HttpProtocol
from sanic.websocket import WebSocketProtocol

SANIC_VERSION = packaging.version.parse(sanic_version)
SANIC_0_7_0 = packaging.version.parse('0.7.0')
if SANIC_VERSION < SANIC_0_7_0:
    raise RuntimeError("Please use Sanic v0.7.0 or greater with this extension.")
SANIC_19_03_0 = packaging.version.parse('19.3.0')
SANIC_18_12_0 = packaging.version.parse('18.12.0')
SANIC_21_03_0 = packaging.version.parse('21.03.0')
IS_19_03 = SANIC_VERSION >= SANIC_19_03_0
IS_18_12 = SANIC_VERSION >= SANIC_18_12_0
IS_21_03 = SANIC_VERSION >= SANIC_21_03_0
if IS_19_03:
    from sanic.request import RequestParameters
    from sanic.log import error_logger
else:
    import logging
    error_logger = logging.getLogger("sanic.error")
if IS_21_03:
    from sanic.compat import CancelledErrors

class WsgiApplication(object):
    __slots__ = ['app', 'apply_middleware']

    def __init__(self, app, apply_middleware=False):
        self.app = app
        self.apply_middleware = apply_middleware


class SanicApplication(object):
    __slots__ = ['app', 'server_settings', 'apply_middleware']

    def __init__(self, app, apply_middleware=False):
        self.app = app
        self.server_settings = {}
        self.apply_middleware = apply_middleware


class SanicCompatURL(object):
    """
    This class exists because the sanic native URL type is private (a non-exposed C module)
    and all of its components are read-only. We need to modify the URL path in the dispatcher
    so we build a feature-compatible writable class of our own to use instead.
    """
    __slots__ = ('schema', 'host', 'port', 'path', 'query', 'fragment', 'userinfo')

    def __init__(self, schema, host, port, path, query, fragment, userinfo):
        self.schema = schema
        self.host = host
        self.port = port
        self.path = path
        self.query = query
        self.fragment = fragment
        self.userinfo = userinfo

class SanicComatRequest(object):
    __slots__ = ("orig_req", "parent_app")
    def __init__(self, orig_req, parent_app):
        self.orig_req = orig_req
        self.parent_app = parent_app

    def __getattr__(self, item):
        if item in SanicComatRequest.__slots__:
            return object.__getattribute__(self, item)
        return getattr(self.orig_req, item)

    def __setattr__(self, key, val):
        if key in SanicComatRequest.__slots__:
            return object.__setattr__(self, key, val)
        return setattr(self.orig_req, key, val)

    async def respond(
            self,
            response=None,
            *args,
            status=200,
            headers=None,
            content_type=None,
    ):
        # From Sanic 21.03
        # This logic of determining which response to use is subject to change
        if response is None:
            response = (self.stream and self.stream.response) or HTTPResponse(
                status=status,
                headers=headers,
                content_type=content_type,
            )
        # Connect the response
        if isinstance(response, BaseHTTPResponse) and self.stream:
            response = self.stream.respond(response)
        # Run child's response middleware
        try:
            response = await self.app._run_response_middleware(
                self, response, request_name=self.name
            )
        except CancelledErrors:
            raise
        except Exception:
            error_logger.exception(
                "Exception occurred in one of response middleware handlers"
            )
        parent_app = self.parent_app
        if parent_app.response_middleware:
            self.orig_req.app = parent_app
            for _middleware in parent_app.response_middleware:
                _response = _middleware(self, response)
                if isawaitable(_response):
                    _response = await _response
                if _response:
                    response = _response
                    if isinstance(response, BaseHTTPResponse):
                        response = self.stream.respond(response)
        return response



class SanicDispatcherMiddleware(object):
    """
    A multi-application dispatcher, and also acts as a sanic-to-wsgiApp adapter.
    Based on the DispatcherMiddleware class in werkzeug.
    """

    __slots__ = ['parent_app', 'parent_handle_request', 'mounts', 'hosts']

    use_wsgi_threads = True

    def __init__(self, parent_app, parent_handle_request, mounts=None, hosts=None):
        self.parent_app = parent_app
        self.parent_handle_request = parent_handle_request
        self.mounts = mounts or {}
        self.hosts = frozenset(hosts) if hosts else frozenset()

    @staticmethod
    def _call_wsgi(script_name, path_info, request, wsgi_app, response_callback):
        http_response = None
        body_bytes = bytearray()

        def _start_response(status, headers, *args, **kwargs):
            """The start_response callback as required by the wsgi spec. This sets up a response including the
            status code and the headers, but doesn't write a body."""
            nonlocal http_response
            nonlocal body_bytes
            if isinstance(status, int):
                code = status
            elif isinstance(status, str):
                code = int(status.split(" ")[0])
            else:
                raise RuntimeError("status cannot be turned into a code.")
            sanic_headers = dict(headers)
            response_constructor_args = {'status': code,  'headers': sanic_headers}
            if 'content_type' in kwargs:
                response_constructor_args['content_type'] = kwargs['content_type']
            elif 'Content-Type' in sanic_headers:
                response_constructor_args['content_type'] = str(sanic_headers['Content-Type']).split(";")[0].strip()
            http_response = HTTPResponse(**response_constructor_args)

            def _write_body(body_data):
                """This doesn't seem to be used, but it is part of the wsgi spec, so need to have it."""
                nonlocal body_bytes
                nonlocal http_response
                if isinstance(body_data, bytes):
                    pass
                else:
                    try:
                        # Try to encode it regularly
                        body_data = body_data.encode()
                    except AttributeError:
                        # Convert it to a str if you can't
                        body_data = str(body_data).encode()
                body_bytes.extend(body_data)
            return _write_body

        environ = {}
        original_script_name = environ.get('SCRIPT_NAME', '')
        environ['SCRIPT_NAME'] = original_script_name + script_name
        environ['PATH_INFO'] = path_info
        if request._parsed_url and request._parsed_url.host is not None:
            host = request._parsed_url.host.decode('utf-8')
        elif 'host' in request.headers:
            host = request.headers['host']
        else:
            host = 'localhost:80'
        if 'content-type' in request.headers:
            content_type = request.headers['content-type']
        else:
            content_type = 'text/plain'
        environ['CONTENT_TYPE'] = content_type
        if 'content-length' in request.headers:
            content_length = request.headers['content-length']
            environ['CONTENT_LENGTH'] = content_length

        split_host = host.split(':', 1)
        host_has_port = len(split_host) > 1
        server_name = split_host[0]
        if request._parsed_url and request._parsed_url.port is not None:
            server_port = request._parsed_url.port.decode('ascii')
        elif host_has_port:
            server_port = split_host[1]
        else:
            server_port = '80'  # TODO: Find a better way of determining the port number when not provided
        if (not host_has_port) and (server_port != '80'):
            host = ":".join((host, server_port))
        environ['SERVER_PORT'] = server_port
        environ['SERVER_NAME'] = server_name
        environ['SERVER_PROTOCOL'] = 'HTTP/1.1' if request.version == "1.1" else 'HTTP/1.0'
        environ['HTTP_HOST'] = host
        environ['QUERY_STRING'] = request.query_string or ''
        environ['REQUEST_METHOD'] = request.method
        environ['wsgi.url_scheme'] = 'http'  # todo: detect http vs https
        environ['wsgi.input'] = BytesIO(request.body) if request.body is not None and len(request.body) > 0\
            else BytesIO(b'')
        try:
            wsgi_return = wsgi_app(environ, _start_response)
        except Exception as e:
            error_logger.exception(e)
            raise e
        if http_response is None:
            http_response = HTTPResponse("WSGI call error.", 500)
        else:
            for body_part in wsgi_return:
                if body_part is not None:
                    if isinstance(body_part, bytes):
                        pass
                    else:
                        try:
                            # Try to encode it regularly
                            body_part = body_part.encode()
                        except AttributeError:
                            # Convert it to a str if you can't
                            body_part = str(body_part).encode()
                    body_bytes.extend(body_part)
            http_response.body = bytes(body_bytes)
        if response_callback:
            return response_callback(http_response)
        return http_response

    if IS_21_03:
        async def call_wsgi_app(self, script_name, path_info, request, wsgi_app):

            if self.use_wsgi_threads:
                return await self.parent_app.loop.run_in_executor(None, self._call_wsgi, script_name, path_info, request, wsgi_app, False)
            else:
                return self._call_wsgi(script_name, path_info, request, wsgi_app, False)
    else:
        async def call_wsgi_app(self, script_name, path_info, request, wsgi_app, response_callback):
            if self.use_wsgi_threads:
                return await self.parent_app.loop.run_in_executor(
                    None, self._call_wsgi, script_name, path_info, request,
                    wsgi_app, response_callback)
            else:
                return self._call_wsgi(script_name, path_info, request, wsgi_app, response_callback)
    @staticmethod
    def get_request_scheme(request):
        try:
            if request.headers.get('upgrade') == 'websocket':
                scheme = b'ws'
            elif request.transport.get_extra_info('sslcontext'):
                scheme = b'https'
            else:
                scheme = b'http'
        except (AttributeError, KeyError):
            scheme = b'http'
        return scheme

    def _get_application_by_route(self, request, use_host=False):
        host = request.headers.get('Host', '')
        scheme = self.get_request_scheme(request)
        path = request._parsed_url.path
        port = request._parsed_url.port
        query_string = request._parsed_url.query
        fragment = request._parsed_url.fragment
        userinfo = request._parsed_url.userinfo
        script = path
        if ':' in host and port is None:
            (host, port) = host.split(':', 1)[0:2]
            port = port.encode('ascii')
        host_bytes = host.encode('utf-8')

        if use_host:
            script = b'%s%s' % (host_bytes, script)
        path_info = b''
        while b'/' in script:
            script_str = script.decode('utf-8')
            try:
                application = self.mounts[script_str]
                break
            except KeyError:
                pass
            script, last_item = script.rsplit(b'/', 1)
            path_info = b'/%s%s' % (last_item, path_info)
        else:
            script_str = script.decode('utf-8')
            application = self.mounts.get(script_str, None)
        if application is not None:
            request._parsed_url = SanicCompatURL(scheme, host_bytes, port,
                                                 path_info, query_string, fragment, userinfo)
            if IS_19_03:
                # To trigger re-parse args
                request.parsed_args = defaultdict(RequestParameters)
            else:
                request.parsed_args = None  # To trigger re-parse args
            path = request.path
        return application, script_str, path

    if IS_21_03:
        async def __call__(self, request):
            # Assume at this point that we have no app. So we cannot know if we are on Websocket or not.
            if self.hosts and len(self.hosts) > 0:
                application, script, path = self._get_application_by_route(request, use_host=True)
                if application is None:
                    application, script, path = self._get_application_by_route(request, use_host=False)
            else:
                application, script, path = self._get_application_by_route(request)
            if application is None:  # no child matches, call the parent
                return await self.parent_handle_request(request)
            return await self._call_2103(request, application, script, path)
    else:
        async def __call__(self, request, write_callback, stream_callback):
            # Assume at this point that we have no app. So we cannot know if we are on Websocket or not.
            if self.hosts and len(self.hosts) > 0:
                application, script, path = self._get_application_by_route(request, use_host=True)
                if application is None:
                    application, script, path = self._get_application_by_route(request, use_host=False)
            else:
                application, script, path = self._get_application_by_route(request)
            if application is None:  # no child matches, call the parent
                return await self.parent_handle_request(request, write_callback, stream_callback)
            return await self._call_old(request, application, script, path, write_callback, stream_callback)

    async def _call_old(self, request, application, script, path, write_callback, stream_callback):
        real_write_callback = write_callback
        real_stream_callback = stream_callback
        response = False
        streaming_response = False
        def _write_callback(child_response):
            nonlocal response
            response = child_response

        def _stream_callback(child_stream):
            nonlocal streaming_response
            streaming_response = child_stream

        replaced_write_callback = _write_callback
        replaced_stream_callback = _stream_callback
        parent_app = self.parent_app
        if application.apply_middleware and parent_app.request_middleware:
            request.app = parent_app
            for middleware in parent_app.request_middleware:
                response = middleware(request)
                if isawaitable(response):
                    response = await response
                if response:
                    break
        child_app = application.app
        if not response and not streaming_response:
            if isinstance(application, WsgiApplication):  # child is wsgi_app
                await self.call_wsgi_app(script, path, request,
                                         child_app, replaced_write_callback)
            else:  # must be a sanic application
                request.app = child_app
                await child_app.handle_request(request, replaced_write_callback, replaced_stream_callback)

        if application.apply_middleware and parent_app.response_middleware:
            request.app = parent_app
            for _middleware in parent_app.response_middleware:
                _response = _middleware(request, response)
                if isawaitable(_response):
                    _response = await _response
                if _response:
                    response = _response
                    break

        while isawaitable(response):
            response = await response
        if streaming_response:
            return real_stream_callback(streaming_response)
        return real_write_callback(response)


    async def _call_2103(self, request, application, script, path):

        our_response = False
        streaming_response = False

        if request.stream.request_body:  # type: ignore
            # Non-streaming handler: preload body
            await request.receive_body()

        parent_app = self.parent_app
        if application.apply_middleware and parent_app.request_middleware:
            request.app = parent_app
            for middleware in parent_app.request_middleware:
                our_response = middleware(request)
                if isawaitable(our_response):
                    our_response = await our_response
                if our_response:
                    break
        child_app = application.app
        if application.apply_middleware:
            request = SanicComatRequest(request, parent_app)
        if not our_response and not streaming_response:
            if isinstance(application, WsgiApplication):  # child is wsgi_app
                our_response = await self.call_wsgi_app(script, path, request, child_app)
            else:  # must be a sanic application
                request.app = child_app
                return await child_app.handle_request(request)
        if our_response is not None:
            try:
                our_response = await request.respond(our_response)
            except BaseException:
                # Skip response middleware
                if request.stream:
                    request.stream.respond(our_response)
                await our_response.send(end_stream=True)
                raise
        else:
            if request.stream:
                our_response = request.stream.response
        if isinstance(our_response, BaseHTTPResponse):
            await our_response.send(end_stream=True)
        return our_response


class SanicDispatcherMiddlewareController(object):
    __slots__ = ['parent_app', 'parent_handle_request', 'parent_url_for', 'applications', 'url_prefix',
                 'filter_host', 'hosts', 'started']

    def __init__(self, app, url_prefix=None, host=None):
        """
        :param Sanic app:
        :param url_prefix:
        """
        self.parent_app = app
        self.applications = {}
        self.url_prefix = None if url_prefix is None else str(url_prefix).rstrip('/')
        self.hosts = set()
        if host:
            self.filter_host = host
            self.hosts.add(host)
        else:
            self.filter_host = None
        self.started = False
        self.parent_app.register_listener(self._before_server_start_listener, 'before_server_start')
        self.parent_app.register_listener(self._after_server_start_listener, 'after_server_start')
        self.parent_app.register_listener(self._before_server_stop_listener, 'before_server_stop')
        self.parent_app.register_listener(self._after_server_stop_listener, 'after_server_stop')
        # Woo, monkey-patch!
        self.parent_handle_request = app.handle_request
        self.parent_url_for = app.url_for
        if IS_21_03:
            self.parent_app.handle_request = self.handle_request_2103
        else:
            self.parent_app.handle_request = self.handle_request
        self.parent_app.url_for = self.patched_url_for

    async def _before_server_start_listener(self, app, loop):
        if self.started is True:
            raise RuntimeError("Cannot start a sanic parent application more than once.")
        has_ws = False
        for route, child_app in self.applications.items():
            if isinstance(child_app, SanicApplication):
                s_app = child_app.app
                is_ws = getattr(s_app, 'websocket_enabled', False)
                protocol = WebSocketProtocol if is_ws else HttpProtocol
                server_settings = s_app._helper(host=None, port=None, loop=loop,
                                                protocol=protocol, run_async=False)
                child_app.server_settings = server_settings
                await s_app.trigger_events(
                    server_settings.get("before_start", []),
                    server_settings.get("loop"),
                )
                has_ws = has_ws or is_ws
        if not getattr(app, 'websocket_enabled', False) and has_ws:
            raise RuntimeError(
                "Found child apps with Websockets enabled, but parent app is not Websockets enabled.\n"
                "Add parent_app.enable_websocket() before starting the app.")

    async def _after_server_start_listener(self, app, loop):
        is_asgi = getattr(app, 'asgi', False)
        if is_asgi:
            error_logger.warning("Sanic-Dispatcher has not been tested on ASGI apps. It may not work correctly.")

        self.started = True

        for route, child_app in self.applications.items():
            if isinstance(child_app, SanicApplication):
                server_settings = child_app.server_settings
                await child_app.app.trigger_events(
                    server_settings.get("after_start", []),
                    server_settings.get("loop"),
                )

    async def _before_server_stop_listener(self, app, loop):
        for route, child_app in self.applications.items():
            if isinstance(child_app, SanicApplication):
                server_settings = child_app.server_settings
                await child_app.app.trigger_events(
                    server_settings.get("before_stop", []),
                    server_settings.get("loop"),
                )

    async def _after_server_stop_listener(self, app, loop):
        for route, child_app in self.applications.items():
            if isinstance(child_app, SanicApplication):
                server_settings = child_app.server_settings
                await child_app.app.trigger_events(
                    server_settings.get("after_stop", []),
                    server_settings.get("loop"),
                )

    def _determine_uri(self, url_prefix, host=None):
        uri = ''
        if self.url_prefix is not None:
            uri = self.url_prefix
        if host is not None:
            uri = str(host) + uri
            self.hosts.add(host)
        elif self.filter_host is not None:
            uri = str(self.filter_host) + uri
        uri += url_prefix
        return uri

    def register_app(self, app, url_prefix, host=None, apply_middleware=False):
        if isinstance(app, Sanic):
            self.register_sanic_application(app, url_prefix, host=host,
                                            apply_middleware=apply_middleware)
        else:
            self.register_wsgi_application(app, url_prefix, host=host,
                                           apply_middleware=apply_middleware)

    def register_sanic_application(self, application, url_prefix, host=None, apply_middleware=False):
        """
        :param Sanic application:
        :param url_prefix:
        :param host:
        :param apply_middleware:
        :return:
        """
        if self.started is True:
            raise warn("Registering an application when the server is already started may work,"
                       "but is not supported.")
        assert isinstance(application, Sanic),\
            "Pass only instances of Sanic to register_sanic_application."
        if str(url_prefix).endswith('/'):
            url_prefix = url_prefix[:-1]
        if host is not None and isinstance(host, (list, set)):
            for _host in host:
                self.register_sanic_application(application, url_prefix, host=_host,
                                                apply_middleware=apply_middleware)
            return

        registered_service_url = self._determine_uri(url_prefix, host)
        self.applications[registered_service_url] = SanicApplication(application, apply_middleware)
        self._update_request_handler()

    def register_wsgi_application(self, application, url_prefix, host=None, apply_middleware=False):
        """
        :param application:
        :param url_prefix:
        :param apply_middleware:
        :return:
        """
        if self.started is True:
            raise RuntimeError("Cannot register an application when the server is already started.")
        if str(url_prefix).endswith('/'):
            url_prefix = url_prefix[:-1]
        if host is not None and isinstance(host, (list, set)):
            for _host in host:
                self.register_wsgi_application(application, url_prefix, host=_host,
                                               apply_middleware=apply_middleware)
            return

        registered_service_url = self._determine_uri(url_prefix, host)
        self.applications[registered_service_url] = WsgiApplication(application, apply_middleware)
        self._update_request_handler()

    def unregister_application(self, application, all_matches=False):
        if isinstance(application, (SanicApplication, WsgiApplication)):
            application = application.app
        urls_to_unregister = []
        for url, reg_application in self.applications.items():
            if reg_application.app == application:
                urls_to_unregister.append(url)
                if not all_matches:
                    break
        for url in urls_to_unregister:
            del self.applications[url]
        self._update_request_handler()

    def unregister_prefix(self, url_prefix, host=None):
        if str(url_prefix).endswith('/'):
            url_prefix = url_prefix[:-1]
        if host is not None and isinstance(host, (list, set)):
            for _host in host:
                self.unregister_prefix(url_prefix, host=_host)
            return
        registered_service_url = self._determine_uri(url_prefix, host)
        try:
            del self.applications[registered_service_url]
        except KeyError:
            pass
        self._update_request_handler()

    def _update_request_handler(self):
        """
        Rebuilds the SanicDispatcherMiddleware every time a new application is registered
        :return:
        """
        dispatcher = SanicDispatcherMiddleware(self.parent_app, self.parent_handle_request, self.applications,
                                               self.hosts)
        self.parent_app.handle_request = dispatcher

    async def handle_request(self, request, write_callback, stream_callback):
        """
        This is only called as a backup handler if _update_request_handler was not yet called.
        :param request:
        :param write_callback:
        :param stream_callback:
        :return:
        """
        dispatcher = SanicDispatcherMiddleware(self.parent_app, self.parent_handle_request, self.applications,
                                               self.hosts)
        self.parent_app.handle_request = dispatcher  # save it for next time
        retval = dispatcher(request, write_callback, stream_callback)
        if isawaitable(retval):
            retval = await retval
        return retval

    async def handle_request_2103(self, request):
        """
        This is only called as a backup handler if _update_request_handler was not yet called.
        :param request:
        :return:
        """
        dispatcher = SanicDispatcherMiddleware(self.parent_app, self.parent_handle_request, self.applications,
                                               self.hosts)
        self.parent_app.handle_request = dispatcher  # save it for next time
        _ = await dispatcher(request)
        # This 2103 handler doesn't return anything

    def _dispatcher_url_for(self, view_name, **kwargs):
        """
        Checks all of the registered applications in the dispatcher for `url_for()`
        :param str view_name:
        :param kwargs:
        :return:
        """
        for url_prefix, reg_application in self.applications.items():
            app = reg_application.app
            try:
                _url_for = getattr(app, 'url_for', None)
                if _url_for is None:
                    continue
                try:
                    _url = _url_for(view_name, **kwargs)
                except URLBuildError:
                    continue
                if _url is not None:
                    return ''.join((url_prefix, str(_url)))
            except (AssertionError, KeyError):
                continue
        return None

    def patched_url_for(self, view_name, **kwargs):
        """
        When `url_for()` is called on the parent app, this gets called instead.
        :param str view_name:
        :param kwargs:
        :return:
        """
        try:
            url = self.parent_url_for(view_name, **kwargs)
        except URLBuildError:
            url = None
        if url is None:
            try:
                url = self._dispatcher_url_for(view_name, **kwargs)
            except URLBuildError:
                url = None
        if url is None:
            raise URLBuildError("Url Not found in the Parent App, nor the Dispatcher routes")

        return url

    def url_for(self, view_name, **kwargs):
        """
        When `url_for()` is called on the dispatcher app, this gets called instead.
        :param str view_name:
        :param kwargs:
        :return:
        """
        try:
            url = self._dispatcher_url_for(view_name, **kwargs)
        except URLBuildError:
            url = None
        if url is None:
            try:
                url = self.parent_url_for(view_name, **kwargs)
            except URLBuildError:
                url = None
        if url is None:
            raise URLBuildError("Url Not found in the Dispatcher routes, nor the Parent App")
        return url
