"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DomainName = exports.SecurityPolicy = void 0;
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const base_path_mapping_1 = require("./base-path-mapping");
const restapi_1 = require("./restapi");
/**
 * The minimum version of the SSL protocol that you want API Gateway to use for HTTPS connections.
 */
var SecurityPolicy;
(function (SecurityPolicy) {
    SecurityPolicy["TLS_1_0"] = "TLS_1_0";
    SecurityPolicy["TLS_1_2"] = "TLS_1_2";
})(SecurityPolicy = exports.SecurityPolicy || (exports.SecurityPolicy = {}));
/**
 *
 */
class DomainName extends core_1.Resource {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        const endpointType = props.endpointType || restapi_1.EndpointType.REGIONAL;
        const edge = endpointType === restapi_1.EndpointType.EDGE;
        if (!core_1.Token.isUnresolved(props.domainName) && /[A-Z]/.test(props.domainName)) {
            throw new Error('domainName does not support uppercase letters. ' +
                `got: '${props.domainName}'`);
        }
        const resource = new apigateway_generated_1.CfnDomainName(this, 'Resource', {
            domainName: props.domainName,
            certificateArn: edge ? props.certificate.certificateArn : undefined,
            regionalCertificateArn: edge ? undefined : props.certificate.certificateArn,
            endpointConfiguration: { types: [endpointType] },
            securityPolicy: props.securityPolicy,
        });
        this.domainName = resource.ref;
        this.domainNameAliasDomainName = edge
            ? resource.attrDistributionDomainName
            : resource.attrRegionalDomainName;
        this.domainNameAliasHostedZoneId = edge
            ? resource.attrDistributionHostedZoneId
            : resource.attrRegionalHostedZoneId;
        if (props.mapping) {
            this.addBasePathMapping(props.mapping);
        }
    }
    /**
     * Imports an existing domain name.
     */
    static fromDomainNameAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.domainName = attrs.domainName;
                this.domainNameAliasDomainName = attrs.domainNameAliasTarget;
                this.domainNameAliasHostedZoneId = attrs.domainNameAliasHostedZoneId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Maps this domain to an API endpoint.
     *
     * @param targetApi That target API endpoint, requests will be mapped to the deployment stage.
     * @param options Options for mapping to base path with or without a stage.
     */
    addBasePathMapping(targetApi, options = {}) {
        const basePath = options.basePath || '/';
        const id = `Map:${basePath}=>${targetApi.node.uniqueId}`;
        return new base_path_mapping_1.BasePathMapping(this, id, {
            domainName: this,
            restApi: targetApi,
            ...options,
        });
    }
}
exports.DomainName = DomainName;
//# sourceMappingURL=data:application/json;base64,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