"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaIntegration = void 0;
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const aws_1 = require("./aws");
/**
 * Integrates an AWS Lambda function to an API Gateway method.
 *
 * @example
 *
 *    const handler = new lambda.Function(this, 'MyFunction', ...);
 *    api.addMethod('GET', new LambdaIntegration(handler));
 */
class LambdaIntegration extends aws_1.AwsIntegration {
    /**
     *
     */
    constructor(handler, options = {}) {
        const proxy = options.proxy === undefined ? true : options.proxy;
        super({
            proxy,
            service: 'lambda',
            path: `2015-03-31/functions/${handler.functionArn}/invocations`,
            options,
        });
        this.handler = handler;
        this.enableTest = options.allowTestInvoke === undefined ? true : false;
    }
    /**
     * Can be overridden by subclasses to allow the integration to interact with the method being integrated, access the REST API object, method ARNs, etc.
     */
    bind(method) {
        const bindResult = super.bind(method);
        const principal = new iam.ServicePrincipal('apigateway.amazonaws.com');
        const desc = `${method.api.node.uniqueId}.${method.httpMethod}.${method.resource.path.replace(/\//g, '.')}`;
        this.handler.addPermission(`ApiPermission.${desc}`, {
            principal,
            scope: method,
            sourceArn: core_1.Lazy.stringValue({ produce: () => method.methodArn }),
        });
        // add permission to invoke from the console
        if (this.enableTest) {
            this.handler.addPermission(`ApiPermission.Test.${desc}`, {
                principal,
                scope: method,
                sourceArn: method.testMethodArn,
            });
        }
        let functionName;
        if (this.handler instanceof lambda.Function) {
            // if not imported, extract the name from the CFN layer to reach
            // the literal value if it is given (rather than a token)
            functionName = this.handler.node.defaultChild.functionName;
        }
        else {
            // imported, just take the function name.
            functionName = this.handler.functionName;
        }
        let deploymentToken;
        if (!core_1.Token.isUnresolved(functionName)) {
            deploymentToken = JSON.stringify({ functionName });
        }
        return {
            ...bindResult,
            deploymentToken,
        };
    }
}
exports.LambdaIntegration = LambdaIntegration;
//# sourceMappingURL=data:application/json;base64,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