# -*- coding: utf-8 -*-
# Copyright 2022 Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
from typing import Awaitable, Callable, Dict, Optional, Sequence, Tuple, Union
import warnings

from google.api_core import gapic_v1, grpc_helpers_async
from google.auth import credentials as ga_credentials  # type: ignore
from google.auth.transport.grpc import SslCredentials  # type: ignore
from google.protobuf import empty_pb2  # type: ignore
import grpc  # type: ignore
from grpc.experimental import aio  # type: ignore

from google.cloud.retail_v2.types import catalog
from google.cloud.retail_v2.types import catalog as gcr_catalog
from google.cloud.retail_v2.types import catalog_service

from .base import DEFAULT_CLIENT_INFO, CatalogServiceTransport
from .grpc import CatalogServiceGrpcTransport


class CatalogServiceGrpcAsyncIOTransport(CatalogServiceTransport):
    """gRPC AsyncIO backend transport for CatalogService.

    Service for managing catalog configuration.

    This class defines the same methods as the primary client, so the
    primary client can load the underlying transport implementation
    and call it.

    It sends protocol buffers over the wire using gRPC (which is built on
    top of HTTP/2); the ``grpcio`` package must be installed.
    """

    _grpc_channel: aio.Channel
    _stubs: Dict[str, Callable] = {}

    @classmethod
    def create_channel(
        cls,
        host: str = "retail.googleapis.com",
        credentials: Optional[ga_credentials.Credentials] = None,
        credentials_file: Optional[str] = None,
        scopes: Optional[Sequence[str]] = None,
        quota_project_id: Optional[str] = None,
        **kwargs,
    ) -> aio.Channel:
        """Create and return a gRPC AsyncIO channel object.
        Args:
            host (Optional[str]): The host for the channel to use.
            credentials (Optional[~.Credentials]): The
                authorization credentials to attach to requests. These
                credentials identify this application to the service. If
                none are specified, the client will attempt to ascertain
                the credentials from the environment.
            credentials_file (Optional[str]): A file with credentials that can
                be loaded with :func:`google.auth.load_credentials_from_file`.
                This argument is ignored if ``channel`` is provided.
            scopes (Optional[Sequence[str]]): A optional list of scopes needed for this
                service. These are only used when credentials are not specified and
                are passed to :func:`google.auth.default`.
            quota_project_id (Optional[str]): An optional project to use for billing
                and quota.
            kwargs (Optional[dict]): Keyword arguments, which are passed to the
                channel creation.
        Returns:
            aio.Channel: A gRPC AsyncIO channel object.
        """

        return grpc_helpers_async.create_channel(
            host,
            credentials=credentials,
            credentials_file=credentials_file,
            quota_project_id=quota_project_id,
            default_scopes=cls.AUTH_SCOPES,
            scopes=scopes,
            default_host=cls.DEFAULT_HOST,
            **kwargs,
        )

    def __init__(
        self,
        *,
        host: str = "retail.googleapis.com",
        credentials: Optional[ga_credentials.Credentials] = None,
        credentials_file: Optional[str] = None,
        scopes: Optional[Sequence[str]] = None,
        channel: Optional[aio.Channel] = None,
        api_mtls_endpoint: Optional[str] = None,
        client_cert_source: Optional[Callable[[], Tuple[bytes, bytes]]] = None,
        ssl_channel_credentials: Optional[grpc.ChannelCredentials] = None,
        client_cert_source_for_mtls: Optional[Callable[[], Tuple[bytes, bytes]]] = None,
        quota_project_id: Optional[str] = None,
        client_info: gapic_v1.client_info.ClientInfo = DEFAULT_CLIENT_INFO,
        always_use_jwt_access: Optional[bool] = False,
        api_audience: Optional[str] = None,
    ) -> None:
        """Instantiate the transport.

        Args:
            host (Optional[str]):
                 The hostname to connect to.
            credentials (Optional[google.auth.credentials.Credentials]): The
                authorization credentials to attach to requests. These
                credentials identify the application to the service; if none
                are specified, the client will attempt to ascertain the
                credentials from the environment.
                This argument is ignored if ``channel`` is provided.
            credentials_file (Optional[str]): A file with credentials that can
                be loaded with :func:`google.auth.load_credentials_from_file`.
                This argument is ignored if ``channel`` is provided.
            scopes (Optional[Sequence[str]]): A optional list of scopes needed for this
                service. These are only used when credentials are not specified and
                are passed to :func:`google.auth.default`.
            channel (Optional[aio.Channel]): A ``Channel`` instance through
                which to make calls.
            api_mtls_endpoint (Optional[str]): Deprecated. The mutual TLS endpoint.
                If provided, it overrides the ``host`` argument and tries to create
                a mutual TLS channel with client SSL credentials from
                ``client_cert_source`` or application default SSL credentials.
            client_cert_source (Optional[Callable[[], Tuple[bytes, bytes]]]):
                Deprecated. A callback to provide client SSL certificate bytes and
                private key bytes, both in PEM format. It is ignored if
                ``api_mtls_endpoint`` is None.
            ssl_channel_credentials (grpc.ChannelCredentials): SSL credentials
                for the grpc channel. It is ignored if ``channel`` is provided.
            client_cert_source_for_mtls (Optional[Callable[[], Tuple[bytes, bytes]]]):
                A callback to provide client certificate bytes and private key bytes,
                both in PEM format. It is used to configure a mutual TLS channel. It is
                ignored if ``channel`` or ``ssl_channel_credentials`` is provided.
            quota_project_id (Optional[str]): An optional project to use for billing
                and quota.
            client_info (google.api_core.gapic_v1.client_info.ClientInfo):
                The client info used to send a user-agent string along with
                API requests. If ``None``, then default info will be used.
                Generally, you only need to set this if you're developing
                your own client library.
            always_use_jwt_access (Optional[bool]): Whether self signed JWT should
                be used for service account credentials.

        Raises:
            google.auth.exceptions.MutualTlsChannelError: If mutual TLS transport
              creation failed for any reason.
          google.api_core.exceptions.DuplicateCredentialArgs: If both ``credentials``
              and ``credentials_file`` are passed.
        """
        self._grpc_channel = None
        self._ssl_channel_credentials = ssl_channel_credentials
        self._stubs: Dict[str, Callable] = {}

        if api_mtls_endpoint:
            warnings.warn("api_mtls_endpoint is deprecated", DeprecationWarning)
        if client_cert_source:
            warnings.warn("client_cert_source is deprecated", DeprecationWarning)

        if channel:
            # Ignore credentials if a channel was passed.
            credentials = False
            # If a channel was explicitly provided, set it.
            self._grpc_channel = channel
            self._ssl_channel_credentials = None
        else:
            if api_mtls_endpoint:
                host = api_mtls_endpoint

                # Create SSL credentials with client_cert_source or application
                # default SSL credentials.
                if client_cert_source:
                    cert, key = client_cert_source()
                    self._ssl_channel_credentials = grpc.ssl_channel_credentials(
                        certificate_chain=cert, private_key=key
                    )
                else:
                    self._ssl_channel_credentials = SslCredentials().ssl_credentials

            else:
                if client_cert_source_for_mtls and not ssl_channel_credentials:
                    cert, key = client_cert_source_for_mtls()
                    self._ssl_channel_credentials = grpc.ssl_channel_credentials(
                        certificate_chain=cert, private_key=key
                    )

        # The base transport sets the host, credentials and scopes
        super().__init__(
            host=host,
            credentials=credentials,
            credentials_file=credentials_file,
            scopes=scopes,
            quota_project_id=quota_project_id,
            client_info=client_info,
            always_use_jwt_access=always_use_jwt_access,
            api_audience=api_audience,
        )

        if not self._grpc_channel:
            self._grpc_channel = type(self).create_channel(
                self._host,
                # use the credentials which are saved
                credentials=self._credentials,
                # Set ``credentials_file`` to ``None`` here as
                # the credentials that we saved earlier should be used.
                credentials_file=None,
                scopes=self._scopes,
                ssl_credentials=self._ssl_channel_credentials,
                quota_project_id=quota_project_id,
                options=[
                    ("grpc.max_send_message_length", -1),
                    ("grpc.max_receive_message_length", -1),
                ],
            )

        # Wrap messages. This must be done after self._grpc_channel exists
        self._prep_wrapped_messages(client_info)

    @property
    def grpc_channel(self) -> aio.Channel:
        """Create the channel designed to connect to this service.

        This property caches on the instance; repeated calls return
        the same channel.
        """
        # Return the channel from cache.
        return self._grpc_channel

    @property
    def list_catalogs(
        self,
    ) -> Callable[
        [catalog_service.ListCatalogsRequest],
        Awaitable[catalog_service.ListCatalogsResponse],
    ]:
        r"""Return a callable for the list catalogs method over gRPC.

        Lists all the [Catalog][google.cloud.retail.v2.Catalog]s
        associated with the project.

        Returns:
            Callable[[~.ListCatalogsRequest],
                    Awaitable[~.ListCatalogsResponse]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "list_catalogs" not in self._stubs:
            self._stubs["list_catalogs"] = self.grpc_channel.unary_unary(
                "/google.cloud.retail.v2.CatalogService/ListCatalogs",
                request_serializer=catalog_service.ListCatalogsRequest.serialize,
                response_deserializer=catalog_service.ListCatalogsResponse.deserialize,
            )
        return self._stubs["list_catalogs"]

    @property
    def update_catalog(
        self,
    ) -> Callable[
        [catalog_service.UpdateCatalogRequest], Awaitable[gcr_catalog.Catalog]
    ]:
        r"""Return a callable for the update catalog method over gRPC.

        Updates the [Catalog][google.cloud.retail.v2.Catalog]s.

        Returns:
            Callable[[~.UpdateCatalogRequest],
                    Awaitable[~.Catalog]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "update_catalog" not in self._stubs:
            self._stubs["update_catalog"] = self.grpc_channel.unary_unary(
                "/google.cloud.retail.v2.CatalogService/UpdateCatalog",
                request_serializer=catalog_service.UpdateCatalogRequest.serialize,
                response_deserializer=gcr_catalog.Catalog.deserialize,
            )
        return self._stubs["update_catalog"]

    @property
    def set_default_branch(
        self,
    ) -> Callable[
        [catalog_service.SetDefaultBranchRequest], Awaitable[empty_pb2.Empty]
    ]:
        r"""Return a callable for the set default branch method over gRPC.

        Set a specified branch id as default branch. API methods such as
        [SearchService.Search][google.cloud.retail.v2.SearchService.Search],
        [ProductService.GetProduct][google.cloud.retail.v2.ProductService.GetProduct],
        [ProductService.ListProducts][google.cloud.retail.v2.ProductService.ListProducts]
        will treat requests using "default_branch" to the actual branch
        id set as default.

        For example, if ``projects/*/locations/*/catalogs/*/branches/1``
        is set as default, setting
        [SearchRequest.branch][google.cloud.retail.v2.SearchRequest.branch]
        to ``projects/*/locations/*/catalogs/*/branches/default_branch``
        is equivalent to setting
        [SearchRequest.branch][google.cloud.retail.v2.SearchRequest.branch]
        to ``projects/*/locations/*/catalogs/*/branches/1``.

        Using multiple branches can be useful when developers would like
        to have a staging branch to test and verify for future usage.
        When it becomes ready, developers switch on the staging branch
        using this API while keeping using
        ``projects/*/locations/*/catalogs/*/branches/default_branch`` as
        [SearchRequest.branch][google.cloud.retail.v2.SearchRequest.branch]
        to route the traffic to this staging branch.

        CAUTION: If you have live predict/search traffic, switching the
        default branch could potentially cause outages if the ID space
        of the new branch is very different from the old one.

        More specifically:

        -  PredictionService will only return product IDs from branch
           {newBranch}.
        -  SearchService will only return product IDs from branch
           {newBranch} (if branch is not explicitly set).
        -  UserEventService will only join events with products from
           branch {newBranch}.

        Returns:
            Callable[[~.SetDefaultBranchRequest],
                    Awaitable[~.Empty]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "set_default_branch" not in self._stubs:
            self._stubs["set_default_branch"] = self.grpc_channel.unary_unary(
                "/google.cloud.retail.v2.CatalogService/SetDefaultBranch",
                request_serializer=catalog_service.SetDefaultBranchRequest.serialize,
                response_deserializer=empty_pb2.Empty.FromString,
            )
        return self._stubs["set_default_branch"]

    @property
    def get_default_branch(
        self,
    ) -> Callable[
        [catalog_service.GetDefaultBranchRequest],
        Awaitable[catalog_service.GetDefaultBranchResponse],
    ]:
        r"""Return a callable for the get default branch method over gRPC.

        Get which branch is currently default branch set by
        [CatalogService.SetDefaultBranch][google.cloud.retail.v2.CatalogService.SetDefaultBranch]
        method under a specified parent catalog.

        Returns:
            Callable[[~.GetDefaultBranchRequest],
                    Awaitable[~.GetDefaultBranchResponse]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "get_default_branch" not in self._stubs:
            self._stubs["get_default_branch"] = self.grpc_channel.unary_unary(
                "/google.cloud.retail.v2.CatalogService/GetDefaultBranch",
                request_serializer=catalog_service.GetDefaultBranchRequest.serialize,
                response_deserializer=catalog_service.GetDefaultBranchResponse.deserialize,
            )
        return self._stubs["get_default_branch"]

    @property
    def get_completion_config(
        self,
    ) -> Callable[
        [catalog_service.GetCompletionConfigRequest],
        Awaitable[catalog.CompletionConfig],
    ]:
        r"""Return a callable for the get completion config method over gRPC.

        Gets a
        [CompletionConfig][google.cloud.retail.v2.CompletionConfig].

        Returns:
            Callable[[~.GetCompletionConfigRequest],
                    Awaitable[~.CompletionConfig]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "get_completion_config" not in self._stubs:
            self._stubs["get_completion_config"] = self.grpc_channel.unary_unary(
                "/google.cloud.retail.v2.CatalogService/GetCompletionConfig",
                request_serializer=catalog_service.GetCompletionConfigRequest.serialize,
                response_deserializer=catalog.CompletionConfig.deserialize,
            )
        return self._stubs["get_completion_config"]

    @property
    def update_completion_config(
        self,
    ) -> Callable[
        [catalog_service.UpdateCompletionConfigRequest],
        Awaitable[catalog.CompletionConfig],
    ]:
        r"""Return a callable for the update completion config method over gRPC.

        Updates the
        [CompletionConfig][google.cloud.retail.v2.CompletionConfig]s.

        Returns:
            Callable[[~.UpdateCompletionConfigRequest],
                    Awaitable[~.CompletionConfig]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "update_completion_config" not in self._stubs:
            self._stubs["update_completion_config"] = self.grpc_channel.unary_unary(
                "/google.cloud.retail.v2.CatalogService/UpdateCompletionConfig",
                request_serializer=catalog_service.UpdateCompletionConfigRequest.serialize,
                response_deserializer=catalog.CompletionConfig.deserialize,
            )
        return self._stubs["update_completion_config"]

    @property
    def get_attributes_config(
        self,
    ) -> Callable[
        [catalog_service.GetAttributesConfigRequest],
        Awaitable[catalog.AttributesConfig],
    ]:
        r"""Return a callable for the get attributes config method over gRPC.

        Gets an
        [AttributesConfig][google.cloud.retail.v2.AttributesConfig].

        Returns:
            Callable[[~.GetAttributesConfigRequest],
                    Awaitable[~.AttributesConfig]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "get_attributes_config" not in self._stubs:
            self._stubs["get_attributes_config"] = self.grpc_channel.unary_unary(
                "/google.cloud.retail.v2.CatalogService/GetAttributesConfig",
                request_serializer=catalog_service.GetAttributesConfigRequest.serialize,
                response_deserializer=catalog.AttributesConfig.deserialize,
            )
        return self._stubs["get_attributes_config"]

    @property
    def update_attributes_config(
        self,
    ) -> Callable[
        [catalog_service.UpdateAttributesConfigRequest],
        Awaitable[catalog.AttributesConfig],
    ]:
        r"""Return a callable for the update attributes config method over gRPC.

        Updates the
        [AttributesConfig][google.cloud.retail.v2.AttributesConfig].

        The catalog attributes in the request will be updated in the
        catalog, or inserted if they do not exist. Existing catalog
        attributes not included in the request will remain unchanged.
        Attributes that are assigned to products, but do not exist at
        the catalog level, are always included in the response. The
        product attribute is assigned default values for missing catalog
        attribute fields, e.g., searchable and dynamic facetable
        options.

        Returns:
            Callable[[~.UpdateAttributesConfigRequest],
                    Awaitable[~.AttributesConfig]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "update_attributes_config" not in self._stubs:
            self._stubs["update_attributes_config"] = self.grpc_channel.unary_unary(
                "/google.cloud.retail.v2.CatalogService/UpdateAttributesConfig",
                request_serializer=catalog_service.UpdateAttributesConfigRequest.serialize,
                response_deserializer=catalog.AttributesConfig.deserialize,
            )
        return self._stubs["update_attributes_config"]

    @property
    def add_catalog_attribute(
        self,
    ) -> Callable[
        [catalog_service.AddCatalogAttributeRequest],
        Awaitable[catalog.AttributesConfig],
    ]:
        r"""Return a callable for the add catalog attribute method over gRPC.

        Adds the specified
        [CatalogAttribute][google.cloud.retail.v2.CatalogAttribute] to
        the [AttributesConfig][google.cloud.retail.v2.AttributesConfig].

        If the
        [CatalogAttribute][google.cloud.retail.v2.CatalogAttribute] to
        add already exists, an ALREADY_EXISTS error is returned.

        Returns:
            Callable[[~.AddCatalogAttributeRequest],
                    Awaitable[~.AttributesConfig]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "add_catalog_attribute" not in self._stubs:
            self._stubs["add_catalog_attribute"] = self.grpc_channel.unary_unary(
                "/google.cloud.retail.v2.CatalogService/AddCatalogAttribute",
                request_serializer=catalog_service.AddCatalogAttributeRequest.serialize,
                response_deserializer=catalog.AttributesConfig.deserialize,
            )
        return self._stubs["add_catalog_attribute"]

    @property
    def remove_catalog_attribute(
        self,
    ) -> Callable[
        [catalog_service.RemoveCatalogAttributeRequest],
        Awaitable[catalog.AttributesConfig],
    ]:
        r"""Return a callable for the remove catalog attribute method over gRPC.

        Removes the specified
        [CatalogAttribute][google.cloud.retail.v2.CatalogAttribute] from
        the [AttributesConfig][google.cloud.retail.v2.AttributesConfig].

        If the
        [CatalogAttribute][google.cloud.retail.v2.CatalogAttribute] to
        remove does not exist, a NOT_FOUND error is returned.

        Returns:
            Callable[[~.RemoveCatalogAttributeRequest],
                    Awaitable[~.AttributesConfig]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "remove_catalog_attribute" not in self._stubs:
            self._stubs["remove_catalog_attribute"] = self.grpc_channel.unary_unary(
                "/google.cloud.retail.v2.CatalogService/RemoveCatalogAttribute",
                request_serializer=catalog_service.RemoveCatalogAttributeRequest.serialize,
                response_deserializer=catalog.AttributesConfig.deserialize,
            )
        return self._stubs["remove_catalog_attribute"]

    @property
    def replace_catalog_attribute(
        self,
    ) -> Callable[
        [catalog_service.ReplaceCatalogAttributeRequest],
        Awaitable[catalog.AttributesConfig],
    ]:
        r"""Return a callable for the replace catalog attribute method over gRPC.

        Replaces the specified
        [CatalogAttribute][google.cloud.retail.v2.CatalogAttribute] in
        the [AttributesConfig][google.cloud.retail.v2.AttributesConfig]
        by updating the catalog attribute with the same
        [CatalogAttribute.key][google.cloud.retail.v2.CatalogAttribute.key].

        If the
        [CatalogAttribute][google.cloud.retail.v2.CatalogAttribute] to
        replace does not exist, a NOT_FOUND error is returned.

        Returns:
            Callable[[~.ReplaceCatalogAttributeRequest],
                    Awaitable[~.AttributesConfig]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "replace_catalog_attribute" not in self._stubs:
            self._stubs["replace_catalog_attribute"] = self.grpc_channel.unary_unary(
                "/google.cloud.retail.v2.CatalogService/ReplaceCatalogAttribute",
                request_serializer=catalog_service.ReplaceCatalogAttributeRequest.serialize,
                response_deserializer=catalog.AttributesConfig.deserialize,
            )
        return self._stubs["replace_catalog_attribute"]

    def close(self):
        return self.grpc_channel.close()


__all__ = ("CatalogServiceGrpcAsyncIOTransport",)
