import logging
logger = logging.getLogger('key_profiles')

# Not intended to modify directly
_kp = {
    "krumhansl_kessler": [
        0.15195022732711172, 0.0533620483369227, 0.08327351040918879,
        0.05575496530270399, 0.10480976310122037, 0.09787030390045463,
        0.06030150753768843, 0.1241923905240488, 0.05719071548217276,
        0.08758076094759511, 0.05479779851639147, 0.06891600861450106,

        0.14221523253201526, 0.06021118849696697, 0.07908335205571781,
        0.12087171422152324, 0.05841383958660975, 0.07930802066951245,
        0.05706582790384183, 0.1067175915524601, 0.08941810829027184,
        0.06043585711076162, 0.07503931700741405, 0.07121995057290496
    ],
    "aarden_essen": [
        0.17766092893562843, 0.001456239417504233, 0.1492649402940239,
        0.0016018593592562562, 0.19804892078043168, 0.11358695456521818,
        0.002912478835008466, 0.2206199117520353, 0.001456239417504233,
        0.08154936738025305, 0.002329979068008373, 0.049512180195127924,

        0.18264800547944018, 0.007376190221285707, 0.14049900421497014,
        0.16859900505797015, 0.0070249402107482066, 0.14436200433086013,
        0.0070249402107482066, 0.18616100558483017, 0.04566210136986304,
        0.019318600579558018, 0.07376190221285707, 0.017562300526869017
    ],
    "sapp": [
        0.2222222222222222, 0.0, 0.1111111111111111, 0.0,
        0.1111111111111111, 0.1111111111111111, 0.0, 0.2222222222222222,
        0.0, 0.1111111111111111, 0.0, 0.1111111111111111,

        0.2222222222222222, 0.0, 0.1111111111111111, 0.1111111111111111,
        0.0, 0.1111111111111111, 0.0, 0.2222222222222222,
        0.1111111111111111, 0.0, 0.05555555555555555, 0.05555555555555555
    ],
    "bellman_budge": [
        0.168, 0.0086, 0.1295, 0.0141, 0.1349, 0.1193,
        0.0125, 0.2028, 0.018000000000000002, 0.0804, 0.0062, 0.1057,

        0.1816, 0.0069, 0.12990000000000002,
        0.1334, 0.010700000000000001, 0.1115,
        0.0138, 0.2107, 0.07490000000000001,
        0.015300000000000001, 0.0092, 0.10210000000000001
    ],
    "temperley": [
        0.17616580310880825, 0.014130946773433817, 0.11493170042392838,
        0.019312293923692884, 0.15779557230334432, 0.10833725859632594,
        0.02260951483749411, 0.16839378238341965, 0.02449364107395195,
        0.08619877531794629, 0.013424399434762127, 0.09420631182289213,

        0.1702127659574468, 0.020081281377002155, 0.1133158020559407,
        0.14774085584508725, 0.011714080803251255, 0.10996892182644036,
        0.02510160172125269, 0.1785799665311977, 0.09658140090843893,
        0.016017212526894576, 0.03179536218025341, 0.07889074826679417
    ],
    'albrecht_shanahan1': [
        0.238, 0.006, 0.111, 0.006, 0.137, 0.094,
        0.016, 0.214, 0.009, 0.080, 0.008, 0.081,

         0.220, 0.006, 0.104, 0.123, 0.019, 0.103,
         0.012, 0.214, 0.062, 0.022, 0.061, 0.052
    ],
    'albrecht_shanahan2': [
        0.21169, 0.00892766, 0.120448, 0.0100265, 0.131444, 0.0911768, 0.0215947, 0.204703, 0.012894, 0.0900445, 0.012617, 0.0844338,

        0.201933, 0.009335, 0.107284, 0.124169, 0.0199224, 0.108324,
        0.014314, 0.202699, 0.0653907, 0.0252515, 0.071959, 0.049419
    ],
    "experiment4": [
        0.17616580310880825, 0.014130946773433817, 0.11493170042392838, 0.019312293923692884, 0.15779557230334432, 0.10833725859632594, 0.02260951483749411, 0.17329251059821005, 0.01959491285916156, 0.08619877531794629, 0.013424399434762127, 0.09420631182289213,

        0.2222222222222222, 0.0, 0.1111111111111111, 0.12222222222222222,
        0.0, 0.1111111111111111, 0.008888888888888889, 0.2222222222222222, 0.1111111111111111, 0.0, 0.05555555555555555, 0.03555555555555555
    ],
    "experiment6": [
        0.13675520459440055, 0.0533620483369227, 0.08327351040918879,
        0.05575496530270399, 0.12000478583393154, 0.09787030390045463,
        0.06030150753768843, 0.1241923905240488, 0.05719071548217276,
        0.08758076094759511, 0.05479779851639147, 0.06891600861450106,

        0.14221523253201526, 0.06021118849696697, 0.07908335205571781,
        0.12087171422152324, 0.05841383958660975, 0.07930802066951245,
        0.05706582790384183, 0.1067175915524601, 0.08941810829027184,
        0.06043585711076162, 0.07503931700741405, 0.07121995057290496
    ],
    "napoles_midi128": [
        0.136755204594401, 0.065608997367791, 0.074946159368270,
        0.050179468772434, 0.100358937544867, 0.103445800430725,
        0.054271356783920, 0.139387413256760, 0.060904522613065,
        0.087580760947595, 0.049318018664752, 0.077243359655420,

        0.148236351381712, 0.048771062682543, 0.079083352055718,
        0.120871714221523, 0.047315210065154, 0.079308020669512,
        0.068950797573579, 0.096045832397214, 0.101959110312289,
        0.059649516962480, 0.085711076162660, 0.064097955515614
    ],
    "simple_natural_minor": [
        0.2222222222222222, 0.0, 0.1111111111111111, 0.0,
        0.1111111111111111, 0.1111111111111111, 0.0, 0.2222222222222222,
        0.0, 0.1111111111111111, 0.0, 0.1111111111111111,

        0.2222222222222222, 0.0, 0.1111111111111111, 0.1111111111111111,
        0.0, 0.1111111111111111, 0.0, 0.2222222222222222,
        0.1111111111111111, 0.0, 0.1111111111111111, 0.0
    ],
    "simple_harmonic_minor": [
        0.2222222222222222, 0.0, 0.1111111111111111, 0.0,
        0.1111111111111111, 0.1111111111111111, 0.0, 0.2222222222222222,
        0.0, 0.1111111111111111, 0.0, 0.1111111111111111,

        0.2222222222222222, 0.0, 0.1111111111111111, 0.1111111111111111,
        0.0, 0.1111111111111111, 0.0, 0.2222222222222222,
        0.1111111111111111, 0.0, 0.0, 0.1111111111111111
    ],
    "simple_melodic_minor": [
        0.2222222222222222, 0.0, 0.1111111111111111, 0.0,
        0.1111111111111111, 0.1111111111111111, 0.0, 0.2222222222222222,
        0.0, 0.1111111111111111, 0.0, 0.1111111111111111,

        0.2222222222222222, 0.0, 0.1111111111111111, 0.1111111111111111,
        0.0, 0.1111111111111111, 0.0, 0.2222222222222222,
        0.05555555555555555, 0.05555555555555555, 0.05555555555555555, 0.05555555555555555
    ],
}

def mix(major_name, minor_name):
    logger.info('Mixing {} and {}'.format(major_name, minor_name))
    if major_name not in _kp:
        logger.error('Could not find {}'.format(major_name))
        return
    if minor_name not in _kp:
        logger.error('Could not find {}'.format(minor_name))
        return
    kp_major = _kp[major_name]
    kp_minor = _kp[minor_name]
    kp_mixed = kp_major[:12] + kp_minor[12:]
    mix_name = '{}_{}'.format(major_name, minor_name)
    insert_new(mix_name, kp_mixed)
    return mix_name

def insert_new(name, kp):
    major_sum = sum(kp[:12])
    minor_sum = sum(kp[12:])
    if 1.0 - abs(major_sum) > 0.01:
        logger.warn('{} sum of major key profile is {} when it is expected to be 1.0'.format(name, major_sum))
    if 1.0 - abs(minor_sum) > 0.01:
        logger.warn('{} sum of minor key profile is {} when it is expected to be 1.0'.format(name, minor_sum))
    _kp[name] = kp
    logger.info('New key profile {}: {}'.format(name, kp))

def flush_key_profiles(key_profiles):
    [_kp.pop(kp) for kp in key_profiles]
    logger.info('Removed the following key profiles: {}'.format(key_profiles))

def keep_key_profiles(key_profiles):
    [_kp.pop(kp) for kp in list(_kp.keys()) if kp not in key_profiles]
    logger.info('Removed all but the following key profiles: {}'.format(key_profiles))

def log_key_profiles_dict():
    for k in _kp:
        logger.info('{}: {}'.format(k, get_as_string(k)))

def get(name):
    if name in _kp:
        return _kp[name]
    else:
        logger.error('{} key profile not found in the dictionary'.format(name))
        return None

def get_as_string(name):
    kp = get(name)
    if kp:
        kp_string = ['{:.15f}'.format(p) for p in kp]
        kp_string = ' '.join(kp_string)
        return kp_string
    return None
