import logging
logger = logging.getLogger('key_transitions')

# Not intended to modify directly
_kt = {
    "ktg_linear": [
        9/132, 4/132, 6/132, 6/132, 5/132, 8/132,
        1/132, 8/132, 5/132, 6/132, 6/132, 4/132,
        8/132, 2/132, 7/132, 3/132, 7/132, 7/132,
        3/132, 7/132, 2/132, 8/132, 5/132, 5/132,
    ],
    "ktg_exponential2": [
        256/1245, 8/1245, 32/1245, 32/1245, 16/1245, 128/1245,
        1/1245, 128/1245, 16/1245, 32/1245, 32/1245, 8/1245,
        128/1245, 2/1245, 64/1245, 4/1245, 64/1245, 64/1245,
        4/1245, 64/1245, 2/1245, 128/1245, 16/1245, 16/1245,
    ],
    "ktg_exponential4": [
        0.429054960883, 0.000418998985, 0.006703983764,
        0.006703983764, 0.001675995941, 0.107263740221,
        0.000006546859, 0.107263740221, 0.001675995941,
        0.006703983764, 0.006703983764, 0.000418998985,

        0.107263740221, 0.000026187437, 0.026815935055,
        0.000104749746, 0.026815935055, 0.026815935055,
        0.000104749746, 0.026815935055, 0.000026187437,
        0.107263740221, 0.001675995941, 0.001675995941
    ],
    "ktg_exponential5": [
        0.500201040802, 0.000160064333, 0.004001608326,
        0.004001608326, 0.000800321665, 0.100040208160,
        0.000001280515, 0.100040208160, 0.000800321665,
        0.004001608326, 0.004001608326, 0.000160064333,

        0.100040208160, 0.000006402573, 0.020008041632,
        0.000032012867, 0.020008041632, 0.020008041632,
        0.000032012867, 0.020008041632, 0.000006402573,
        0.100040208160, 0.000800321665, 0.000800321665
    ],
    "ktg_exponential10": [
        100000000/144442221, 1000/144442221, 100000/144442221,
        100000/144442221, 10000/144442221, 10000000/144442221,
        1/144442221, 10000000/144442221, 10000/144442221,
        100000/144442221, 100000/144442221, 1000/144442221,
        10000000/144442221, 10/144442221, 1000000/144442221,
        100/144442221, 1000000/144442221, 1000000/144442221,
        100/144442221, 1000000/144442221, 10/144442221,
        10000000/144442221, 10000/144442221, 10000/144442221
    ],
    "ktg_exponential12": [
        0.733338050205, 0.000002947121, 0.000424385446, 0.000424385446,
        0.000035365454, 0.061111504184, 0.000000001706, 0.061111504184,
        0.000035365454, 0.000424385446, 0.000424385446, 0.000002947121,

        0.061111504184, 0.000000020466, 0.005092625349, 0.000000245593,
        0.005092625349, 0.005092625349, 0.000000245593, 0.005092625349,
        0.000000020466, 0.061111504184, 0.000035365454, 0.000035365454,
    ],
    "ktg_exponential15": [
        0.777779485108, 0.000001024236, 0.000230453181,
        0.000230453181, 0.000015363545, 0.051851965674,
        0.000000000303, 0.051851965674, 0.000015363545,
        0.000230453181, 0.000230453181, 0.000001024236,

        0.051851965674, 0.000000004552, 0.003456797712,
        0.000000068282, 0.003456797712, 0.003456797712,
        0.000000068282, 0.003456797712, 0.000000004552,
        0.051851965674, 0.000015363545, 0.000015363545
    ],
    "ktg_exponential20": [
        0.826087405512, 0.000000258152, 0.000103260926,
        0.000103260926, 0.000005163046, 0.041304370276,
        0.000000000032, 0.041304370276, 0.000005163046,
        0.000103260926, 0.000103260926, 0.000000258152,

        0.041304370276, 0.000000000645, 0.002065218514,
        0.000000012908, 0.002065218514, 0.002065218514,
        0.000000012908, 0.002065218514, 0.000000000645,
        0.041304370276, 0.000005163046, 0.000005163046
    ],
    "ktg_experiment6": [
        0.8765246313562368, 3.993824473043542e-08, 3.450952918589394e-05, 3.450952918589394e-05, 1.1739889361311444e-06, 0.02981872670344273, 1.5724030710075827e-12, 0.02981872670344273, 1.1739889361311444e-06, 3.450952918589394e-05, 3.450952918589394e-05, 3.993824473043542e-08,

        0.02981872670344273, 4.622095490077414e-11, 0.0010144112674150716, 1.3586698673708549e-09, 0.0010144112674150716, 0.0010144112674150716, 1.3586698673708549e-09, 0.0010144112674150716, 4.622095490077414e-11, 0.02981872670344273, 1.1739889361311444e-06, 1.1739889361311444e-06
    ],
    'ktg_napoles_midi128': [
        0.7877447304, 0.0000007887, 0.0001980156, 0.0001980156,
        0.0000124969, 0.0497153048, 0.0000000002, 0.0497153048,
        0.0000124969, 0.0001980156, 0.0001980156, 0.0000007887,
        0.0497153048, 0.0000000031, 0.0031375793, 0.0000000498,
        0.0031375793, 0.0031375793, 0.0000000498, 0.0031375793,
        0.0000000031, 0.0497153048, 0.0000124969, 0.0000124969
    ],
    "null": [
        1.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    ]
}

_ratios = {
    "exponential2": 2,
    "exponential10": 10
}

def insert_new(name, kt):
    kt_sum = sum(kt)
    if 1.0 - abs(kt_sum) > 0.01:
        logger.warn('{} sum is {} when it is expected to be 1.0'.format(name, kt_sum))
    _kt[name] = kt
    logger.info('New key transition {}: {}'.format(name, kt))

def store_ratio(name, ratio):
    _ratios[name] = ratio
    logger.info('New ratio {}: {}'.format(name, ratio))

def flush_key_transitions(key_transitions):
    [_kt.pop(kt) for kt in key_transitions]
    [_ratios.pop(r) for r in key_transitions if r in _ratios]
    logger.info('Removed the following key transitions: {}'.format(key_transitions))

def keep_key_transitions(key_transitions):
    [_kt.pop(kt) for kt in list(_kt.keys()) if kt not in key_transitions]
    [_ratios.pop(r) for r in list(_ratios.keys()) if r not in key_transitions]
    logger.info('Removed all but the following key transitions: {}'.format(key_transitions))

def log_key_transitions_dict():
    logger.info('Current key transitions:')
    for k in _kt:
        logger.info('{}: {}'.format(k, get_as_string(k)))
        if k in _ratios:
            logger.info('ratio={}'.format(_ratios[k]))

def get(name):
    if name in _kt:
        return _kt[name]
    else:
        logger.error('{} key transition not found in the dictionary'.format(name))
        return None

def get_ratio(name):
    if name in _ratios:
        return _ratios[name]
    else:
        logger.error('{} kt ratio not found in the dictionary'.format(name))
        return None

def get_as_string(name):
    kt = get(name)
    if kt:
        kt_string = ['{:.15f}'.format(p) for p in kt]
        kt_string = ' '.join(kt_string)
        return kt_string
    return None
