"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const lambda = require("@aws-cdk/aws-lambda");
const sns = require("@aws-cdk/aws-sns");
const sqs = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const subs = require("../lib");
// tslint:disable:object-literal-key-quotes
let stack;
let topic;
beforeEach(() => {
    stack = new core_1.Stack();
    topic = new sns.Topic(stack, 'MyTopic', {
        topicName: 'topicName',
        displayName: 'displayName'
    });
});
test('url subscription', () => {
    topic.addSubscription(new subs.UrlSubscription('https://foobar.com/'));
    expect(stack).toMatchTemplate({
        "Resources": {
            "MyTopic86869434": {
                "Type": "AWS::SNS::Topic",
                "Properties": {
                    "DisplayName": "displayName",
                    "TopicName": "topicName"
                }
            },
            "MyTopichttpsfoobarcomDEA92AB5": {
                "Type": "AWS::SNS::Subscription",
                "Properties": {
                    "Endpoint": "https://foobar.com/",
                    "Protocol": "https",
                    "TopicArn": {
                        "Ref": "MyTopic86869434"
                    }
                }
            }
        }
    });
});
test('url subscription (with raw delivery)', () => {
    topic.addSubscription(new subs.UrlSubscription('https://foobar.com/', {
        rawMessageDelivery: true
    }));
    expect(stack).toMatchTemplate({
        "Resources": {
            "MyTopic86869434": {
                "Type": "AWS::SNS::Topic",
                "Properties": {
                    "DisplayName": "displayName",
                    "TopicName": "topicName"
                }
            },
            "MyTopichttpsfoobarcomDEA92AB5": {
                "Type": "AWS::SNS::Subscription",
                "Properties": {
                    "Endpoint": "https://foobar.com/",
                    "Protocol": "https",
                    "TopicArn": { "Ref": "MyTopic86869434" },
                    "RawMessageDelivery": true
                }
            }
        }
    });
});
test('url subscription (unresolved url with protocol)', () => {
    const urlToken = core_1.Token.asString({ Ref: "my-url-1" });
    topic.addSubscription(new subs.UrlSubscription(urlToken, { protocol: sns.SubscriptionProtocol.HTTPS }));
    expect(stack).toMatchTemplate({
        "Resources": {
            "MyTopic86869434": {
                "Type": "AWS::SNS::Topic",
                "Properties": {
                    "DisplayName": "displayName",
                    "TopicName": "topicName"
                }
            },
            "MyTopicTokenSubscription141DD1BE2": {
                "Type": "AWS::SNS::Subscription",
                "Properties": {
                    "Endpoint": {
                        "Ref": "my-url-1"
                    },
                    "Protocol": "https",
                    "TopicArn": { "Ref": "MyTopic86869434" },
                }
            }
        }
    });
});
test('url subscription (double unresolved url with protocol)', () => {
    const urlToken1 = core_1.Token.asString({ Ref: "my-url-1" });
    const urlToken2 = core_1.Token.asString({ Ref: "my-url-2" });
    topic.addSubscription(new subs.UrlSubscription(urlToken1, { protocol: sns.SubscriptionProtocol.HTTPS }));
    topic.addSubscription(new subs.UrlSubscription(urlToken2, { protocol: sns.SubscriptionProtocol.HTTPS }));
    expect(stack).toMatchTemplate({
        "Resources": {
            "MyTopic86869434": {
                "Type": "AWS::SNS::Topic",
                "Properties": {
                    "DisplayName": "displayName",
                    "TopicName": "topicName"
                }
            },
            "MyTopicTokenSubscription141DD1BE2": {
                "Type": "AWS::SNS::Subscription",
                "Properties": {
                    "Endpoint": {
                        "Ref": "my-url-1"
                    },
                    "Protocol": "https",
                    "TopicArn": { "Ref": "MyTopic86869434" },
                }
            },
            "MyTopicTokenSubscription293BFE3F9": {
                "Type": "AWS::SNS::Subscription",
                "Properties": {
                    "Endpoint": {
                        "Ref": "my-url-2"
                    },
                    "Protocol": "https",
                    "TopicArn": { "Ref": "MyTopic86869434" },
                }
            }
        }
    });
});
test('url subscription (unknown protocol)', () => {
    expect(() => topic.addSubscription(new subs.UrlSubscription('some-protocol://foobar.com/')))
        .toThrowError(/URL must start with either http:\/\/ or https:\/\//);
});
test('url subscription (unresolved url without protocol)', () => {
    const urlToken = core_1.Token.asString({ Ref: "my-url-1" });
    expect(() => topic.addSubscription(new subs.UrlSubscription(urlToken)))
        .toThrowError(/Must provide protocol if url is unresolved/);
});
test('queue subscription', () => {
    const queue = new sqs.Queue(stack, 'MyQueue');
    topic.addSubscription(new subs.SqsSubscription(queue));
    expect(stack).toMatchTemplate({
        "Resources": {
            "MyTopic86869434": {
                "Type": "AWS::SNS::Topic",
                "Properties": {
                    "DisplayName": "displayName",
                    "TopicName": "topicName"
                }
            },
            "MyQueueE6CA6235": {
                "Type": "AWS::SQS::Queue"
            },
            "MyQueuePolicy6BBEDDAC": {
                "Type": "AWS::SQS::QueuePolicy",
                "Properties": {
                    "PolicyDocument": {
                        "Statement": [
                            {
                                "Action": "sqs:SendMessage",
                                "Condition": {
                                    "ArnEquals": {
                                        "aws:SourceArn": {
                                            "Ref": "MyTopic86869434"
                                        }
                                    }
                                },
                                "Effect": "Allow",
                                "Principal": {
                                    "Service": "sns.amazonaws.com"
                                },
                                "Resource": {
                                    "Fn::GetAtt": [
                                        "MyQueueE6CA6235",
                                        "Arn"
                                    ]
                                }
                            }
                        ],
                        "Version": "2012-10-17"
                    },
                    "Queues": [
                        {
                            "Ref": "MyQueueE6CA6235"
                        }
                    ]
                }
            },
            "MyQueueMyTopic9B00631B": {
                "Type": "AWS::SNS::Subscription",
                "Properties": {
                    "Protocol": "sqs",
                    "TopicArn": {
                        "Ref": "MyTopic86869434"
                    },
                    "Endpoint": {
                        "Fn::GetAtt": [
                            "MyQueueE6CA6235",
                            "Arn"
                        ]
                    }
                }
            }
        }
    });
});
test('queue subscription (with raw delivery)', () => {
    const queue = new sqs.Queue(stack, 'MyQueue');
    topic.addSubscription(new subs.SqsSubscription(queue, { rawMessageDelivery: true }));
    expect(stack).toHaveResource('AWS::SNS::Subscription', {
        "Endpoint": {
            "Fn::GetAtt": [
                "MyQueueE6CA6235",
                "Arn"
            ]
        },
        "Protocol": "sqs",
        "TopicArn": {
            "Ref": "MyTopic86869434"
        },
        "RawMessageDelivery": true
    });
});
test('lambda subscription', () => {
    const fction = new lambda.Function(stack, 'MyFunc', {
        runtime: lambda.Runtime.NODEJS_10_X,
        handler: 'index.handler',
        code: lambda.Code.fromInline('exports.handler = function(e, c, cb) { return cb() }')
    });
    topic.addSubscription(new subs.LambdaSubscription(fction));
    expect(stack).toMatchTemplate({
        "Resources": {
            "MyTopic86869434": {
                "Type": "AWS::SNS::Topic",
                "Properties": {
                    "DisplayName": "displayName",
                    "TopicName": "topicName"
                }
            },
            "MyFuncServiceRole54065130": {
                "Type": "AWS::IAM::Role",
                "Properties": {
                    "AssumeRolePolicyDocument": {
                        "Statement": [
                            {
                                "Action": "sts:AssumeRole",
                                "Effect": "Allow",
                                "Principal": {
                                    "Service": "lambda.amazonaws.com"
                                }
                            }
                        ],
                        "Version": "2012-10-17"
                    },
                    "ManagedPolicyArns": [
                        {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    {
                                        "Ref": "AWS::Partition"
                                    },
                                    ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                                ]
                            ]
                        }
                    ]
                }
            },
            "MyFunc8A243A2C": {
                "Type": "AWS::Lambda::Function",
                "Properties": {
                    "Code": {
                        "ZipFile": "exports.handler = function(e, c, cb) { return cb() }"
                    },
                    "Handler": "index.handler",
                    "Role": {
                        "Fn::GetAtt": [
                            "MyFuncServiceRole54065130",
                            "Arn"
                        ]
                    },
                    "Runtime": "nodejs10.x"
                },
                "DependsOn": [
                    "MyFuncServiceRole54065130"
                ]
            },
            "MyFuncAllowInvokeMyTopicDD0A15B8": {
                "Type": "AWS::Lambda::Permission",
                "Properties": {
                    "Action": "lambda:InvokeFunction",
                    "FunctionName": {
                        "Fn::GetAtt": [
                            "MyFunc8A243A2C",
                            "Arn"
                        ]
                    },
                    "Principal": "sns.amazonaws.com",
                    "SourceArn": {
                        "Ref": "MyTopic86869434"
                    }
                }
            },
            "MyFuncMyTopic93B6FB00": {
                "Type": "AWS::SNS::Subscription",
                "Properties": {
                    "Protocol": "lambda",
                    "TopicArn": {
                        "Ref": "MyTopic86869434"
                    },
                    "Endpoint": {
                        "Fn::GetAtt": [
                            "MyFunc8A243A2C",
                            "Arn"
                        ]
                    }
                }
            }
        }
    });
});
test('email subscription', () => {
    topic.addSubscription(new subs.EmailSubscription('foo@bar.com'));
    expect(stack).toMatchTemplate({
        "Resources": {
            "MyTopic86869434": {
                "Type": "AWS::SNS::Topic",
                "Properties": {
                    "DisplayName": "displayName",
                    "TopicName": "topicName"
                }
            },
            "MyTopicfoobarcomA344CADA": {
                "Type": "AWS::SNS::Subscription",
                "Properties": {
                    "Endpoint": "foo@bar.com",
                    "Protocol": "email",
                    "TopicArn": {
                        "Ref": "MyTopic86869434"
                    }
                }
            }
        }
    });
});
test('email subscription with unresolved', () => {
    const emailToken = core_1.Token.asString({ Ref: "my-email-1" });
    topic.addSubscription(new subs.EmailSubscription(emailToken));
    expect(stack).toMatchTemplate({
        "Resources": {
            "MyTopic86869434": {
                "Type": "AWS::SNS::Topic",
                "Properties": {
                    "DisplayName": "displayName",
                    "TopicName": "topicName"
                }
            },
            "MyTopicTokenSubscription141DD1BE2": {
                "Type": "AWS::SNS::Subscription",
                "Properties": {
                    "Endpoint": {
                        "Ref": "my-email-1"
                    },
                    "Protocol": "email",
                    "TopicArn": {
                        "Ref": "MyTopic86869434"
                    }
                }
            }
        }
    });
});
test('email and url subscriptions with unresolved', () => {
    const emailToken = core_1.Token.asString({ Ref: "my-email-1" });
    const urlToken = core_1.Token.asString({ Ref: "my-url-1" });
    topic.addSubscription(new subs.EmailSubscription(emailToken));
    topic.addSubscription(new subs.UrlSubscription(urlToken, { protocol: sns.SubscriptionProtocol.HTTPS }));
    expect(stack).toMatchTemplate({
        "Resources": {
            "MyTopic86869434": {
                "Type": "AWS::SNS::Topic",
                "Properties": {
                    "DisplayName": "displayName",
                    "TopicName": "topicName"
                }
            },
            "MyTopicTokenSubscription141DD1BE2": {
                "Type": "AWS::SNS::Subscription",
                "Properties": {
                    "Endpoint": {
                        "Ref": "my-email-1"
                    },
                    "Protocol": "email",
                    "TopicArn": {
                        "Ref": "MyTopic86869434"
                    }
                }
            },
            "MyTopicTokenSubscription293BFE3F9": {
                "Type": "AWS::SNS::Subscription",
                "Properties": {
                    "Endpoint": {
                        "Ref": "my-url-1"
                    },
                    "Protocol": "https",
                    "TopicArn": {
                        "Ref": "MyTopic86869434"
                    }
                }
            }
        }
    });
});
test('email and url subscriptions with unresolved - four subscriptions', () => {
    const emailToken1 = core_1.Token.asString({ Ref: "my-email-1" });
    const emailToken2 = core_1.Token.asString({ Ref: "my-email-2" });
    const emailToken3 = core_1.Token.asString({ Ref: "my-email-3" });
    const emailToken4 = core_1.Token.asString({ Ref: "my-email-4" });
    topic.addSubscription(new subs.EmailSubscription(emailToken1));
    topic.addSubscription(new subs.EmailSubscription(emailToken2));
    topic.addSubscription(new subs.EmailSubscription(emailToken3));
    topic.addSubscription(new subs.EmailSubscription(emailToken4));
    expect(stack).toMatchTemplate({
        "Resources": {
            "MyTopic86869434": {
                "Type": "AWS::SNS::Topic",
                "Properties": {
                    "DisplayName": "displayName",
                    "TopicName": "topicName"
                }
            },
            "MyTopicTokenSubscription141DD1BE2": {
                "Type": "AWS::SNS::Subscription",
                "Properties": {
                    "Endpoint": {
                        "Ref": "my-email-1"
                    },
                    "Protocol": "email",
                    "TopicArn": {
                        "Ref": "MyTopic86869434"
                    }
                }
            },
            "MyTopicTokenSubscription293BFE3F9": {
                "Type": "AWS::SNS::Subscription",
                "Properties": {
                    "Endpoint": {
                        "Ref": "my-email-2"
                    },
                    "Protocol": "email",
                    "TopicArn": {
                        "Ref": "MyTopic86869434"
                    }
                }
            },
            "MyTopicTokenSubscription335C2B4CA": {
                "Type": "AWS::SNS::Subscription",
                "Properties": {
                    "Endpoint": {
                        "Ref": "my-email-3"
                    },
                    "Protocol": "email",
                    "TopicArn": {
                        "Ref": "MyTopic86869434"
                    }
                }
            },
            "MyTopicTokenSubscription4DBE52A3F": {
                "Type": "AWS::SNS::Subscription",
                "Properties": {
                    "Endpoint": {
                        "Ref": "my-email-4"
                    },
                    "Protocol": "email",
                    "TopicArn": {
                        "Ref": "MyTopic86869434"
                    }
                }
            }
        }
    });
});
test('multiple subscriptions', () => {
    const queue = new sqs.Queue(stack, 'MyQueue');
    const func = new lambda.Function(stack, 'MyFunc', {
        runtime: lambda.Runtime.NODEJS_10_X,
        handler: 'index.handler',
        code: lambda.Code.fromInline('exports.handler = function(e, c, cb) { return cb() }')
    });
    topic.addSubscription(new subs.SqsSubscription(queue));
    topic.addSubscription(new subs.LambdaSubscription(func));
    expect(stack).toMatchTemplate({
        "Resources": {
            "MyTopic86869434": {
                "Type": "AWS::SNS::Topic",
                "Properties": {
                    "DisplayName": "displayName",
                    "TopicName": "topicName"
                }
            },
            "MyQueueE6CA6235": {
                "Type": "AWS::SQS::Queue"
            },
            "MyQueuePolicy6BBEDDAC": {
                "Type": "AWS::SQS::QueuePolicy",
                "Properties": {
                    "PolicyDocument": {
                        "Statement": [
                            {
                                "Action": "sqs:SendMessage",
                                "Condition": {
                                    "ArnEquals": {
                                        "aws:SourceArn": {
                                            "Ref": "MyTopic86869434"
                                        }
                                    }
                                },
                                "Effect": "Allow",
                                "Principal": {
                                    "Service": "sns.amazonaws.com"
                                },
                                "Resource": {
                                    "Fn::GetAtt": [
                                        "MyQueueE6CA6235",
                                        "Arn"
                                    ]
                                }
                            }
                        ],
                        "Version": "2012-10-17"
                    },
                    "Queues": [
                        {
                            "Ref": "MyQueueE6CA6235"
                        }
                    ]
                }
            },
            "MyQueueMyTopic9B00631B": {
                "Type": "AWS::SNS::Subscription",
                "Properties": {
                    "Protocol": "sqs",
                    "TopicArn": {
                        "Ref": "MyTopic86869434"
                    },
                    "Endpoint": {
                        "Fn::GetAtt": [
                            "MyQueueE6CA6235",
                            "Arn"
                        ]
                    }
                }
            },
            "MyFuncServiceRole54065130": {
                "Type": "AWS::IAM::Role",
                "Properties": {
                    "AssumeRolePolicyDocument": {
                        "Statement": [
                            {
                                "Action": "sts:AssumeRole",
                                "Effect": "Allow",
                                "Principal": {
                                    "Service": "lambda.amazonaws.com"
                                }
                            }
                        ],
                        "Version": "2012-10-17"
                    },
                    "ManagedPolicyArns": [
                        {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    {
                                        "Ref": "AWS::Partition"
                                    },
                                    ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                                ]
                            ]
                        }
                    ]
                }
            },
            "MyFunc8A243A2C": {
                "Type": "AWS::Lambda::Function",
                "Properties": {
                    "Code": {
                        "ZipFile": "exports.handler = function(e, c, cb) { return cb() }"
                    },
                    "Handler": "index.handler",
                    "Role": {
                        "Fn::GetAtt": [
                            "MyFuncServiceRole54065130",
                            "Arn"
                        ]
                    },
                    "Runtime": "nodejs10.x"
                },
                "DependsOn": [
                    "MyFuncServiceRole54065130"
                ]
            },
            "MyFuncAllowInvokeMyTopicDD0A15B8": {
                "Type": "AWS::Lambda::Permission",
                "Properties": {
                    "Action": "lambda:InvokeFunction",
                    "FunctionName": {
                        "Fn::GetAtt": [
                            "MyFunc8A243A2C",
                            "Arn"
                        ]
                    },
                    "Principal": "sns.amazonaws.com",
                    "SourceArn": {
                        "Ref": "MyTopic86869434"
                    }
                }
            },
            "MyFuncMyTopic93B6FB00": {
                "Type": "AWS::SNS::Subscription",
                "Properties": {
                    "Protocol": "lambda",
                    "TopicArn": {
                        "Ref": "MyTopic86869434"
                    },
                    "Endpoint": {
                        "Fn::GetAtt": [
                            "MyFunc8A243A2C",
                            "Arn"
                        ]
                    }
                }
            }
        }
    });
});
test('throws with mutliple subscriptions of the same subscriber', () => {
    const queue = new sqs.Queue(stack, 'MyQueue');
    topic.addSubscription(new subs.SqsSubscription(queue));
    expect(() => topic.addSubscription(new subs.SqsSubscription(queue)))
        .toThrowError(/A subscription with id \"MyTopic\" already exists under the scope MyQueue/);
});
test('with filter policy', () => {
    const fction = new lambda.Function(stack, 'MyFunc', {
        runtime: lambda.Runtime.NODEJS_10_X,
        handler: 'index.handler',
        code: lambda.Code.fromInline('exports.handler = function(e, c, cb) { return cb() }')
    });
    topic.addSubscription(new subs.LambdaSubscription(fction, {
        filterPolicy: {
            color: sns.SubscriptionFilter.stringFilter({
                whitelist: ['red'],
                matchPrefixes: ['bl', 'ye'],
            }),
            size: sns.SubscriptionFilter.stringFilter({
                blacklist: ['small', 'medium'],
            }),
            price: sns.SubscriptionFilter.numericFilter({
                between: { start: 100, stop: 200 }
            })
        }
    }));
    expect(stack).toHaveResource('AWS::SNS::Subscription', {
        "FilterPolicy": {
            "color": [
                "red",
                {
                    "prefix": "bl"
                },
                {
                    "prefix": "ye"
                }
            ],
            "size": [
                {
                    "anything-but": [
                        "small",
                        "medium"
                    ]
                }
            ],
            "price": [
                {
                    "numeric": [
                        ">=",
                        100,
                        "<=",
                        200
                    ]
                }
            ]
        }
    });
});
test('region property is present on an imported topic', () => {
    const imported = sns.Topic.fromTopicArn(stack, 'mytopic', 'arn:aws:sns:us-east-1:1234567890:mytopic');
    const queue = new sqs.Queue(stack, 'myqueue');
    imported.addSubscription(new subs.SqsSubscription(queue));
    expect(stack).toHaveResource('AWS::SNS::Subscription', {
        Region: 'us-east-1'
    });
});
test('region property on an imported topic as a parameter', () => {
    const topicArn = new core_1.CfnParameter(stack, 'topicArn');
    const imported = sns.Topic.fromTopicArn(stack, 'mytopic', topicArn.valueAsString);
    const queue = new sqs.Queue(stack, 'myqueue');
    imported.addSubscription(new subs.SqsSubscription(queue));
    expect(stack).toHaveResource('AWS::SNS::Subscription', {
        Region: {
            "Fn::Select": [3, { "Fn::Split": [":", { "Ref": "topicArn" }] }]
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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