"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * A CodePipeline source action for BitBucket.
 *
 * @experimental
 */
class BitBucketSourceAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            category: codepipeline.ActionCategory.SOURCE,
            owner: 'AWS',
            provider: 'CodeStarSourceConnection',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        this.props = props;
    }
    bound(_scope, _stage, options) {
        var _a;
        // https://docs.aws.amazon.com/codepipeline/latest/userguide/security-iam.html#how-to-update-role-new-services
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: [
                'codestar-connections:UseConnection',
            ],
            resources: [
                this.props.connectionArn,
            ],
        }));
        // the action needs to write the output to the pipeline bucket
        options.bucket.grantReadWrite(options.role);
        // if codeBuildCloneOutput is true,
        // save the connectionArn in the Artifact instance
        // to be read by the CodeBuildAction later
        if (this.props.codeBuildCloneOutput === true) {
            this.props.output.setMetadata(BitBucketSourceAction._CONNECTION_ARN_PROPERTY, this.props.connectionArn);
        }
        return {
            configuration: {
                ConnectionArn: this.props.connectionArn,
                FullRepositoryId: `${this.props.owner}/${this.props.repo}`,
                BranchName: (_a = this.props.branch) !== null && _a !== void 0 ? _a : 'master',
                OutputArtifactFormat: this.props.codeBuildCloneOutput === true
                    ? 'CODEBUILD_CLONE_REF'
                    : undefined,
            },
        };
    }
}
exports.BitBucketSourceAction = BitBucketSourceAction;
/**
 * The name of the property that holds the ARN of the CodeStar Connection
 * inside of the CodePipeline Artifact's metadata.
 *
 * @internal
 */
BitBucketSourceAction._CONNECTION_ARN_PROPERTY = 'CodeStarConnectionArnProperty';
//# sourceMappingURL=data:application/json;base64,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