"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codepipeline = require("@aws-cdk/aws-codepipeline");
const targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * The ECR Repository source CodePipeline Action.
 *
 * Will trigger the pipeline as soon as the target tag in the repository
 * changes, but only if there is a CloudTrail Trail in the account that
 * captures the ECR event.
 */
class EcrSourceAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            resource: props.repository,
            category: codepipeline.ActionCategory.SOURCE,
            provider: 'ECR',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        this.props = props;
    }
    /** The variables emitted by this action. */
    get variables() {
        return {
            registryId: this.variableExpression('RegistryId'),
            repositoryName: this.variableExpression('RepositoryName'),
            imageDigest: this.variableExpression('ImageDigest'),
            imageTag: this.variableExpression('ImageTag'),
            imageUri: this.variableExpression('ImageURI'),
        };
    }
    bound(_scope, stage, options) {
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['ecr:DescribeImages'],
            resources: [this.props.repository.repositoryArn],
        }));
        this.props.repository.onCloudTrailImagePushed(stage.pipeline.node.uniqueId + 'SourceEventRule', {
            target: new targets.CodePipeline(stage.pipeline),
            imageTag: this.props.imageTag,
        });
        // the Action Role also needs to write to the Pipeline's bucket
        options.bucket.grantWrite(options.role);
        return {
            configuration: {
                RepositoryName: this.props.repository.repositoryName,
                ImageTag: this.props.imageTag,
            },
        };
    }
}
exports.EcrSourceAction = EcrSourceAction;
//# sourceMappingURL=data:application/json;base64,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