"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/// !cdk-integ PipelineStack
const codebuild = require("@aws-cdk/aws-codebuild");
const codecommit = require("@aws-cdk/aws-codecommit");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const codepipeline_actions = require("../lib");
const app = new cdk.App();
/// !show
const lambdaStack = new cdk.Stack(app, 'LambdaStack');
const lambdaCode = lambda.Code.fromCfnParameters();
new lambda.Function(lambdaStack, 'Lambda', {
    code: lambdaCode,
    handler: 'index.handler',
    runtime: lambda.Runtime.NODEJS_10_X,
});
// other resources that your Lambda needs, added to the lambdaStack...
const pipelineStack = new cdk.Stack(app, 'PipelineStack');
const pipeline = new codepipeline.Pipeline(pipelineStack, 'Pipeline');
// add the source code repository containing this code to your Pipeline,
// and the source code of the Lambda Function, if they're separate
const cdkSourceOutput = new codepipeline.Artifact();
const cdkSourceAction = new codepipeline_actions.CodeCommitSourceAction({
    repository: new codecommit.Repository(pipelineStack, 'CdkCodeRepo', {
        repositoryName: 'CdkCodeRepo',
    }),
    actionName: 'CdkCode_Source',
    output: cdkSourceOutput,
});
const lambdaSourceOutput = new codepipeline.Artifact();
const lambdaSourceAction = new codepipeline_actions.CodeCommitSourceAction({
    repository: new codecommit.Repository(pipelineStack, 'LambdaCodeRepo', {
        repositoryName: 'LambdaCodeRepo',
    }),
    actionName: 'LambdaCode_Source',
    output: lambdaSourceOutput,
});
pipeline.addStage({
    stageName: 'Source',
    actions: [cdkSourceAction, lambdaSourceAction],
});
// synthesize the Lambda CDK template, using CodeBuild
// the below values are just examples, assuming your CDK code is in TypeScript/JavaScript -
// adjust the build environment and/or commands accordingly
const cdkBuildProject = new codebuild.Project(pipelineStack, 'CdkBuildProject', {
    environment: {
        buildImage: codebuild.LinuxBuildImage.UBUNTU_14_04_NODEJS_10_1_0,
    },
    buildSpec: codebuild.BuildSpec.fromObject({
        version: '0.2',
        phases: {
            install: {
                commands: 'npm install',
            },
            build: {
                commands: [
                    'npm run build',
                    'npm run cdk synth LambdaStack -- -o .',
                ],
            },
        },
        artifacts: {
            files: 'LambdaStack.template.yaml',
        },
    }),
});
const cdkBuildOutput = new codepipeline.Artifact();
const cdkBuildAction = new codepipeline_actions.CodeBuildAction({
    actionName: 'CDK_Build',
    project: cdkBuildProject,
    input: cdkSourceOutput,
    outputs: [cdkBuildOutput],
});
// build your Lambda code, using CodeBuild
// again, this example assumes your Lambda is written in TypeScript/JavaScript -
// make sure to adjust the build environment and/or commands if they don't match your specific situation
const lambdaBuildProject = new codebuild.Project(pipelineStack, 'LambdaBuildProject', {
    environment: {
        buildImage: codebuild.LinuxBuildImage.UBUNTU_14_04_NODEJS_10_1_0,
    },
    buildSpec: codebuild.BuildSpec.fromObject({
        version: '0.2',
        phases: {
            install: {
                commands: 'npm install',
            },
            build: {
                commands: 'npm run build',
            },
        },
        artifacts: {
            files: [
                'index.js',
                'node_modules/**/*',
            ],
        },
    }),
});
const lambdaBuildOutput = new codepipeline.Artifact();
const lambdaBuildAction = new codepipeline_actions.CodeBuildAction({
    actionName: 'Lambda_Build',
    project: lambdaBuildProject,
    input: lambdaSourceOutput,
    outputs: [lambdaBuildOutput],
});
pipeline.addStage({
    stageName: 'Build',
    actions: [cdkBuildAction, lambdaBuildAction],
});
// finally, deploy your Lambda Stack
pipeline.addStage({
    stageName: 'Deploy',
    actions: [
        new codepipeline_actions.CloudFormationCreateUpdateStackAction({
            actionName: 'Lambda_CFN_Deploy',
            templatePath: cdkBuildOutput.atPath('LambdaStack.template.yaml'),
            stackName: 'LambdaStackDeployedName',
            adminPermissions: true,
            parameterOverrides: {
                ...lambdaCode.assign(lambdaBuildOutput.s3Location),
            },
            extraInputs: [
                lambdaBuildOutput,
            ],
        }),
    ],
});
//# sourceMappingURL=data:application/json;base64,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