# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['fawltydeps']

package_data = \
{'': ['*']}

install_requires = \
['importlib_metadata>=6.0.0,<7.0.0',
 'isort>=5.10,<6.0',
 'pip-requirements-parser>=32.0.1,<33.0.0',
 'pydantic>=1.10.4,<2.0.0']

extras_require = \
{':python_version < "3.11"': ['tomli>=2.0.1,<3.0.0'],
 ':python_version < "3.8"': ['typing-extensions>=4.4.0,<5.0.0']}

entry_points = \
{'console_scripts': ['fawltydeps = fawltydeps.main:main']}

setup_kwargs = {
    'name': 'fawltydeps',
    'version': '0.7.0',
    'description': 'Find undeclared and unused 3rd-party dependencies in your Python project.',
    'long_description': '[![PyPI Latest Release](https://img.shields.io/pypi/v/fawltydeps.svg)](https://pypi.org/project/fawltydeps/) [![Supported Python versions](https://img.shields.io/pypi/pyversions/fawltydeps.svg)](https://pypi.org/project/fawltydeps/) ![Build](https://img.shields.io/github/actions/workflow/status/tweag/fawltydeps/ci.yaml) [![Licence](https://img.shields.io/pypi/l/fawltydeps.svg)](https://pypi.org/project/fawltydeps/)\n# FawltyDeps\n\nFawltyDeps is a dependency checker for Python that finds _undeclared_ and/or\n_unused_ 3rd-party dependencies in your Python project.\nThe name is inspired by the Monty Python-adjacent\n[Fawlty Towers](https://en.wikipedia.org/wiki/Fawlty_Towers) sitcom.\n\n![FawltyDeps demo](https://github.com/tweag/FawltyDeps/raw/main/docs/fawltydeps_demo_tqdm.gif)\n\n## Table of contents\n\n[Key Concepts](#key-concepts)\n\n[Installation](#installation)\n\n[Usage](#usage)\n\n[Configuration](#configuration)\n\n[Documentation](#documentation)\n\n[Development](#development)\n\n[Integration tests](#integration-tests)\n\n[Contributing](#contributing)\n\n[FAQ](#faq)\n\n## Key Concepts\n\n- **_undeclared_ dependency**: a package that\'s used (in particular, `import`ed) by a project and which lacks a corresponding declaration to ensure that it\'s available.\n  For example, you `import numpy`, but you\'ve forgotten to include `numpy` in your `requirements.txt`.\n  Pragmatically, this means the project is prone to runtime errors.\n- **_unused_ dependency**: a package that\'s declared as necessary for a project but which is never used by project code.\n  For example, you have `numpy` listed in your `requirements.txt`, but you never actually `import numpy`.\n  Pragmatically, this means that project installation may consume more space than needed and will be more likely to break with future software releases; in short, these are costs paid for no benefit.\n\n## Installation\n\nThe library is distributed with PyPI, so simply:\n\n```\npip install fawltydeps\n```\n\nor any other way to install Python packages from PyPI should be enough to make it available in your environment.\n\nConsider adding `fawltydeps` to your development dependencies, to help you catch undeclared and unused dependencies in your projects.\n\n## Usage\n\nTo check the project in the current directory run:\n\n```\nfawltydeps\n```\n\nThis will find imports in all the Python code under the current directory,\nextract dependencies declared by your project, and then report\n[_undeclared_ and _unused_ dependencies](#key-concepts).\n\n### Available Actions\n\nFawltyDeps provides the following options for controlling what actions to perform. Only\none of these can be used at a time:\n\n- `--check`: Report both undeclared and unused dependencies\n- `--check-undeclared`: Report only undeclared dependencies\n- `--check-unused`: Report only unused dependencies\n- `--list-imports`: List third-party imports extracted from the project\n- `--list-deps`: List declared dependencies extracted from the project\n\nWhen none of these are specified, the default action is `--check`.\n\n### Where to find code and dependency declarations\n\nBy default, FawltyDeps will look for Python code (`*.py` and `*.ipynb`) and\ndependency declarations (see list of supported files below) under the current\ndirectory. If you want FawltyDeps to look elsewhere, you can pass a different\ndirectory (aka `basepath`) as a positional argument:\n\n```\nfawltydeps my_project/\n```\n\nIf you want to separately declare the source of the code and the source of the\ndependencies, you may use the `--code` and `--deps` options documented in the\nnext section. In short, giving the `basepath` positional argument is equivalent\nto passing both the `--code` and the `--deps` options, like this:\n\n```\nfawltydeps --code my_project/ --deps my_project/\n```\n\n#### Where to find Python code\n\nThe `--code` option tells FawltyDeps where to find the Python code to parse for\n`import` statements. You can pass any number of these:\n\n- a single file: Either a Python file (`*.py`) or a Jupyter Notebook (`*.ipynb`)\n- a directory: FawltyDeps will find all Python files and Jupyter notebooks under this directory.\n- `-`: Passing a single dash (`--code=-`) tells FawltyDeps to read Python code\n  from stdin.\n\nIf no `--code` option is passed, FawltyDeps will find all Python code under the\n`basepath`, if given, or the current directory (i.e. same as `--code=.`).\nTo include both code from stdin (`import foo`) and a file path (`file.py`), use:\n\n```\necho "import foo" | fawltydeps --list-imports --code - file.py\n```\n\n#### Where to find declared dependencies\n\nThe `--deps` option tells FawltyDeps where to look for your project\'s declared\ndependencies. A number of file formats are supported:\n\n- `*requirements*.txt` and `*requirements*.in`\n- `pyproject.toml` (following PEP 621 or Poetry conventions)\n- `setup.py` (only limited support for simple files with a single `setup()`\n  call and no computation involved for setting the `install_requires` and\n  `extras_require` arguments)\n- `setup.cfg`\n\nThe `--deps` option accepts a space-separated list of files or directories.\nEach file will be parsed for declared dependencies; each directory will\nbe searched, parsing all of the supported files (see the above list) found\nwithin. You would typically want to pass individual files, if you want to\nbe explicit about where to find the declared dependencies.\n\nIf no `--deps` option is passed, FawltyDeps will look for the above files under\nthe `basepath`, if given, or the current directory (i.e. same as `--deps .`).\n\n### Resolving dependencies via your Python environment\n\nWhen FawltyDeps looks for undeclared and unused dependencies, it needs to match\n`import` statements in your code with corresponding package dependencies\ndeclared in your project configuration.\n\nTo solve this, FawltyDeps looks at the packages installed in your _current\nPython environment_ and what import names each of them provide in order to\ncorrectly match your dependencies against your imports.\n\nThe _current Python environment_ in this case is the environment in which\nFawltyDeps itself is installed. This works well when you, for example,\n`pip install fawltydeps` into the same virtualenv as your project dependencies.\n\nIf you instead want FawltyDeps to look into a _different_ Python environment for\nmapping dependencies to import names, you can use the `--pyenv` option,\nfor example:\n\n```\nfawltydeps --code my_package/ --deps pyproject.toml --pyenv .venv/\n```\n\nThis will tell FawltyDeps:\n\n- to look for `import` statements in the `my_package/` directory,\n- to parse dependencies from `pyprojects.toml`, and\n- to use the Python environment at `.venv/` to map dependency names in\n  `pyproject.toml` into import names used in your code under `my_package/`\n\nWhen FawltyDeps is unable to find an installed package that corresponds to a\ndeclared dependency, FawltyDeps will fall back to an "identity mapping", where\nit _assumes_ that the dependency provides a single import of the same name,\ni.e. it will expect that when you depend on `some_package`, then that should\ncorrespond to `import some_package` statements in your code.\n\nThis fallback assumption is not always correct, but it allows FawltyDeps to\nproduce results (albeit sometimes inaccurate) when the current Python\nenvironment does not contain all of your declared dependencies. Please see\nFAQ below about [why FawltyDeps must run in the same Python environment as your\nproject dependencies](#why-must-fawltydeps-run-in-the-same-python-environment-as-my-project-dependencies).\n\n### Ignoring irrelevant results\n\nThere may be `import` statements in your code that should not be considered an\nundeclared dependency. This might happen if you for example do a conditional\n`import` with a `try: ... except ImportError: ...` block (or similar).\nFawltyDeps is not able to recognize whether these dependencies should have been\ndeclared or not, but you can ask for them to be ignored with the\n`--ignore-undeclared` option, for example:\n`--ignore-undeclared some_module some_other_module`\n\nConversely, there may be dependencies that you have declared without intending\nto `import` them. This is often the case for developer tools like Black or Mypy\nthat are part of your project\'s development environment.\nFawltyDeps cannot automatically tell which of your declared dependencies are\nmeant to be `import`ed or not, but you ask for specific deps to be ignored with\nthe `--ignore-unused` option, for example:\n`--ignore-unused black mypy`\n\n### Output formats\n\nThe default output from FawltyDeps is a summary outlining the relevant\ndependencies found (according to the selected actions).\nHowever you can also ask for more information from FawltyDeps:\n\n- `--summary`: Default (human-readable) summary output\n- `--detailed`: Longer (human-readable) output that includes the location of\n  the relevant dependencies.\n- `--json`: Verbose JSON-formatted output for other tools to consume and\n  process further.\n\nOnly one of these options can be used at a time.\n\n### More help\n\nRun `fawltydeps --help` to get the full list of available options.\n\n## Configuration\n\nYou can use a `[tool.fawltydeps]` section in `pyproject.toml` to configure the\ndefault behavior of FawltyDeps. Here\'s a fairly comprehensive example:\n\n```toml\n[tool.fawltydeps]\ncode = ["myproject"]  # Only search for imports under ./myproject\ndeps = ["pyproject.toml"]  # Only look for declared dependencies here\nignore_unused = ["black"]  # We use `black`, but we don\'t intend to import it\noutput_format = "human_detailed"  # Detailed report by default\n```\n\nHere is a complete list of configuration directives we support:\n\n- `actions`: A list of one or more of these actions to perform: `list_imports`,\n  `list_deps`, `check_undeclared`, `check_unused`. The default behavior\n  corresponds to `actions = ["check_undeclared", "check_unused"]`.\n- `code`: Files or directories containing the code to parse for import statements.\n  Defaults to the current directory, i.e. like `code = ["."]`.\n- `deps`: Files or directories containing the declared dependencies.\n  Defaults to the current directory, i.e. like `deps = ["."]`.\n- `pyenv`: The path to the Python environment to use for resolving project\n  dependencies to provided import names. Defaults to the Python environment\n  where FawltyDeps is installed.\n- `output_format`: Which output format to use by default. One of `human_summary`,\n  `human_detailed`, or `json`.\n  The default corresponds to `output_format = "human_summary"`.\n- `ignore_undeclared`: A list of specific dependencies to ignore when reporting\n  undeclared dependencies, for example: `["some_module", "some_other_module"]`.\n  The default is the empty list: `ignore_undeclared = []`.\n- `ignore_unused`: A list of specific dependencies to ignore when reporting\n  unused dependencies, for example: `["black", "mypy"]`.\n  The default is the empty list: `ignore_unused = []`.\n- `deps_parser_choice`: Manually select which format to use for parsing\n  declared dependencies. Must be one of `"requirements.txt"`, `"setup.py"`,\n  `"setup.cfg"`, `"pyproject.toml"`, or leave it unset (i.e. the default) for\n  auto-detection (based on filename).\n- `verbosity`: An integer controlling the default log level of FawltyDeps:\n  - `-2`: Only `CRITICAL`-level log messages are shown.\n  - `-1`: `ERROR`-level log messages and above are shown.\n  - `0`: `WARNING`-level log messages and above are shown. This is the default.\n  - `1`: `INFO`-level log messages and above are shown.\n  - `2`: All log messages (including `DEBUG`) are shown.\n\n### Environment variables\n\nIn addition to configuring FawltyDeps via `pyproject.toml` as show above, you\nmay also pass the above configuration directives via the environment, using a\n`fawltydeps_` prefix. For example, to enable JSON output via the environment,\nset `fawltydeps_output_format=json` in FawltyDeps\' environment.\n\n### Configuration cascade\n\n- Command-line options take precedence, and override corresponding settings\n  passed via the environment or `pyproject.toml`.\n- Environment variables override corresponding settings from `pyproject.toml`.\n- Configuration in `pyproject.toml` override only the ultimate hardcoded defaults.\n- The ultimate defaults when no cutomizations takes place are hardcoded inside\n  FawltyDeps, and are documented above.\n\n## Documentation\n\nThis project began with an exploration and design phase, yielding this [design document](./docs/DesignDoc.md), which lays out the main objective for this project and compares various strategies considered\n\nIn the [code design](./docs/CodeDesign.md) section of documentation we lay out rules which we adopt to guide code architecture decisions and maintain code quality as the project evolves.\n\n## Development\n\n### Poetry\n\nThe project uses [Poetry](https://python-poetry.org/). Install Poetry, and then\nrun:\n\n```sh\npoetry install --with=dev\n```\n\nto create a virtualenv with all (development) dependencies installed.\n\nFrom there you can run:\n\n```sh\npoetry shell\n```\n\nto jump into a development shell with this virtualenv activated. Here you will\nhave all the dependencies declared in our [`pyproject.toml`](./pyproject.toml)\ninstalled. (Without this shell activated you will have to prefix the more\nspecific commands below with `poetry run ...`).\n\n### Nox\n\nWe use [Nox](https://nox.thea.codes/en/stable/) for test/workflow automation:\n\n```sh\nnox --list        # List sessions\nnox               # Run all available sessions\nnox -R            # Run all available sessions, while reusing virtualenvs (i.e. faster)\nnox -s tests      # Run unit tests on supported Python versions (that are available)\nnox -s tests-3.7  # Run unit tests on Python v3.7 (assuming it is available locally)\nnox -s integration_tests-3.11  # Run integration tests on Python 3.11\nnox -s lint       # Run linters (mypy + pylint) on all supported Python versions\nnox -s format     # Check formatting (isort + black)\nnox -s reformat   # Fix formatting (isort + black)\n```\n\nIf you want to run a command individually, the corresponding session is defined inside\n[`noxfile.py`](./noxfile.py). For example, these\ncommands will work:\n\n```sh\npytest                   # Run unit tests\npytest -m integration    # Run integration tests\nmypy                     # Run static type checking\npylint fawltydeps tests  # Run Pylint\nisort fawltydeps tests   # Fix sorting of import statements\nblack .                  # Fix code formatting\n```\n\n### Shortcut: Nix\n\nWe have a [`shell.nix`](./shell.nix) which provides Poetry in addition to all of\nour supported Python versions. If you have [Nix](https://nixos.org) available\non your machine, then running:\n\n```sh\nnix-shell\n```\n\nwill put you inside a shell where the Poetry virtualenv (with all development\ndependencies) is activated, and all supported Python versions are available.\nThis also provides isolation from whatever Python version(s) and packages are\ninstalled on your system.\n\nFrom there, a simple `nox` will run all tests + linters against all supported\nPython versions, as well as checking/formatting the code.\n\n## Integration tests\n\nIn addition to comprehensive unit tests under `tests/`, we also verify\nFawltyDeps\' behavior with integration tests which (among other things) include\ntesting with real-world projects. To that end, we have a framework in\n[`tests/test_real_projects.py`](./tests/test_real_projects.py) for downloading\nand unpacking tarballs of 3rd-party projects, and then running fawltydeps on them,\nwhile verifying their output. These projects, along with the expected FawltyDeps\noutputs, are defined in TOML files under\n[`tests/real_projects`](./tests/real_projects).\n\n### Contributing more projects to the test suite\n\nFor bug reports, when a user reports that FawltyDeps does not work as it should\non their project, we aim to follow this process:\n\n- If the project is freely available, we can add a relevant version of the\n  project under `tests/real_projects`.\n- We can then isolate the problems/issues/features and define/express them\n  succinctly as one or more sample projects under `tests/sample_projects`.\n- We examine the issue more closely and update core logic, adding/altering unit\n  tests along the way.\n\nThe resulting updates are introduced to `fawltydeps` and reflected in our\nexpectations, first in the TOML for the sample project(s) and then finally in\nthe `real_projects` TOML.\n\nIf you find a project where FawltyDeps is not doing a good job, we appreciate\nif you add that project under [`tests/real_projects`](./tests/real_projects).\nTo see how these tests work, look at the existing files in that directory.\n\n## FAQ\n\n### I run `fawltydeps` and get some undeclared dependencies. What can I do with it?\n\nYou can run a detailed report to see the exact location (file and line number), in which\nthe undeclared dependencies were imported:\n\n```\nfawltydeps --detailed\n```\n\nand debug each occurrence. Typically an undeclared dependency can be fixed in a couple of ways:\n\n- A true undeclared dependency is fixed by _declaring_ it, e.g. adding it to your `pyproject.toml` or similar.\n- If you disagree with FawltyDeps\' classification, you can always use `--ignore-undeclared` to silence the error. If you\'re sure this dependency should not have been reported by FawltyDeps, you may consider filing a bug report.\n\n### How not to display tools like `black` and `pylint` in _unused dependencies_?\n\nBy default, all packages declared as dependencies by your project are included\nin the FawltyDeps analysis, even if they only contain tools that were not meant\nto be `import`ed, but rather meant to be run by, say, in a pre-commit hook or a\nCI script. In such cases you may use either:\n\n```\nfawltydeps --ignore-unused black pylint\n```\n\nor add an equivalent directive to the FawltyDeps configuration in your\n`pyproject.toml` (see below).\n\n### How can I store my `fawltydeps` command line options in a configuration file?\n\nYou can run:\n\n```\nfawltydeps --generate-toml-config\n```\n\nto generate a `[tool.fawltydeps]` section with the current configuration that\nyou can then directly copy into your `pyproject.toml`. Options that have their\ndefault value are commented in this output, so you have quickly see where your\nsettings differ from the FawltyDeps defaults.\n\nThis also works together with other command line options, so for example in the\nprevious question, you could add `--generate-toml-config` to the command line\n(i.e. run `fawltydeps --ignore-unused black pylint --generate-toml-config`),\nto get this:\n\n```\n[tool.fawltydeps]\n# Default options are commented...\nignore_unused = ["black", "pylint"]\n```\n\n### How to use FawltyDeps in a monorepo?\n\nRunning `fawltydeps` without arguments at the root of a monorepo\nwill most likely not give you a useful result:\nit will collect dependencies and import statements from across the _entire_ monorepo.\nThe produced report may be overwhelming and at the same time not granular enough.\n\nInstead, you should run FawltyDeps for each package separately.\nThis collects dependencies and import statements for one package at a time.\n\nHaving:\n\n```.\n├ lib1\n| ├ pyproject.toml\n| ├ ....\n├ lib2\n| ├ pyproject.toml\n| ├ ....\n```\n\nrun for each `libX`:\n\n```\nfawltydeps libX\n```\n\n### Why must FawltyDeps run in the same Python environment as my project dependencies?\n\nAs explained above in the section on [resolving dependencies via your Python\nenvironment](#resolving-dependencies-via-your-python-environment), the core\nlogic of FawltyDeps needs to match `import` statements in your code with\ndependencies declared in your project configuration. This is straightforward\nfor many packages: for example you `pip install requests` and then\nyou can `import requests` in your code. However, this mapping from the name you\ninstall to the name you `import` is not always self-evident:\n\n- There are sometimes differences between the package name that you\n  declare as a dependency, and the `import` name it provides. For example, you\n  depend on `PyYAML`, but you `import yaml`.\n- A dependency can expose more than one import name. For example the\n  `setuptools` package exposes three `import`able packages: `_distutils_hack`,\n  `pkg_resources`, and `setuptools`. So when you `import pkg_resources`,\n  FawltyDeps need to figure out that this corresponds to the `setuptools`\n  dependency.\n\nTo solve this, FawltyDeps looks at the packages installed in your current Python\nenvironment (or the environment given by the `--pyenv` option) to correctly map\ndependencies (package names) into the imports that they provide.\n\nHowever, when an installed package is not found for a declared dependency, the\n_identity mapping_ that FawltyDeps falls back to will still do a good job for\nthe majority of dependencies where the import name is indeed identical to the\npackage name that you depend on.\n\nThis is an area of active development in FawltyDeps, and we are\n[working on better solutions](https://github.com/tweag/FawltyDeps/issues/195),\nto avoid having to fall back to this identity mapping.\n\n### Why does FawltyDeps fail to match `sklearn` with `scikit-learn`?\n\nThere are cases, where FawltyDeps may not match imports and obviously related\ndependencies, like `sklearn` and `scikit-learn`. It will report `sklearn` as\n_undeclared_ and `scikit-learn` as an _unused_ dependency.\n\nThis is very much related to the above question. `scikit-learn` is an example\nof a package that exposes a different import name: `sklearn`.\nWhen `scikit-learn` is not installed in the current Python environment (the one\nthat FawltyDeps uses to find these mappings), then FawltyDeps is unable to make\nthe connection between these two names.\n\nTo solve this problem, make sure that you either install and run FawltyDeps\nin a development environment (e.g. virtualenv) where your project\'s dependencies\n(including `scikit-learn`) are also installed. Alternatively, you can use the\n`--pyenv` option to point at a Python environment where `scikit-learn` and your\nother dependencies are installed.\n',
    'author': 'Maria Knorps',
    'author_email': 'maria.knorps@tweag.io',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/tweag/FawltyDeps',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.7.2,<4.0.0',
}


setup(**setup_kwargs)
