"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Queue = exports.FifoThroughputLimit = exports.DeduplicationScope = exports.QueueEncryption = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const queue_base_1 = require("./queue-base");
const sqs_generated_1 = require("./sqs.generated");
const validate_props_1 = require("./validate-props");
/**
 * What kind of encryption to apply to this queue.
 *
 * @stability stable
 */
var QueueEncryption;
(function (QueueEncryption) {
    QueueEncryption["UNENCRYPTED"] = "NONE";
    QueueEncryption["KMS_MANAGED"] = "MANAGED";
    QueueEncryption["KMS"] = "KMS";
})(QueueEncryption = exports.QueueEncryption || (exports.QueueEncryption = {}));
/**
 * What kind of deduplication scope to apply.
 *
 * @stability stable
 */
var DeduplicationScope;
(function (DeduplicationScope) {
    DeduplicationScope["MESSAGE_GROUP"] = "messageGroup";
    DeduplicationScope["QUEUE"] = "queue";
})(DeduplicationScope = exports.DeduplicationScope || (exports.DeduplicationScope = {}));
/**
 * Whether the FIFO queue throughput quota applies to the entire queue or per message group.
 *
 * @stability stable
 */
var FifoThroughputLimit;
(function (FifoThroughputLimit) {
    FifoThroughputLimit["PER_QUEUE"] = "perQueue";
    FifoThroughputLimit["PER_MESSAGE_GROUP_ID"] = "perMessageGroupId";
})(FifoThroughputLimit = exports.FifoThroughputLimit || (exports.FifoThroughputLimit = {}));
/**
 * A new Amazon SQS queue.
 *
 * @stability stable
 */
class Queue extends queue_base_1.QueueBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _b;
        super(scope, id, {
            physicalName: props.queueName,
        });
        /**
         * Controls automatic creation of policy objects.
         *
         * Set by subclasses.
         *
         * @stability stable
         */
        this.autoCreatePolicy = true;
        jsiiDeprecationWarnings._aws_cdk_aws_sqs_QueueProps(props);
        validate_props_1.validateProps(props);
        const redrivePolicy = props.deadLetterQueue
            ? {
                deadLetterTargetArn: props.deadLetterQueue.queue.queueArn,
                maxReceiveCount: props.deadLetterQueue.maxReceiveCount,
            }
            : undefined;
        const { encryptionMasterKey, encryptionProps } = _determineEncryptionProps.call(this);
        const fifoProps = this.determineFifoProps(props);
        this.fifo = fifoProps.fifoQueue || false;
        const queue = new sqs_generated_1.CfnQueue(this, 'Resource', {
            queueName: this.physicalName,
            ...fifoProps,
            ...encryptionProps,
            redrivePolicy,
            delaySeconds: props.deliveryDelay && props.deliveryDelay.toSeconds(),
            maximumMessageSize: props.maxMessageSizeBytes,
            messageRetentionPeriod: props.retentionPeriod && props.retentionPeriod.toSeconds(),
            receiveMessageWaitTimeSeconds: props.receiveMessageWaitTime && props.receiveMessageWaitTime.toSeconds(),
            visibilityTimeout: props.visibilityTimeout && props.visibilityTimeout.toSeconds(),
        });
        queue.applyRemovalPolicy((_b = props.removalPolicy) !== null && _b !== void 0 ? _b : core_1.RemovalPolicy.DESTROY);
        this.queueArn = this.getResourceArnAttribute(queue.attrArn, {
            service: 'sqs',
            resource: this.physicalName,
        });
        this.queueName = this.getResourceNameAttribute(queue.attrQueueName);
        this.encryptionMasterKey = encryptionMasterKey;
        this.queueUrl = queue.ref;
        this.deadLetterQueue = props.deadLetterQueue;
        function _determineEncryptionProps() {
            let encryption = props.encryption || QueueEncryption.UNENCRYPTED;
            if (encryption !== QueueEncryption.KMS && props.encryptionMasterKey) {
                encryption = QueueEncryption.KMS; // KMS is implied by specifying an encryption key
            }
            if (encryption === QueueEncryption.UNENCRYPTED) {
                return { encryptionProps: {} };
            }
            if (encryption === QueueEncryption.KMS_MANAGED) {
                return {
                    encryptionProps: {
                        kmsMasterKeyId: 'alias/aws/sqs',
                        kmsDataKeyReusePeriodSeconds: props.dataKeyReuse && props.dataKeyReuse.toSeconds(),
                    },
                };
            }
            if (encryption === QueueEncryption.KMS) {
                const masterKey = props.encryptionMasterKey || new kms.Key(this, 'Key', {
                    description: `Created by ${this.node.path}`,
                });
                return {
                    encryptionMasterKey: masterKey,
                    encryptionProps: {
                        kmsMasterKeyId: masterKey.keyArn,
                        kmsDataKeyReusePeriodSeconds: props.dataKeyReuse && props.dataKeyReuse.toSeconds(),
                    },
                };
            }
            throw new Error(`Unexpected 'encryptionType': ${encryption}`);
        }
    }
    /**
     * Import an existing SQS queue provided an ARN.
     *
     * @param scope The parent creating construct.
     * @param id The construct's name.
     * @param queueArn queue ARN (i.e. arn:aws:sqs:us-east-2:444455556666:queue1).
     * @stability stable
     */
    static fromQueueArn(scope, id, queueArn) {
        return Queue.fromQueueAttributes(scope, id, { queueArn });
    }
    /**
     * Import an existing queue.
     *
     * @stability stable
     */
    static fromQueueAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_sqs_QueueAttributes(attrs);
        const stack = core_1.Stack.of(scope);
        const parsedArn = stack.splitArn(attrs.queueArn, core_1.ArnFormat.NO_RESOURCE_NAME);
        const queueName = attrs.queueName || parsedArn.resource;
        const queueUrl = attrs.queueUrl || `https://sqs.${parsedArn.region}.${stack.urlSuffix}/${parsedArn.account}/${queueName}`;
        class Import extends queue_base_1.QueueBase {
            constructor() {
                super(...arguments);
                this.queueArn = attrs.queueArn; // arn:aws:sqs:us-east-1:123456789012:queue1
                this.queueUrl = queueUrl;
                this.queueName = queueName;
                this.encryptionMasterKey = attrs.keyArn
                    ? kms.Key.fromKeyArn(this, 'Key', attrs.keyArn)
                    : undefined;
                this.fifo = this.determineFifo();
                this.autoCreatePolicy = false;
            }
            /**
             * Determine fifo flag based on queueName and fifo attribute
             */
            determineFifo() {
                if (core_1.Token.isUnresolved(this.queueArn)) {
                    return attrs.fifo || false;
                }
                else {
                    if (typeof attrs.fifo !== 'undefined') {
                        if (attrs.fifo && !queueName.endsWith('.fifo')) {
                            throw new Error("FIFO queue names must end in '.fifo'");
                        }
                        if (!attrs.fifo && queueName.endsWith('.fifo')) {
                            throw new Error("Non-FIFO queue name may not end in '.fifo'");
                        }
                    }
                    return queueName.endsWith('.fifo') ? true : false;
                }
            }
        }
        return new Import(scope, id);
    }
    /**
     * Look at the props, see if the FIFO props agree, and return the correct subset of props
     */
    determineFifoProps(props) {
        // Check if any of the signals that we have say that this is a FIFO queue.
        let fifoQueue = props.fifo;
        const queueName = props.queueName;
        if (typeof fifoQueue === 'undefined' && queueName && !core_1.Token.isUnresolved(queueName) && queueName.endsWith('.fifo')) {
            fifoQueue = true;
        }
        if (typeof fifoQueue === 'undefined' && props.contentBasedDeduplication) {
            fifoQueue = true;
        }
        if (typeof fifoQueue === 'undefined' && props.deduplicationScope) {
            fifoQueue = true;
        }
        if (typeof fifoQueue === 'undefined' && props.fifoThroughputLimit) {
            fifoQueue = true;
        }
        // If we have a name, see that it agrees with the FIFO setting
        if (typeof queueName === 'string') {
            if (fifoQueue && !queueName.endsWith('.fifo')) {
                throw new Error("FIFO queue names must end in '.fifo'");
            }
            if (!fifoQueue && queueName.endsWith('.fifo')) {
                throw new Error("Non-FIFO queue name may not end in '.fifo'");
            }
        }
        if (props.contentBasedDeduplication && !fifoQueue) {
            throw new Error('Content-based deduplication can only be defined for FIFO queues');
        }
        if (props.deduplicationScope && !fifoQueue) {
            throw new Error('Deduplication scope can only be defined for FIFO queues');
        }
        if (props.fifoThroughputLimit && !fifoQueue) {
            throw new Error('FIFO throughput limit can only be defined for FIFO queues');
        }
        return {
            contentBasedDeduplication: props.contentBasedDeduplication,
            deduplicationScope: props.deduplicationScope,
            fifoThroughputLimit: props.fifoThroughputLimit,
            fifoQueue,
        };
    }
}
exports.Queue = Queue;
_a = JSII_RTTI_SYMBOL_1;
Queue[_a] = { fqn: "@aws-cdk/aws-sqs.Queue", version: "1.144.0" };
//# sourceMappingURL=data:application/json;base64,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