"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OriginVerify = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const aws_wafv2_1 = require("aws-cdk-lib/aws-wafv2");
const constructs_1 = require("constructs");
const add_1 = require("./errors/add");
/**
 * Associates an origin with WAFv2 WebACL to verify traffic contains specific
 * header with a secret value.
 */
class OriginVerify extends constructs_1.Construct {
    /**
     * Associates an origin with WAFv2 WebACL to verify traffic contains specific
     * header with a secret value.
     *
     * Use `verifyHeader` value to assign custom headers into CloudFront config.
     *
     * @example
     * import { OriginVerify } from '@alma-cdk/origin-verify';
     * import { Distribution } from 'aws-cdk-lib/aws-cloudfront';
     *
     * const api: RestApi; // TODO: implement the RestApi
     * const apiDomain: string; // TODO: implement the domain
     *
     * const verification = new OriginVerify(this, 'OriginVerify', {
     *   origin: api.deploymentStage,
     * });
     *
     * new Distribution(this, 'CDN', {
     *   defaultBehavior: {
     *     origin: new HttpOrigin(apiDomain, {
     *       customHeaders: {
     *         [verification.headerName]: verification.headerValue,
     *       },
     *       protocolPolicy: OriginProtocolPolicy.HTTPS_ONLY,
     *     })
     *   },
     * })
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Define the exposed header information
        const secretValue = this.resolveSecret(props.secretValue);
        const verifyHeader = {
            headerName: props.headerName || OriginVerify.OriginVerifyHeader,
            headerValue: secretValue.unsafeUnwrap(),
        };
        this.headerName = verifyHeader.headerName;
        this.headerValue = verifyHeader.headerValue;
        const acl = this.defineAcl(verifyHeader, props);
        this.associate(acl, this.resolveOriginArn(props.origin));
    }
    /** Generates a new Secrets Manager Secret if none provided via props. */
    resolveSecret(secret) {
        if (typeof secret !== 'undefined') {
            return secret;
        }
        return new aws_secretsmanager_1.Secret(this, 'OriginVerifySecret').secretValue;
    }
    /** Define a new WAFv2 WebACL. */
    defineAcl(header, props) {
        return new aws_wafv2_1.CfnWebACL(this, 'WebACL', {
            defaultAction: {
                block: {},
            },
            scope: 'REGIONAL',
            visibilityConfig: {
                cloudWatchMetricsEnabled: true,
                metricName: props.aclMetricName || 'OriginVerifyWebAcl',
                sampledRequestsEnabled: true,
            },
            rules: [
                ...(props.rules || []),
                this.allowCloudFrontRequests(header, props.ruleMetricName),
            ],
        });
    }
    /** Allow traffic with specific header secret. */
    allowCloudFrontRequests(header, ruleMetricName) {
        return {
            name: 'AllowCloudFrontRequests',
            priority: 0,
            visibilityConfig: {
                cloudWatchMetricsEnabled: false,
                metricName: ruleMetricName || 'OriginVerifyAllowedRequests',
                sampledRequestsEnabled: false,
            },
            action: {
                allow: {},
            },
            statement: this.allowVerifiedOrigin(header),
        };
    }
    /** Define WAFv2 Statement matching specific header and its value. */
    allowVerifiedOrigin(header) {
        return {
            byteMatchStatement: {
                fieldToMatch: {
                    singleHeader: {
                        Name: header.headerName,
                    },
                },
                // Use of unsafeUnwrap allowed as we must be able to assign the secret
                // value into WebACL rule (and that rule stays within AWS Account).
                searchString: header.headerValue,
                positionalConstraint: 'EXACTLY',
                textTransformations: [
                    {
                        priority: 0,
                        type: 'NONE',
                    },
                ],
            },
        };
    }
    /** Associates a WAFv2 WebACL into an AWS Resource (defined by ARN). */
    associate(acl, arn) {
        new aws_wafv2_1.CfnWebACLAssociation(this, 'WebACLAssociation', {
            webAclArn: acl.attrArn,
            resourceArn: arn,
        });
    }
    /** Type guard for Application Load Balancer. */
    isAlb(origin) {
        return 'loadBalancerArn' in origin;
    }
    /** Type guard for API Gateway Stage. */
    isStage(origin) {
        return 'stageName' in origin;
    }
    isCfnGraphQLApi(origin) {
        return 'attrGraphQlUrl' in origin;
    }
    /** Resolves origin (either IStage or IApplicationLoadBalancer) ARN. */
    resolveOriginArn(origin) {
        if (this.isAlb(origin)) {
            return origin.loadBalancerArn;
        }
        if (this.isStage(origin)) {
            return this.resolveStageArn(origin);
        }
        if (this.isCfnGraphQLApi(origin)) {
            return origin.attrArn;
        }
        add_1.addError(this, 'Invalid origin: Must be either Api Gateway IStage, IApplicationLoadBalancer or AppSync CfnGraphQLApi');
        return '';
    }
    /** Formates API Gateway Stage ARN. */
    resolveStageArn(stage) {
        const region = aws_cdk_lib_1.Stack.of(stage).region;
        const apiId = stage.restApi.restApiId;
        const stageName = stage.stageName;
        return `arn:aws:apigateway:${region}::/restapis/${apiId}/stages/${stageName}`;
    }
}
exports.OriginVerify = OriginVerify;
_a = JSII_RTTI_SYMBOL_1;
OriginVerify[_a] = { fqn: "@alma-cdk/origin-verify.OriginVerify", version: "0.0.26" };
/** Origin Request Header Default Name */
OriginVerify.OriginVerifyHeader = 'x-origin-verify';
//# sourceMappingURL=data:application/json;base64,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