"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Appflow = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [appflow](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonappflow.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Appflow extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [appflow](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonappflow.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'appflow';
        this.accessLevelList = {
            "Write": [
                "CreateConnectorProfile",
                "CreateFlow",
                "DeleteConnectorProfile",
                "DeleteFlow",
                "RunFlow",
                "StartFlow",
                "StopFlow",
                "UpdateConnectorProfile",
                "UpdateFlow"
            ],
            "Read": [
                "DescribeConnectorEntity",
                "DescribeConnectorFields",
                "DescribeConnectorProfiles",
                "DescribeConnectors",
                "DescribeFlow",
                "DescribeFlowExecution",
                "DescribeFlowExecutionRecords",
                "DescribeFlows",
                "ListConnectorFields"
            ],
            "List": [
                "ListConnectorEntities",
                "ListFlows",
                "ListTagsForResource"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to create a login profile to be used with Amazon AppFlow flows
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_CreateConnectorProfile.html
     */
    toCreateConnectorProfile() {
        this.to('appflow:CreateConnectorProfile');
        return this;
    }
    /**
     * Grants permission to create an Amazon AppFlow flow
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_CreateFlow.html
     */
    toCreateFlow() {
        this.to('appflow:CreateFlow');
        return this;
    }
    /**
     * Grants permission to delete a login profile configured in Amazon AppFlow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_DescribeConnectorProfiles.html
     */
    toDeleteConnectorProfile() {
        this.to('appflow:DeleteConnectorProfile');
        return this;
    }
    /**
     * Grants permission to delete an Amazon AppFlow flow
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_DeleteFlow.html
     */
    toDeleteFlow() {
        this.to('appflow:DeleteFlow');
        return this;
    }
    /**
     * Grants permission to describe all fields for an object in a login profile configured in Amazon AppFlow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_DescribeConnectorEntity.html
     */
    toDescribeConnectorEntity() {
        this.to('appflow:DescribeConnectorEntity');
        return this;
    }
    /**
     * Grants permission to describe all fields for an object in a login profile configured in Amazon AppFlow (Console Only)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appflow/latest/userguide/identity-access-management.html#appflow-api-actions
     */
    toDescribeConnectorFields() {
        this.to('appflow:DescribeConnectorFields');
        return this;
    }
    /**
     * Grants permission to describe all login profiles configured in Amazon AppFlow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_DescribeConnectorProfiles.html
     */
    toDescribeConnectorProfiles() {
        this.to('appflow:DescribeConnectorProfiles');
        return this;
    }
    /**
     * Grants permission to describe all connectors supported by Amazon AppFlow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_DescribeConnectors.html
     */
    toDescribeConnectors() {
        this.to('appflow:DescribeConnectors');
        return this;
    }
    /**
     * Grants permission to describe a specific flow configured in Amazon AppFlow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_DescribeFlow.html
     */
    toDescribeFlow() {
        this.to('appflow:DescribeFlow');
        return this;
    }
    /**
     * Grants permission to describe all flow executions for a flow configured in Amazon AppFlow (Console Only)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appflow/latest/userguide/identity-access-management.html#appflow-api-actions
     */
    toDescribeFlowExecution() {
        this.to('appflow:DescribeFlowExecution');
        return this;
    }
    /**
     * Grants permission to describe all flow executions for a flow configured in Amazon AppFlow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_DescribeFlowExecutionRecords.html
     */
    toDescribeFlowExecutionRecords() {
        this.to('appflow:DescribeFlowExecutionRecords');
        return this;
    }
    /**
     * Grants permission to describe all flows configured in Amazon AppFlow (Console Only)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appflow/latest/userguide/identity-access-management.html#appflow-api-actions
     */
    toDescribeFlows() {
        this.to('appflow:DescribeFlows');
        return this;
    }
    /**
     * Grants permission to list all objects for a login profile configured in Amazon AppFlow
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_ListConnectorEntities.html
     */
    toListConnectorEntities() {
        this.to('appflow:ListConnectorEntities');
        return this;
    }
    /**
     * Grants permission to list all objects for a login profile configured in Amazon AppFlow (Console Only)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appflow/latest/userguide/identity-access-management.html#appflow-api-actions
     */
    toListConnectorFields() {
        this.to('appflow:ListConnectorFields');
        return this;
    }
    /**
     * Grants permission to list all flows configured in Amazon AppFlow
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_ListFlows.html
     */
    toListFlows() {
        this.to('appflow:ListFlows');
        return this;
    }
    /**
     * Grants permission to list tags for a flow
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('appflow:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to run a flow configured in Amazon AppFlow (Console Only)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appflow/latest/userguide/identity-access-management.html#appflow-api-actions
     */
    toRunFlow() {
        this.to('appflow:RunFlow');
        return this;
    }
    /**
     * Grants permission to activate (for scheduled and event-triggered flows) or run (for on-demand flows) a flow configured in Amazon AppFlow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_StartFlow.html
     */
    toStartFlow() {
        this.to('appflow:StartFlow');
        return this;
    }
    /**
     * Grants permission to deactivate a scheduled or event-triggered flow configured in Amazon AppFlow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_StopFlow.html
     */
    toStopFlow() {
        this.to('appflow:StopFlow');
        return this;
    }
    /**
     * Grants permission to tag a flow
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('appflow:TagResource');
        return this;
    }
    /**
     * Grants permission to untag a flow
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('appflow:UntagResource');
        return this;
    }
    /**
     * Grants permission to update a login profile configured in Amazon AppFlow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_UpdateConnectorProfile.html
     */
    toUpdateConnectorProfile() {
        this.to('appflow:UpdateConnectorProfile');
        return this;
    }
    /**
     * Grants permission to update a flow configured in Amazon AppFlow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_UpdateFlow.html
     */
    toUpdateFlow() {
        this.to('appflow:UpdateFlow');
        return this;
    }
    /**
     * Adds a resource of type connectorprofile to the statement
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_ConnectorProfile.html
     *
     * @param profileName - Identifier for the profileName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onConnectorprofile(profileName, account, region, partition) {
        var arn = 'arn:${Partition}:appflow:${Region}:${Account}:connectorprofile/${ProfileName}';
        arn = arn.replace('${ProfileName}', profileName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type flow to the statement
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_FlowDefinition.html
     *
     * @param flowName - Identifier for the flowName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFlow(flowName, account, region, partition) {
        var arn = 'arn:${Partition}:appflow:${Region}:${Account}:flow/${FlowName}';
        arn = arn.replace('${FlowName}', flowName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Appflow = Appflow;
//# sourceMappingURL=data:application/json;base64,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