"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Braket = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [braket](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonbraket.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Braket extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [braket](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonbraket.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'braket';
        this.accessLevelList = {
            "Write": [
                "CancelQuantumTask",
                "CreateQuantumTask"
            ],
            "Read": [
                "GetDevice",
                "GetQuantumTask",
                "ListTagsForResource",
                "SearchDevices",
                "SearchQuantumTasks"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to cancel a quantum task.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/braket/latest/APIReference/API_CancelQuantumTask.html
     */
    toCancelQuantumTask() {
        this.to('braket:CancelQuantumTask');
        return this;
    }
    /**
     * Grants permission to create a quantum task.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/braket/latest/APIReference/API_CreateQuantumTask.html
     */
    toCreateQuantumTask() {
        this.to('braket:CreateQuantumTask');
        return this;
    }
    /**
     * Grants permission to retrieve information about the devices available in Amazon Braket.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/braket/latest/APIReference/API_GetDevice.html
     */
    toGetDevice() {
        this.to('braket:GetDevice');
        return this;
    }
    /**
     * Grants permission to retrieve quantum tasks.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/braket/latest/APIReference/API_GetQuantumTask.html
     */
    toGetQuantumTask() {
        this.to('braket:GetQuantumTask');
        return this;
    }
    /**
     * Lists the tags that have been applied to the quantum task resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/braket/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('braket:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to search for devices available in Amazon Braket.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/braket/latest/APIReference/API_SearchDevices.html
     */
    toSearchDevices() {
        this.to('braket:SearchDevices');
        return this;
    }
    /**
     * Grants permission to search for quantum tasks.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/braket/latest/APIReference/API_SearchQuantumTasks.html
     */
    toSearchQuantumTasks() {
        this.to('braket:SearchQuantumTasks');
        return this;
    }
    /**
     * Adds one or more tags to a quantum task.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/braket/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('braket:TagResource');
        return this;
    }
    /**
     * Remove one or more tags from a quantum task resource. A tag consists of a key-value pair
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/braket/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('braket:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type quantum-task to the statement
     *
     * https://docs.aws.amazon.com/braket/latest/developerguide/braket-manage-access.html#resources
     *
     * @param randomId - Identifier for the randomId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onQuantumTask(randomId, account, region, partition) {
        var arn = 'arn:${Partition}:braket:${Region}:${Account}:quantum-task/${RandomId}';
        arn = arn.replace('${RandomId}', randomId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Braket = Braket;
//# sourceMappingURL=data:application/json;base64,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