"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Clouddirectory = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [clouddirectory](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonclouddirectory.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Clouddirectory extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [clouddirectory](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonclouddirectory.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'clouddirectory';
        this.accessLevelList = {
            "Write": [
                "AddFacetToObject",
                "ApplySchema",
                "AttachObject",
                "AttachPolicy",
                "AttachToIndex",
                "AttachTypedLink",
                "BatchWrite",
                "CreateDirectory",
                "CreateFacet",
                "CreateIndex",
                "CreateObject",
                "CreateSchema",
                "CreateTypedLinkFacet",
                "DeleteDirectory",
                "DeleteFacet",
                "DeleteObject",
                "DeleteSchema",
                "DeleteTypedLinkFacet",
                "DetachFromIndex",
                "DetachObject",
                "DetachPolicy",
                "DetachTypedLink",
                "DisableDirectory",
                "EnableDirectory",
                "PublishSchema",
                "PutSchemaFromJson",
                "RemoveFacetFromObject",
                "UpdateFacet",
                "UpdateLinkAttributes",
                "UpdateObjectAttributes",
                "UpdateSchema",
                "UpdateTypedLinkFacet"
            ],
            "Read": [
                "BatchRead",
                "GetDirectory",
                "GetFacet",
                "GetLinkAttributes",
                "GetObjectAttributes",
                "GetObjectInformation",
                "GetSchemaAsJson",
                "GetTypedLinkFacetInformation",
                "ListAttachedIndices",
                "ListFacetAttributes",
                "ListFacetNames",
                "ListIncomingTypedLinks",
                "ListIndex",
                "ListObjectAttributes",
                "ListObjectChildren",
                "ListObjectParentPaths",
                "ListObjectParents",
                "ListObjectPolicies",
                "ListOutgoingTypedLinks",
                "ListPolicyAttachments",
                "ListTagsForResource",
                "ListTypedLinkFacetAttributes",
                "ListTypedLinkFacetNames",
                "LookupPolicy"
            ],
            "List": [
                "ListAppliedSchemaArns",
                "ListDevelopmentSchemaArns",
                "ListDirectories",
                "ListPublishedSchemaArns"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Adds a new Facet to an object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_AddFacetToObject.html
     */
    toAddFacetToObject() {
        this.to('clouddirectory:AddFacetToObject');
        return this;
    }
    /**
     * Copies input published schema into Directory with same name and version as that of published schema.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ApplySchema.html
     */
    toApplySchema() {
        this.to('clouddirectory:ApplySchema');
        return this;
    }
    /**
     * Attaches an existing object to another existing object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_AttachObject.html
     */
    toAttachObject() {
        this.to('clouddirectory:AttachObject');
        return this;
    }
    /**
     * Attaches a policy object to any other object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_AttachPolicy.html
     */
    toAttachPolicy() {
        this.to('clouddirectory:AttachPolicy');
        return this;
    }
    /**
     * Attaches the specified object to the specified index.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_AttachToIndex.html
     */
    toAttachToIndex() {
        this.to('clouddirectory:AttachToIndex');
        return this;
    }
    /**
     * Attaches a typed link b/w a source & target object reference.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_AttachTypedLink.html
     */
    toAttachTypedLink() {
        this.to('clouddirectory:AttachTypedLink');
        return this;
    }
    /**
     * Performs all the read operations in a batch. Each individual operation inside BatchRead needs to be granted permissions explicitly.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_BatchRead.html
     */
    toBatchRead() {
        this.to('clouddirectory:BatchRead');
        return this;
    }
    /**
     * Performs all the write operations in a batch. Each individual operation inside BatchWrite needs to be granted permissions explicitly.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_BatchWrite.html
     */
    toBatchWrite() {
        this.to('clouddirectory:BatchWrite');
        return this;
    }
    /**
     * Creates a Directory by copying the published schema into the directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateDirectory.html
     */
    toCreateDirectory() {
        this.to('clouddirectory:CreateDirectory');
        return this;
    }
    /**
     * Creates a new Facet in a schema.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateFacet.html
     */
    toCreateFacet() {
        this.to('clouddirectory:CreateFacet');
        return this;
    }
    /**
     * Creates an index object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateIndex.html
     */
    toCreateIndex() {
        this.to('clouddirectory:CreateIndex');
        return this;
    }
    /**
     * Creates an object in a Directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateObject.html
     */
    toCreateObject() {
        this.to('clouddirectory:CreateObject');
        return this;
    }
    /**
     * Creates a new schema in a development state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateSchema.html
     */
    toCreateSchema() {
        this.to('clouddirectory:CreateSchema');
        return this;
    }
    /**
     * Creates a new Typed Link facet in a schema.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateTypedLinkFacet.html
     */
    toCreateTypedLinkFacet() {
        this.to('clouddirectory:CreateTypedLinkFacet');
        return this;
    }
    /**
     * Deletes a directory. Only disabled directories can be deleted.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DeleteDirectory.html
     */
    toDeleteDirectory() {
        this.to('clouddirectory:DeleteDirectory');
        return this;
    }
    /**
     * Deletes a given Facet. All attributes and Rules associated with the facet will be deleted.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DeleteFacet.html
     */
    toDeleteFacet() {
        this.to('clouddirectory:DeleteFacet');
        return this;
    }
    /**
     * Deletes an object and its associated attributes.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DeleteObject.html
     */
    toDeleteObject() {
        this.to('clouddirectory:DeleteObject');
        return this;
    }
    /**
     * Deletes a given schema.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DeleteSchema.html
     */
    toDeleteSchema() {
        this.to('clouddirectory:DeleteSchema');
        return this;
    }
    /**
     * Deletes a given TypedLink Facet. All attributes and Rules associated with the facet will be deleted.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DeleteTypedLinkFacet.html
     */
    toDeleteTypedLinkFacet() {
        this.to('clouddirectory:DeleteTypedLinkFacet');
        return this;
    }
    /**
     * Detaches the specified object from the specified index.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DetachFromIndex.html
     */
    toDetachFromIndex() {
        this.to('clouddirectory:DetachFromIndex');
        return this;
    }
    /**
     * Detaches a given object from the parent object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DetachObject.html
     */
    toDetachObject() {
        this.to('clouddirectory:DetachObject');
        return this;
    }
    /**
     * Detaches a policy from an object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DetachPolicy.html
     */
    toDetachPolicy() {
        this.to('clouddirectory:DetachPolicy');
        return this;
    }
    /**
     * Detaches a given typed link b/w given source and target object reference.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DetachTypedLink.html
     */
    toDetachTypedLink() {
        this.to('clouddirectory:DetachTypedLink');
        return this;
    }
    /**
     * Disables the specified directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DisableDirectory.html
     */
    toDisableDirectory() {
        this.to('clouddirectory:DisableDirectory');
        return this;
    }
    /**
     * Enables the specified directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_EnableDirectory.html
     */
    toEnableDirectory() {
        this.to('clouddirectory:EnableDirectory');
        return this;
    }
    /**
     * Retrieves metadata about a directory.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetDirectory.html
     */
    toGetDirectory() {
        this.to('clouddirectory:GetDirectory');
        return this;
    }
    /**
     * Gets details of the Facet, such as Facet Name, Attributes, Rules, or ObjectType.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetFacet.html
     */
    toGetFacet() {
        this.to('clouddirectory:GetFacet');
        return this;
    }
    /**
     * Retrieves attributes that are associated with a typed link.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetLinkAttributes.html
     */
    toGetLinkAttributes() {
        this.to('clouddirectory:GetLinkAttributes');
        return this;
    }
    /**
     * Retrieves attributes within a facet that are associated with an object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetObjectAttributes.html
     */
    toGetObjectAttributes() {
        this.to('clouddirectory:GetObjectAttributes');
        return this;
    }
    /**
     * Retrieves metadata about an object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetObjectInformation.html
     */
    toGetObjectInformation() {
        this.to('clouddirectory:GetObjectInformation');
        return this;
    }
    /**
     * Retrieves a JSON representation of the schema.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetSchemaAsJson.html
     */
    toGetSchemaAsJson() {
        this.to('clouddirectory:GetSchemaAsJson');
        return this;
    }
    /**
     * Returns identity attributes order information associated with a given typed link facet.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetTypedLinkFacetInformation.html
     */
    toGetTypedLinkFacetInformation() {
        this.to('clouddirectory:GetTypedLinkFacetInformation');
        return this;
    }
    /**
     * Lists schemas applied to a directory.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListAppliedSchemaArns.html
     */
    toListAppliedSchemaArns() {
        this.to('clouddirectory:ListAppliedSchemaArns');
        return this;
    }
    /**
     * Lists indices attached to an object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListAttachedIndices.html
     */
    toListAttachedIndices() {
        this.to('clouddirectory:ListAttachedIndices');
        return this;
    }
    /**
     * Retrieves the ARNs of schemas in the development state.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListDevelopmentSchemaArns.html
     */
    toListDevelopmentSchemaArns() {
        this.to('clouddirectory:ListDevelopmentSchemaArns');
        return this;
    }
    /**
     * Lists directories created within an account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListDirectories.html
     */
    toListDirectories() {
        this.to('clouddirectory:ListDirectories');
        return this;
    }
    /**
     * Retrieves attributes attached to the facet.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListFacetAttributes.html
     */
    toListFacetAttributes() {
        this.to('clouddirectory:ListFacetAttributes');
        return this;
    }
    /**
     * Retrieves the names of facets that exist in a schema.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListFacetNames.html
     */
    toListFacetNames() {
        this.to('clouddirectory:ListFacetNames');
        return this;
    }
    /**
     * Returns a paginated list of all incoming TypedLinks for a given object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListIncomingTypedLinks.html
     */
    toListIncomingTypedLinks() {
        this.to('clouddirectory:ListIncomingTypedLinks');
        return this;
    }
    /**
     * Lists objects attached to the specified index.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListIndex.html
     */
    toListIndex() {
        this.to('clouddirectory:ListIndex');
        return this;
    }
    /**
     * Lists all attributes associated with an object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListObjectAttributes.html
     */
    toListObjectAttributes() {
        this.to('clouddirectory:ListObjectAttributes');
        return this;
    }
    /**
     * Returns a paginated list of child objects associated with a given object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListObjectChildren.html
     */
    toListObjectChildren() {
        this.to('clouddirectory:ListObjectChildren');
        return this;
    }
    /**
     * Retrieves all available parent paths for any object type such as node, leaf node, policy node, and index node objects.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListObjectParentPaths.html
     */
    toListObjectParentPaths() {
        this.to('clouddirectory:ListObjectParentPaths');
        return this;
    }
    /**
     * Lists parent objects associated with a given object in pagination fashion.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListObjectParents.html
     */
    toListObjectParents() {
        this.to('clouddirectory:ListObjectParents');
        return this;
    }
    /**
     * Returns policies attached to an object in pagination fashion.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListObjectPolicies.html
     */
    toListObjectPolicies() {
        this.to('clouddirectory:ListObjectPolicies');
        return this;
    }
    /**
     * Returns a paginated list of all outgoing TypedLinks for a given object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListOutgoingTypedLinks.html
     */
    toListOutgoingTypedLinks() {
        this.to('clouddirectory:ListOutgoingTypedLinks');
        return this;
    }
    /**
     * Returns all of the ObjectIdentifiers to which a given policy is attached.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListPolicyAttachments.html
     */
    toListPolicyAttachments() {
        this.to('clouddirectory:ListPolicyAttachments');
        return this;
    }
    /**
     * Retrieves published schema ARNs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListPublishedSchemaArns.html
     */
    toListPublishedSchemaArns() {
        this.to('clouddirectory:ListPublishedSchemaArns');
        return this;
    }
    /**
     * Returns tags for a resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('clouddirectory:ListTagsForResource');
        return this;
    }
    /**
     * Returns a paginated list of attributes associated with typed link facet.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListTypedLinkFacetAttributes.html
     */
    toListTypedLinkFacetAttributes() {
        this.to('clouddirectory:ListTypedLinkFacetAttributes');
        return this;
    }
    /**
     * Returns a paginated list of typed link facet names that exist in a schema.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListTypedLinkFacetNames.html
     */
    toListTypedLinkFacetNames() {
        this.to('clouddirectory:ListTypedLinkFacetNames');
        return this;
    }
    /**
     * Lists all policies from the root of the Directory to the object specified.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_LookupPolicy.html
     */
    toLookupPolicy() {
        this.to('clouddirectory:LookupPolicy');
        return this;
    }
    /**
     * Publishes a development schema with a version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_PublishSchema.html
     */
    toPublishSchema() {
        this.to('clouddirectory:PublishSchema');
        return this;
    }
    /**
     * Allows a schema to be updated using JSON upload. Only available for development schemas.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_PutSchemaFromJson.html
     */
    toPutSchemaFromJson() {
        this.to('clouddirectory:PutSchemaFromJson');
        return this;
    }
    /**
     * Removes the specified facet from the specified object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_RemoveFacetFromObject.html
     */
    toRemoveFacetFromObject() {
        this.to('clouddirectory:RemoveFacetFromObject');
        return this;
    }
    /**
     * Adds tags to a resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('clouddirectory:TagResource');
        return this;
    }
    /**
     * Removes tags from a resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('clouddirectory:UntagResource');
        return this;
    }
    /**
     * Adds/Updates/Deletes existing Attributes, Rules, or ObjectType of a Facet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UpdateFacet.html
     */
    toUpdateFacet() {
        this.to('clouddirectory:UpdateFacet');
        return this;
    }
    /**
     * Updates a given typed link’s attributes. Attributes to be updated must not contribute to the typed link’s identity, as defined by its IdentityAttributeOrder.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UpdateLinkAttributes.html
     */
    toUpdateLinkAttributes() {
        this.to('clouddirectory:UpdateLinkAttributes');
        return this;
    }
    /**
     * Updates a given object's attributes.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UpdateObjectAttributes.html
     */
    toUpdateObjectAttributes() {
        this.to('clouddirectory:UpdateObjectAttributes');
        return this;
    }
    /**
     * Updates the schema name with a new name.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UpdateSchema.html
     */
    toUpdateSchema() {
        this.to('clouddirectory:UpdateSchema');
        return this;
    }
    /**
     * Adds/Updates/Deletes existing Attributes, Rules, identity attribute order of a TypedLink Facet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UpdateTypedLinkFacet.html
     */
    toUpdateTypedLinkFacet() {
        this.to('clouddirectory:UpdateTypedLinkFacet');
        return this;
    }
    /**
     * Adds a resource of type appliedSchema to the statement
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/cd_key_concepts.html#whatisdirectory
     *
     * @param directoryId - Identifier for the directoryId.
     * @param schemaName - Identifier for the schemaName.
     * @param version - Identifier for the version.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAppliedSchema(directoryId, schemaName, version, account, region, partition) {
        var arn = 'arn:${Partition}:clouddirectory:${Region}:${Account}:directory/${DirectoryId}/schema/${SchemaName}/${Version}';
        arn = arn.replace('${DirectoryId}', directoryId);
        arn = arn.replace('${SchemaName}', schemaName);
        arn = arn.replace('${Version}', version);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type developmentSchema to the statement
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/cd_key_concepts.html#whatisdirectory
     *
     * @param schemaName - Identifier for the schemaName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDevelopmentSchema(schemaName, account, region, partition) {
        var arn = 'arn:${Partition}:clouddirectory:${Region}:${Account}:schema/development/${SchemaName}';
        arn = arn.replace('${SchemaName}', schemaName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type directory to the statement
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/cd_key_concepts.html#whatisdirectory
     *
     * @param directoryId - Identifier for the directoryId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDirectory(directoryId, account, region, partition) {
        var arn = 'arn:${Partition}:clouddirectory:${Region}:${Account}:directory/${DirectoryId}';
        arn = arn.replace('${DirectoryId}', directoryId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type publishedSchema to the statement
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/cd_key_concepts.html#whatisdirectory
     *
     * @param schemaName - Identifier for the schemaName.
     * @param version - Identifier for the version.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPublishedSchema(schemaName, version, account, region, partition) {
        var arn = 'arn:${Partition}:clouddirectory:${Region}:${Account}:schema/published/${SchemaName}/${Version}';
        arn = arn.replace('${SchemaName}', schemaName);
        arn = arn.replace('${Version}', version);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Clouddirectory = Clouddirectory;
//# sourceMappingURL=data:application/json;base64,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