"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cloudfront = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [cloudfront](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncloudfront.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Cloudfront extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [cloudfront](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncloudfront.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'cloudfront';
        this.accessLevelList = {
            "Write": [
                "CreateCachePolicy",
                "CreateCloudFrontOriginAccessIdentity",
                "CreateDistribution",
                "CreateFieldLevelEncryptionConfig",
                "CreateFieldLevelEncryptionProfile",
                "CreateInvalidation",
                "CreateOriginRequestPolicy",
                "CreatePublicKey",
                "CreateStreamingDistribution",
                "DeleteCachePolicy",
                "DeleteCloudFrontOriginAccessIdentity",
                "DeleteDistribution",
                "DeleteFieldLevelEncryptionConfig",
                "DeleteFieldLevelEncryptionProfile",
                "DeleteOriginRequestPolicy",
                "DeletePublicKey",
                "DeleteStreamingDistribution",
                "UpdateCachePolicy",
                "UpdateCloudFrontOriginAccessIdentity",
                "UpdateDistribution",
                "UpdateFieldLevelEncryptionConfig",
                "UpdateFieldLevelEncryptionProfile",
                "UpdateOriginRequestPolicy",
                "UpdatePublicKey",
                "UpdateStreamingDistribution"
            ],
            "Tagging": [
                "CreateDistributionWithTags",
                "CreateStreamingDistributionWithTags",
                "TagResource",
                "UntagResource"
            ],
            "Read": [
                "GetCachePolicy",
                "GetCachePolicyConfig",
                "GetCloudFrontOriginAccessIdentity",
                "GetCloudFrontOriginAccessIdentityConfig",
                "GetDistribution",
                "GetDistributionConfig",
                "GetFieldLevelEncryption",
                "GetFieldLevelEncryptionConfig",
                "GetFieldLevelEncryptionProfile",
                "GetFieldLevelEncryptionProfileConfig",
                "GetInvalidation",
                "GetOriginRequestPolicy",
                "GetOriginRequestPolicyConfig",
                "GetPublicKey",
                "GetPublicKeyConfig",
                "GetStreamingDistribution",
                "GetStreamingDistributionConfig",
                "ListTagsForResource"
            ],
            "List": [
                "ListCachePolicies",
                "ListCloudFrontOriginAccessIdentities",
                "ListDistributions",
                "ListDistributionsByCachePolicyId",
                "ListDistributionsByOriginRequestPolicyId",
                "ListDistributionsByWebACLId",
                "ListFieldLevelEncryptionConfigs",
                "ListFieldLevelEncryptionProfiles",
                "ListInvalidations",
                "ListOriginRequestPolicies",
                "ListPublicKeys",
                "ListStreamingDistributions"
            ]
        };
    }
    /**
     * This action adds a new cache policy to CloudFront.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateCachePolicy.html
     */
    toCreateCachePolicy() {
        this.to('cloudfront:CreateCachePolicy');
        return this;
    }
    /**
     * This action creates a new CloudFront origin access identity.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateCloudFrontOriginAccessIdentity.html
     */
    toCreateCloudFrontOriginAccessIdentity() {
        this.to('cloudfront:CreateCloudFrontOriginAccessIdentity');
        return this;
    }
    /**
     * This action creates a new web distribution.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateDistribution.html
     */
    toCreateDistribution() {
        this.to('cloudfront:CreateDistribution');
        return this;
    }
    /**
     * This action creates a new web distribution with tags.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateDistributionWithTags.html
     */
    toCreateDistributionWithTags() {
        this.to('cloudfront:CreateDistributionWithTags');
        return this;
    }
    /**
     * This action creates a new field-level encryption configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateFieldLevelEncryptionConfig.html
     */
    toCreateFieldLevelEncryptionConfig() {
        this.to('cloudfront:CreateFieldLevelEncryptionConfig');
        return this;
    }
    /**
     * This action creates a field-level encryption profile.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateFieldLevelEncryptionProfile.html
     */
    toCreateFieldLevelEncryptionProfile() {
        this.to('cloudfront:CreateFieldLevelEncryptionProfile');
        return this;
    }
    /**
     * This action creates a new invalidation batch request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateInvalidation.html
     */
    toCreateInvalidation() {
        this.to('cloudfront:CreateInvalidation');
        return this;
    }
    /**
     * This action adds a new origin request policy to CloudFront.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateOriginRequestPolicy.html
     */
    toCreateOriginRequestPolicy() {
        this.to('cloudfront:CreateOriginRequestPolicy');
        return this;
    }
    /**
     * This action adds a new public key to CloudFront.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreatePublicKey.html
     */
    toCreatePublicKey() {
        this.to('cloudfront:CreatePublicKey');
        return this;
    }
    /**
     * This action creates a new RTMP distribution.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateStreamingDistribution.html
     */
    toCreateStreamingDistribution() {
        this.to('cloudfront:CreateStreamingDistribution');
        return this;
    }
    /**
     * This action creates a new RTMP distribution with tags.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateStreamingDistributionWithTags.html
     */
    toCreateStreamingDistributionWithTags() {
        this.to('cloudfront:CreateStreamingDistributionWithTags');
        return this;
    }
    /**
     * This action deletes a cache policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteCachePolicy.html
     */
    toDeleteCachePolicy() {
        this.to('cloudfront:DeleteCachePolicy');
        return this;
    }
    /**
     * This action deletes a CloudFront origin access identity.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteCloudFrontOriginAccessIdentity.html
     */
    toDeleteCloudFrontOriginAccessIdentity() {
        this.to('cloudfront:DeleteCloudFrontOriginAccessIdentity');
        return this;
    }
    /**
     * This action deletes a web distribution.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteDistribution.html
     */
    toDeleteDistribution() {
        this.to('cloudfront:DeleteDistribution');
        return this;
    }
    /**
     * This action deletes a field-level encryption configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteFieldLevelEncryptionConfig.html
     */
    toDeleteFieldLevelEncryptionConfig() {
        this.to('cloudfront:DeleteFieldLevelEncryptionConfig');
        return this;
    }
    /**
     * This action deletes a field-level encryption profile.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteFieldLevelEncryptionProfile.html
     */
    toDeleteFieldLevelEncryptionProfile() {
        this.to('cloudfront:DeleteFieldLevelEncryptionProfile');
        return this;
    }
    /**
     * This action deletes an origin request policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteOriginRequestPolicy.html
     */
    toDeleteOriginRequestPolicy() {
        this.to('cloudfront:DeleteOriginRequestPolicy');
        return this;
    }
    /**
     * This action deletes a public key from CloudFront.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeletePublicKey.html
     */
    toDeletePublicKey() {
        this.to('cloudfront:DeletePublicKey');
        return this;
    }
    /**
     * This action deletes an RTMP distribution.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteStreamingDistribution.html
     */
    toDeleteStreamingDistribution() {
        this.to('cloudfront:DeleteStreamingDistribution');
        return this;
    }
    /**
     * Get the cache policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetCachePolicy.html
     */
    toGetCachePolicy() {
        this.to('cloudfront:GetCachePolicy');
        return this;
    }
    /**
     * Get the cache policy configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetCachePolicyConfig.html
     */
    toGetCachePolicyConfig() {
        this.to('cloudfront:GetCachePolicyConfig');
        return this;
    }
    /**
     * Get the information about a CloudFront origin access identity.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetCloudFrontOriginAccessIdentity.html
     */
    toGetCloudFrontOriginAccessIdentity() {
        this.to('cloudfront:GetCloudFrontOriginAccessIdentity');
        return this;
    }
    /**
     * Get the configuration information about a Cloudfront origin access identity.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetCloudFrontOriginAccessIdentityConfig.html
     */
    toGetCloudFrontOriginAccessIdentityConfig() {
        this.to('cloudfront:GetCloudFrontOriginAccessIdentityConfig');
        return this;
    }
    /**
     * Get the information about a web distribution.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetDistribution.html
     */
    toGetDistribution() {
        this.to('cloudfront:GetDistribution');
        return this;
    }
    /**
     * Get the configuration information about a distribution.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetDistributionConfig.html
     */
    toGetDistributionConfig() {
        this.to('cloudfront:GetDistributionConfig');
        return this;
    }
    /**
     * Get the field-level encryption configuration information.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetFieldLevelEncryption.html
     */
    toGetFieldLevelEncryption() {
        this.to('cloudfront:GetFieldLevelEncryption');
        return this;
    }
    /**
     * Get the field-level encryption configuration information.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetFieldLevelEncryptionConfig.html
     */
    toGetFieldLevelEncryptionConfig() {
        this.to('cloudfront:GetFieldLevelEncryptionConfig');
        return this;
    }
    /**
     * Get the field-level encryption configuration information.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetFieldLevelEncryptionProfile.html
     */
    toGetFieldLevelEncryptionProfile() {
        this.to('cloudfront:GetFieldLevelEncryptionProfile');
        return this;
    }
    /**
     * Get the field-level encryption profile configuration information.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetFieldLevelEncryptionProfileConfig.html
     */
    toGetFieldLevelEncryptionProfileConfig() {
        this.to('cloudfront:GetFieldLevelEncryptionProfileConfig');
        return this;
    }
    /**
     * Get the information about an invalidation.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetInvalidation.html
     */
    toGetInvalidation() {
        this.to('cloudfront:GetInvalidation');
        return this;
    }
    /**
     * Get the origin request policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetOriginRequestPolicy.html
     */
    toGetOriginRequestPolicy() {
        this.to('cloudfront:GetOriginRequestPolicy');
        return this;
    }
    /**
     * Get the origin request policy configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetOriginRequestPolicyConfig.html
     */
    toGetOriginRequestPolicyConfig() {
        this.to('cloudfront:GetOriginRequestPolicyConfig');
        return this;
    }
    /**
     * Get the public key information.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetPublicKey.html
     */
    toGetPublicKey() {
        this.to('cloudfront:GetPublicKey');
        return this;
    }
    /**
     * Get the public key configuration information.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetPublicKeyConfig.html
     */
    toGetPublicKeyConfig() {
        this.to('cloudfront:GetPublicKeyConfig');
        return this;
    }
    /**
     * Get the information about an RTMP distribution.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetStreamingDistribution.html
     */
    toGetStreamingDistribution() {
        this.to('cloudfront:GetStreamingDistribution');
        return this;
    }
    /**
     * Get the configuration information about a streaming distribution.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetStreamingDistributionConfig.html
     */
    toGetStreamingDistributionConfig() {
        this.to('cloudfront:GetStreamingDistributionConfig');
        return this;
    }
    /**
     * List all cache policies that have been created in CloudFront for this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListCachePolicies.html
     */
    toListCachePolicies() {
        this.to('cloudfront:ListCachePolicies');
        return this;
    }
    /**
     * List your CloudFront origin access identities.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListCloudFrontOriginAccessIdentities.html
     */
    toListCloudFrontOriginAccessIdentities() {
        this.to('cloudfront:ListCloudFrontOriginAccessIdentities');
        return this;
    }
    /**
     * List the distributions associated with your AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributions.html
     */
    toListDistributions() {
        this.to('cloudfront:ListDistributions');
        return this;
    }
    /**
     * List distribution IDs for distributions that have a cache behavior that's associated with the specified cache policy.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionsByCachePolicyId.html
     */
    toListDistributionsByCachePolicyId() {
        this.to('cloudfront:ListDistributionsByCachePolicyId');
        return this;
    }
    /**
     * List distribution IDs for distributions that have a cache behavior that's associated with the specified origin request policy.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionsByOriginRequestPolicyId.html
     */
    toListDistributionsByOriginRequestPolicyId() {
        this.to('cloudfront:ListDistributionsByOriginRequestPolicyId');
        return this;
    }
    /**
     * List the distributions associated with your AWS account with given AWS WAF web ACL.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionsByWebACLId.html
     */
    toListDistributionsByWebACLId() {
        this.to('cloudfront:ListDistributionsByWebACLId');
        return this;
    }
    /**
     * List all field-level encryption configurations that have been created in CloudFront for this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListFieldLevelEncryptionConfigs.html
     */
    toListFieldLevelEncryptionConfigs() {
        this.to('cloudfront:ListFieldLevelEncryptionConfigs');
        return this;
    }
    /**
     * List all field-level encryption profiles that have been created in CloudFront for this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListFieldLevelEncryptionProfiles.html
     */
    toListFieldLevelEncryptionProfiles() {
        this.to('cloudfront:ListFieldLevelEncryptionProfiles');
        return this;
    }
    /**
     * List your invalidation batches.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListInvalidations.html
     */
    toListInvalidations() {
        this.to('cloudfront:ListInvalidations');
        return this;
    }
    /**
     * List all origin request policies that have been created in CloudFront for this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListOriginRequestPolicies.html
     */
    toListOriginRequestPolicies() {
        this.to('cloudfront:ListOriginRequestPolicies');
        return this;
    }
    /**
     * List all public keys that have been added to CloudFront for this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListPublicKeys.html
     */
    toListPublicKeys() {
        this.to('cloudfront:ListPublicKeys');
        return this;
    }
    /**
     * List your RTMP distributions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListStreamingDistributions.html
     */
    toListStreamingDistributions() {
        this.to('cloudfront:ListStreamingDistributions');
        return this;
    }
    /**
     * List tags for a CloudFront resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('cloudfront:ListTagsForResource');
        return this;
    }
    /**
     * Add tags to a CloudFront resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('cloudfront:TagResource');
        return this;
    }
    /**
     * Remove tags from a CloudFront resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('cloudfront:UntagResource');
        return this;
    }
    /**
     * This action updates a cache policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateCachePolicy.html
     */
    toUpdateCachePolicy() {
        this.to('cloudfront:UpdateCachePolicy');
        return this;
    }
    /**
     * This action sets the configuration for a CloudFront origin access identity.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateCloudFrontOriginAccessIdentity.html
     */
    toUpdateCloudFrontOriginAccessIdentity() {
        this.to('cloudfront:UpdateCloudFrontOriginAccessIdentity');
        return this;
    }
    /**
     * This action updates the configuration for a web distribution.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateDistribution.html
     */
    toUpdateDistribution() {
        this.to('cloudfront:UpdateDistribution');
        return this;
    }
    /**
     * This action updates a field-level encryption configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateFieldLevelEncryptionConfig.html
     */
    toUpdateFieldLevelEncryptionConfig() {
        this.to('cloudfront:UpdateFieldLevelEncryptionConfig');
        return this;
    }
    /**
     * This action updates a field-level encryption profile.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateFieldLevelEncryptionProfile.html
     */
    toUpdateFieldLevelEncryptionProfile() {
        this.to('cloudfront:UpdateFieldLevelEncryptionProfile');
        return this;
    }
    /**
     * This action updates an origin request policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateOriginRequestPolicy.html
     */
    toUpdateOriginRequestPolicy() {
        this.to('cloudfront:UpdateOriginRequestPolicy');
        return this;
    }
    /**
     * This action updates public key information.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdatePublicKey.html
     */
    toUpdatePublicKey() {
        this.to('cloudfront:UpdatePublicKey');
        return this;
    }
    /**
     * This action updates the configuration for an RTMP distribution.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateStreamingDistribution.html
     */
    toUpdateStreamingDistribution() {
        this.to('cloudfront:UpdateStreamingDistribution');
        return this;
    }
    /**
     * Adds a resource of type distribution to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-working-with.html
     *
     * @param distributionId - Identifier for the distributionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDistribution(distributionId, account, partition) {
        var arn = 'arn:${Partition}:cloudfront::${Account}:distribution/${DistributionId}';
        arn = arn.replace('${DistributionId}', distributionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type streaming-distribution to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-working-with.html
     *
     * @param distributionId - Identifier for the distributionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStreamingDistribution(distributionId, account, partition) {
        var arn = 'arn:${Partition}:cloudfront::${Account}:streaming-distribution/${DistributionId}';
        arn = arn.replace('${DistributionId}', distributionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type origin-access-identity to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html#private-content-restricting-access-to-s3-overview
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onOriginAccessIdentity(id, account, partition) {
        var arn = 'arn:${Partition}:cloudfront::${Account}:origin-access-identity/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type field-level-encryption to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/field-level-encryption.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onFieldLevelEncryption(id, account, partition) {
        var arn = 'arn:${Partition}:cloudfront::${Account}:field-level-encryption/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type field-level-encryption-profile to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/field-level-encryption.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onFieldLevelEncryptionProfile(id, account, partition) {
        var arn = 'arn:${Partition}:cloudfront::${Account}:field-level-encryption-profile/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type cache-policy to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/working-with-policies.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCachePolicy(id, account, partition) {
        var arn = 'arn:${Partition}:cloudfront::${Account}:cache-policy/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type origin-request-policy to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/working-with-policies.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onOriginRequestPolicy(id, account, partition) {
        var arn = 'arn:${Partition}:cloudfront::${Account}:origin-request-policy/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Cloudfront = Cloudfront;
//# sourceMappingURL=data:application/json;base64,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