"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Logs = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [logs](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncloudwatchlogs.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Logs extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [logs](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncloudwatchlogs.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'logs';
        this.accessLevelList = {
            "Write": [
                "AssociateKmsKey",
                "CancelExportTask",
                "CreateExportTask",
                "CreateLogDelivery",
                "CreateLogGroup",
                "CreateLogStream",
                "DeleteDestination",
                "DeleteLogDelivery",
                "DeleteLogGroup",
                "DeleteLogStream",
                "DeleteMetricFilter",
                "DeleteResourcePolicy",
                "DeleteRetentionPolicy",
                "DeleteSubscriptionFilter",
                "DisassociateKmsKey",
                "PutDestination",
                "PutDestinationPolicy",
                "PutLogEvents",
                "PutMetricFilter",
                "PutResourcePolicy",
                "PutRetentionPolicy",
                "PutSubscriptionFilter",
                "TagLogGroup",
                "UntagLogGroup",
                "UpdateLogDelivery"
            ],
            "List": [
                "DescribeDestinations",
                "DescribeExportTasks",
                "DescribeLogGroups",
                "DescribeLogStreams",
                "DescribeMetricFilters",
                "DescribeQueries",
                "DescribeResourcePolicies",
                "DescribeSubscriptionFilters",
                "ListLogDeliveries",
                "ListTagsLogGroup"
            ],
            "Read": [
                "FilterLogEvents",
                "GetLogDelivery",
                "GetLogEvents",
                "GetLogGroupFields",
                "GetLogRecord",
                "GetQueryResults",
                "StartQuery",
                "StopQuery",
                "TestMetricFilter"
            ]
        };
    }
    /**
     * Associates the specified AWS Key Management Service (AWS KMS) customer master key (CMK) with the specified log group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_AssociateKmsKey.html
     */
    toAssociateKmsKey() {
        this.to('logs:AssociateKmsKey');
        return this;
    }
    /**
     * Cancels an export task if it is in PENDING or RUNNING state
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CancelExportTask.html
     */
    toCancelExportTask() {
        this.to('logs:CancelExportTask');
        return this;
    }
    /**
     * Creates an ExportTask which allows you to efficiently export data from a Log Group to your Amazon S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CreateExportTask.html
     */
    toCreateExportTask() {
        this.to('logs:CreateExportTask');
        return this;
    }
    /**
     * Creates the log delivery
     *
     * Access Level: Write
     */
    toCreateLogDelivery() {
        this.to('logs:CreateLogDelivery');
        return this;
    }
    /**
     * Creates a new log group with the specified name
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CreateLogGroup.html
     */
    toCreateLogGroup() {
        this.to('logs:CreateLogGroup');
        return this;
    }
    /**
     * Creates a new log stream with the specified name
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CreateLogStream.html
     */
    toCreateLogStream() {
        this.to('logs:CreateLogStream');
        return this;
    }
    /**
     * Deletes the destination with the specified name and eventually disables all the subscription filters that publish to it
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteDestination.html
     */
    toDeleteDestination() {
        this.to('logs:DeleteDestination');
        return this;
    }
    /**
     * Deletes the log delivery information for specified log delivery
     *
     * Access Level: Write
     */
    toDeleteLogDelivery() {
        this.to('logs:DeleteLogDelivery');
        return this;
    }
    /**
     * Deletes the log group with the specified name and permanently deletes all the archived log events associated with it
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteLogGroup.html
     */
    toDeleteLogGroup() {
        this.to('logs:DeleteLogGroup');
        return this;
    }
    /**
     * Deletes a log stream and permanently deletes all the archived log events associated with it
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteLogStream.html
     */
    toDeleteLogStream() {
        this.to('logs:DeleteLogStream');
        return this;
    }
    /**
     * Deletes a metric filter associated with the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteMetricFilter.html
     */
    toDeleteMetricFilter() {
        this.to('logs:DeleteMetricFilter');
        return this;
    }
    /**
     * Deletes a resource policy from this account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        this.to('logs:DeleteResourcePolicy');
        return this;
    }
    /**
     * Deletes the retention policy of the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteRetentionPolicy.html
     */
    toDeleteRetentionPolicy() {
        this.to('logs:DeleteRetentionPolicy');
        return this;
    }
    /**
     * Deletes a subscription filter associated with the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteSubscriptionFilter.html
     */
    toDeleteSubscriptionFilter() {
        this.to('logs:DeleteSubscriptionFilter');
        return this;
    }
    /**
     * Returns all the destinations that are associated with the AWS account making the request
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeDestinations.html
     */
    toDescribeDestinations() {
        this.to('logs:DescribeDestinations');
        return this;
    }
    /**
     * Returns all the export tasks that are associated with the AWS account making the request
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeExportTasks.html
     */
    toDescribeExportTasks() {
        this.to('logs:DescribeExportTasks');
        return this;
    }
    /**
     * Returns all the log groups that are associated with the AWS account making the request
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeLogGroups.html
     */
    toDescribeLogGroups() {
        this.to('logs:DescribeLogGroups');
        return this;
    }
    /**
     * Returns all the log streams that are associated with the specified log group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeLogStreams.html
     */
    toDescribeLogStreams() {
        this.to('logs:DescribeLogStreams');
        return this;
    }
    /**
     * Returns all the metrics filters associated with the specified log group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeMetricFilters.html
     */
    toDescribeMetricFilters() {
        this.to('logs:DescribeMetricFilters');
        return this;
    }
    /**
     * Returns a list of CloudWatch Logs Insights queries that are scheduled, executing, or have been executed recently in this account. You can request all queries, or limit it to queries of a specific log group or queries with a certain status.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeQueries.html
     */
    toDescribeQueries() {
        this.to('logs:DescribeQueries');
        return this;
    }
    /**
     * Return all the resource policies in this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeResourcePolicies.html
     */
    toDescribeResourcePolicies() {
        this.to('logs:DescribeResourcePolicies');
        return this;
    }
    /**
     * Returns all the subscription filters associated with the specified log group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeSubscriptionFilters.html
     */
    toDescribeSubscriptionFilters() {
        this.to('logs:DescribeSubscriptionFilters');
        return this;
    }
    /**
     * Disassociates the associated AWS Key Management Service (AWS KMS) customer master key (CMK) from the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DisassociateKmsKey.html
     */
    toDisassociateKmsKey() {
        this.to('logs:DisassociateKmsKey');
        return this;
    }
    /**
     * Retrieves log events, optionally filtered by a filter pattern from the specified log group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_FilterLogEvents.html
     */
    toFilterLogEvents() {
        this.to('logs:FilterLogEvents');
        return this;
    }
    /**
     * Gets the log delivery information for specified log delivery
     *
     * Access Level: Read
     */
    toGetLogDelivery() {
        this.to('logs:GetLogDelivery');
        return this;
    }
    /**
     * Retrieves log events from the specified log stream
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetLogEvents.html
     */
    toGetLogEvents() {
        this.to('logs:GetLogEvents');
        return this;
    }
    /**
     * Returns a list of the fields that are included in log events in the specified log group, along with the percentage of log events that contain each field. The search is limited to a time period that you specify.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetLogGroupFields.html
     */
    toGetLogGroupFields() {
        this.to('logs:GetLogGroupFields');
        return this;
    }
    /**
     * Retrieves all the fields and values of a single log event. All fields are retrieved, even if the original query that produced the logRecordPointer retrieved only a subset of fields. Fields are returned as field name/field value pairs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetLogRecord.html
     */
    toGetLogRecord() {
        this.to('logs:GetLogRecord');
        return this;
    }
    /**
     * Returns the results from the specified query. If the query is in progress, partial results of that current execution are returned. Only the fields requested in the query are returned.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetQueryResults.html
     */
    toGetQueryResults() {
        this.to('logs:GetQueryResults');
        return this;
    }
    /**
     * Lists all the log deliveries for specified account and/or log source
     *
     * Access Level: List
     */
    toListLogDeliveries() {
        this.to('logs:ListLogDeliveries');
        return this;
    }
    /**
     * Lists the tags for the specified log group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_ListTagsLogGroup.html
     */
    toListTagsLogGroup() {
        this.to('logs:ListTagsLogGroup');
        return this;
    }
    /**
     * Creates or updates a Destination
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutDestination.html
     */
    toPutDestination() {
        this.to('logs:PutDestination');
        return this;
    }
    /**
     * Creates or updates an access policy associated with an existing Destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutDestinationPolicy.html
     */
    toPutDestinationPolicy() {
        this.to('logs:PutDestinationPolicy');
        return this;
    }
    /**
     * Uploads a batch of log events to the specified log stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutLogEvents.html
     */
    toPutLogEvents() {
        this.to('logs:PutLogEvents');
        return this;
    }
    /**
     * Creates or updates a metric filter and associates it with the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutMetricFilter.html
     */
    toPutMetricFilter() {
        this.to('logs:PutMetricFilter');
        return this;
    }
    /**
     * Creates or updates a resource policy allowing other AWS services to put log events to this account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        this.to('logs:PutResourcePolicy');
        return this;
    }
    /**
     * Sets the retention of the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutRetentionPolicy.html
     */
    toPutRetentionPolicy() {
        this.to('logs:PutRetentionPolicy');
        return this;
    }
    /**
     * Creates or updates a subscription filter and associates it with the specified log group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutSubscriptionFilter.html
     */
    toPutSubscriptionFilter() {
        this.to('logs:PutSubscriptionFilter');
        return this;
    }
    /**
     * Schedules a query of a log group using CloudWatch Logs Insights. You specify the log group and time range to query, and the query string to use.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_StartQuery.html
     */
    toStartQuery() {
        this.to('logs:StartQuery');
        return this;
    }
    /**
     * Stops a CloudWatch Logs Insights query that is in progress. If the query has already ended, the operation returns an error indicating that the specified query is not running.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_StopQuery.html
     */
    toStopQuery() {
        this.to('logs:StopQuery');
        return this;
    }
    /**
     * Adds or updates the specified tags for the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_TagLogGroup.html
     */
    toTagLogGroup() {
        this.to('logs:TagLogGroup');
        return this;
    }
    /**
     * Tests the filter pattern of a metric filter against a sample of log event messages
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_TestMetricFilter.html
     */
    toTestMetricFilter() {
        this.to('logs:TestMetricFilter');
        return this;
    }
    /**
     * Removes the specified tags from the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UntagLogGroup.html
     */
    toUntagLogGroup() {
        this.to('logs:UntagLogGroup');
        return this;
    }
    /**
     * Updates the log delivery information for specified log delivery
     *
     * Access Level: Write
     */
    toUpdateLogDelivery() {
        this.to('logs:UpdateLogDelivery');
        return this;
    }
    /**
     * Adds a resource of type log-group to the statement
     *
     * @param logGroupName - Identifier for the logGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLogGroup(logGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:logs:${Region}:${Account}:log-group:${LogGroupName}';
        arn = arn.replace('${LogGroupName}', logGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type log-stream to the statement
     *
     * @param logGroupName - Identifier for the logGroupName.
     * @param logStreamName - Identifier for the logStreamName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLogStream(logGroupName, logStreamName, account, region, partition) {
        var arn = 'arn:${Partition}:logs:${Region}:${Account}:log-group:${LogGroupName}:log-stream:${LogStreamName}';
        arn = arn.replace('${LogGroupName}', logGroupName);
        arn = arn.replace('${LogStreamName}', logStreamName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Logs = Logs;
//# sourceMappingURL=data:application/json;base64,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