"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Connect = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [connect](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonconnect.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Connect extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [connect](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonconnect.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'connect';
        this.accessLevelList = {
            "Write": [
                "AssociateRoutingProfileQueues",
                "CreateContactFlow",
                "CreateInstance",
                "CreateRoutingProfile",
                "CreateUser",
                "DeleteUser",
                "DestroyInstance",
                "DisassociateRoutingProfileQueues",
                "GetFederationTokens",
                "ModifyInstance",
                "ResumeContactRecording",
                "StartChatContact",
                "StartContactRecording",
                "StartOutboundVoiceContact",
                "StopContact",
                "StopContactRecording",
                "SuspendContactRecording",
                "UpdateContactAttributes",
                "UpdateContactFlowContent",
                "UpdateContactFlowName",
                "UpdateRoutingProfileConcurrency",
                "UpdateRoutingProfileDefaultOutboundQueue",
                "UpdateRoutingProfileName",
                "UpdateRoutingProfileQueues",
                "UpdateUserHierarchy",
                "UpdateUserIdentityInfo",
                "UpdateUserPhoneConfig",
                "UpdateUserRoutingProfile",
                "UpdateUserSecurityProfiles"
            ],
            "Read": [
                "DescribeContactFlow",
                "DescribeInstance",
                "DescribeRoutingProfile",
                "DescribeUser",
                "DescribeUserHierarchyGroup",
                "DescribeUserHierarchyStructure",
                "GetContactAttributes",
                "GetCurrentMetricData",
                "GetFederationToken",
                "GetMetricData",
                "ListRoutingProfileQueues",
                "ListTagsForResource"
            ],
            "List": [
                "ListContactFlows",
                "ListHoursOfOperations",
                "ListInstances",
                "ListPhoneNumbers",
                "ListPrompts",
                "ListQueues",
                "ListRoutingProfiles",
                "ListSecurityProfiles",
                "ListUserHierarchyGroups",
                "ListUsers"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permissions to associate queues with a routing profile in an Amazon Connect instance.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_AssociateRoutingProfileQueues.html
     */
    toAssociateRoutingProfileQueues() {
        this.to('connect:AssociateRoutingProfileQueues');
        return this;
    }
    /**
     * Grants permissions to create a contact flow in an Amazon Connect instance.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateContactFlow.html
     */
    toCreateContactFlow() {
        this.to('connect:CreateContactFlow');
        return this;
    }
    /**
     * Grants permissions to create a new Amazon Connect instance. The associated required actions grant permissions to configure instance settings.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ds:CreateAlias
     * - ds:DeleteDirectory
     * - ds:DescribeDirectories
     * - firehose:DescribeDeliveryStream
     * - firehose:ListDeliveryStreams
     * - iam:CreateServiceLinkedRole
     * - kinesis:DescribeStream
     * - kinesis:ListStreams
     * - kms:CreateGrant
     * - kms:DescribeKey
     * - kms:ListAliases
     * - kms:RetireGrant
     * - s3:CreateBucket
     * - s3:GetBucketLocation
     * - s3:ListAllMyBuckets
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/amazon-connect-instances.html
     */
    toCreateInstance() {
        this.to('connect:CreateInstance');
        return this;
    }
    /**
     * Grants permission to create a routing profile in an Amazon Connect instance.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateRoutingProfile.html
     */
    toCreateRoutingProfile() {
        this.to('connect:CreateRoutingProfile');
        return this;
    }
    /**
     * Grants permission to create a user for the specified Amazon Connect instance.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateUser.html
     */
    toCreateUser() {
        this.to('connect:CreateUser');
        return this;
    }
    /**
     * Grants permissions to delete a user in an Amazon Connect instance.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteUser.html
     */
    toDeleteUser() {
        this.to('connect:DeleteUser');
        return this;
    }
    /**
     * Grants permissions to describe a contact flow in an Amazon Connect instance.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeContactFlow.html
     */
    toDescribeContactFlow() {
        this.to('connect:DescribeContactFlow');
        return this;
    }
    /**
     * Grants permissions to view details of an Amazon Connect instance. This is required to create an instance.
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - firehose:DescribeDeliveryStream
     * - firehose:ListDeliveryStreams
     * - kinesis:DescribeStream
     * - kinesis:ListStreams
     * - kms:DescribeKey
     * - kms:ListAliases
     * - s3:ListAllMyBuckets
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/amazon-connect-instances.html
     */
    toDescribeInstance() {
        this.to('connect:DescribeInstance');
        return this;
    }
    /**
     * Grants permissions to describe a routing profile in an Amazon Connect instance.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeRoutingProfile.html
     */
    toDescribeRoutingProfile() {
        this.to('connect:DescribeRoutingProfile');
        return this;
    }
    /**
     * Grants permissions to describe a user in an Amazon Connect instance.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeUser.html
     */
    toDescribeUser() {
        this.to('connect:DescribeUser');
        return this;
    }
    /**
     * Grants permissions to describe a hierarchy group for an Amazon Connect instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeUserHierarchyGroup.html
     */
    toDescribeUserHierarchyGroup() {
        this.to('connect:DescribeUserHierarchyGroup');
        return this;
    }
    /**
     * Grants permissions to describe the hierarchy structure for an Amazon Connect instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeUserHierarchyStructure.html
     */
    toDescribeUserHierarchyStructure() {
        this.to('connect:DescribeUserHierarchyStructure');
        return this;
    }
    /**
     * Grants permissions to delete an Amazon Connect instance. When you remove an instance, the link to an existing AWS directory is also removed.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/amazon-connect-instances.html
     */
    toDestroyInstance() {
        this.to('connect:DestroyInstance');
        return this;
    }
    /**
     * Grants permissions to disassociate queues from a routing profile in an Amazon Connect instance.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DisassociateRoutingProfileQueues.html
     */
    toDisassociateRoutingProfileQueues() {
        this.to('connect:DisassociateRoutingProfileQueues');
        return this;
    }
    /**
     * Grants permissions to retrieve the contact attributes for the specified contact.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_GetContactAttributes.html
     */
    toGetContactAttributes() {
        this.to('connect:GetContactAttributes');
        return this;
    }
    /**
     * Grants permissions to retrieve current metric data for the queues in an Amazon Connect instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_GetCurrentMetricData.html
     */
    toGetCurrentMetricData() {
        this.to('connect:GetCurrentMetricData');
        return this;
    }
    /**
     * Allows federation into an instance when using SAML-based authentication for identity management.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_GetFederationToken.html
     */
    toGetFederationToken() {
        this.to('connect:GetFederationToken');
        return this;
    }
    /**
     * Grants permissions to federate in to an Amazon Connect instance (Log in as administrator functionality in the AWS console).
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - connect:DescribeInstance
     * - connect:ListInstances
     * - ds:DescribeDirectories
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/amazon-connect-instances.html
     */
    toGetFederationTokens() {
        this.to('connect:GetFederationTokens');
        return this;
    }
    /**
     * Grants permissions to retrieve historical metric data for queues in an Amazon Connect instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_GetMetricData.html
     */
    toGetMetricData() {
        this.to('connect:GetMetricData');
        return this;
    }
    /**
     * Grants permissions to list contact flow resources in an Amazon Connect instance.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListContactFlows.html
     */
    toListContactFlows() {
        this.to('connect:ListContactFlows');
        return this;
    }
    /**
     * Grants permissions to list hours of operation resources in an Amazon Connect instance.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListHoursOfOperations.html
     */
    toListHoursOfOperations() {
        this.to('connect:ListHoursOfOperations');
        return this;
    }
    /**
     * Grants permissions to view the Amazon Connect instances associated with an AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/amazon-connect-instances.html
     */
    toListInstances() {
        this.to('connect:ListInstances');
        return this;
    }
    /**
     * Grants permissions to list phone number resources in an Amazon Connect instance.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListPhoneNumbers.html
     */
    toListPhoneNumbers() {
        this.to('connect:ListPhoneNumbers');
        return this;
    }
    /**
     * Grants permissions to list prompt resources in an Amazon Connect instance.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListPrompts.html
     */
    toListPrompts() {
        this.to('connect:ListPrompts');
        return this;
    }
    /**
     * Grants permissions to list queue resources in an Amazon Connect instance.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListQueues.html
     */
    toListQueues() {
        this.to('connect:ListQueues');
        return this;
    }
    /**
     * Grants permissions to list queue resources in a routing profile in an Amazon Connect instance.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListRoutingProfileQueues.html
     */
    toListRoutingProfileQueues() {
        this.to('connect:ListRoutingProfileQueues');
        return this;
    }
    /**
     * Grants permissions to list routing profile resources in an Amazon Connect instance.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListRoutingProfiles.html
     */
    toListRoutingProfiles() {
        this.to('connect:ListRoutingProfiles');
        return this;
    }
    /**
     * Grants permissions to list security profile resources in an Amazon Connect instance.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListSecurityProfiles.html
     */
    toListSecurityProfiles() {
        this.to('connect:ListSecurityProfiles');
        return this;
    }
    /**
     * Grants permissions to list tags for an Amazon Connect resource.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('connect:ListTagsForResource');
        return this;
    }
    /**
     * Grants permissions to list the hierarchy group resources in an Amazon Connect instance.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListUserHierarchyGroups.html
     */
    toListUserHierarchyGroups() {
        this.to('connect:ListUserHierarchyGroups');
        return this;
    }
    /**
     * Grants permissions to list user resources in an Amazon Connect instance.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListUsers.html
     */
    toListUsers() {
        this.to('connect:ListUsers');
        return this;
    }
    /**
     * Grants permissions to modify configuration settings for an existing Amazon Connect instance. The associated required actions grant permission modify the settings for the instance.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - firehose:DescribeDeliveryStream
     * - firehose:ListDeliveryStreams
     * - kinesis:DescribeStream
     * - kinesis:ListStreams
     * - kms:CreateGrant
     * - kms:DescribeKey
     * - kms:ListAliases
     * - kms:RetireGrant
     * - s3:CreateBucket
     * - s3:GetBucketLocation
     * - s3:ListAllMyBuckets
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/amazon-connect-instances.html
     */
    toModifyInstance() {
        this.to('connect:ModifyInstance');
        return this;
    }
    /**
     * Grants permissions to resume recording for the specified contact.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ResumeContactRecording.html
     */
    toResumeContactRecording() {
        this.to('connect:ResumeContactRecording');
        return this;
    }
    /**
     * Grants permissions to initiate a chat using the Amazon Connect API.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_StartChatContact.html
     */
    toStartChatContact() {
        this.to('connect:StartChatContact');
        return this;
    }
    /**
     * Grants permissions to start recording for the specified contact.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_StartContactRecording.html
     */
    toStartContactRecording() {
        this.to('connect:StartContactRecording');
        return this;
    }
    /**
     * Grants permissions to initiate outbound calls using the Amazon Connect API.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_StartOutboundVoiceContact.html
     */
    toStartOutboundVoiceContact() {
        this.to('connect:StartOutboundVoiceContact');
        return this;
    }
    /**
     * Grants permissions to stop contacts that were initiated using the Amazon Connect API. If you use this operation on an active contact the contact ends, even if the agent is active on a call with a customer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_StopContact.html
     */
    toStopContact() {
        this.to('connect:StopContact');
        return this;
    }
    /**
     * Grants permissions to stop recording for the specified contact.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_StopContactRecording.html
     */
    toStopContactRecording() {
        this.to('connect:StopContactRecording');
        return this;
    }
    /**
     * Grants permissions to suspend recording for the specified contact.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_SuspendContactRecording.html
     */
    toSuspendContactRecording() {
        this.to('connect:SuspendContactRecording');
        return this;
    }
    /**
     * Grants permissions to tag an Amazon Connect resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('connect:TagResource');
        return this;
    }
    /**
     * Grants permissions to untag an Amazon Connect resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('connect:UntagResource');
        return this;
    }
    /**
     * Grants permissions to create or update the contact attributes associated with the specified contact.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateContactAttributes.html
     */
    toUpdateContactAttributes() {
        this.to('connect:UpdateContactAttributes');
        return this;
    }
    /**
     * Grants permissions to update contact flow content in an Amazon Connect instance.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateContactFlowContent.html
     */
    toUpdateContactFlowContent() {
        this.to('connect:UpdateContactFlowContent');
        return this;
    }
    /**
     * Grants permissions to update the name and description of a contact flow in an Amazon Connect instance.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateContactFlowName.html
     */
    toUpdateContactFlowName() {
        this.to('connect:UpdateContactFlowName');
        return this;
    }
    /**
     * Grants permissions to update the concurrency in a routing profile in an Amazon Connect instance.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateRoutingProfileConcurrency.html
     */
    toUpdateRoutingProfileConcurrency() {
        this.to('connect:UpdateRoutingProfileConcurrency');
        return this;
    }
    /**
     * Grants permissions to update the outbound queue in a routing profile in an Amazon Connect instance.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateRoutingProfileDefaultOutboundQueue.html
     */
    toUpdateRoutingProfileDefaultOutboundQueue() {
        this.to('connect:UpdateRoutingProfileDefaultOutboundQueue');
        return this;
    }
    /**
     * Grants permissions to update a routing profile name and description in an Amazon Connect instance.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateRoutingProfileName.html
     */
    toUpdateRoutingProfileName() {
        this.to('connect:UpdateRoutingProfileName');
        return this;
    }
    /**
     * Grants permissions to update the queues in routing profile in an Amazon Connect instance.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateRoutingProfileQueues.html
     */
    toUpdateRoutingProfileQueues() {
        this.to('connect:UpdateRoutingProfileQueues');
        return this;
    }
    /**
     * Grants permissions to update a hierarchy group for a user in an Amazon Connect instance.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateUserHierarchy.html
     */
    toUpdateUserHierarchy() {
        this.to('connect:UpdateUserHierarchy');
        return this;
    }
    /**
     * Grants permissions to update identity information for a user in an Amazon Connect instance.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateUserIdentityInfo.html
     */
    toUpdateUserIdentityInfo() {
        this.to('connect:UpdateUserIdentityInfo');
        return this;
    }
    /**
     * Grants permissions to update phone configuration settings for a user in an Amazon Connect instance.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateUserPhoneConfig.html
     */
    toUpdateUserPhoneConfig() {
        this.to('connect:UpdateUserPhoneConfig');
        return this;
    }
    /**
     * Grants permissions to update a routing profile for a user in an Amazon Connect instance.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateUserRoutingProfile.html
     */
    toUpdateUserRoutingProfile() {
        this.to('connect:UpdateUserRoutingProfile');
        return this;
    }
    /**
     * Grants permissions to update security profiles for a user in an Amazon Connect instance.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateUserSecurityProfiles.html
     */
    toUpdateUserSecurityProfiles() {
        this.to('connect:UpdateUserSecurityProfiles');
        return this;
    }
    /**
     * Adds a resource of type instance to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-instances.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onInstance(instanceId, account, region, partition) {
        var arn = 'arn:${Partition}:connect:${Region}:${Account}:instance/${InstanceId}';
        arn = arn.replace('${InstanceId}', instanceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type contact to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/connect-contact-attributes.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param contactId - Identifier for the contactId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onContact(instanceId, contactId, account, region, partition) {
        var arn = 'arn:${Partition}:connect:${Region}:${Account}:instance/${InstanceId}/contact/${ContactId}';
        arn = arn.replace('${InstanceId}', instanceId);
        arn = arn.replace('${ContactId}', contactId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type user to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/connect-agents.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param userId - Identifier for the userId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onUser(instanceId, userId, account, region, partition) {
        var arn = 'arn:${Partition}:connect:${Region}:${Account}:instance/${InstanceId}/agent/${UserId}';
        arn = arn.replace('${InstanceId}', instanceId);
        arn = arn.replace('${UserId}', userId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type routing-profile to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/routing-profiles.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param routingProfileId - Identifier for the routingProfileId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRoutingProfile(instanceId, routingProfileId, account, region, partition) {
        var arn = 'arn:${Partition}:connect:${Region}:${Account}:instance/${InstanceId}/routing-profile/${RoutingProfileId}';
        arn = arn.replace('${InstanceId}', instanceId);
        arn = arn.replace('${RoutingProfileId}', routingProfileId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type security-profile to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/connect-security-profiles.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param securityProfileId - Identifier for the securityProfileId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSecurityProfile(instanceId, securityProfileId, account, region, partition) {
        var arn = 'arn:${Partition}:connect:${Region}:${Account}:instance/${InstanceId}/security-profile/${SecurityProfileId}';
        arn = arn.replace('${InstanceId}', instanceId);
        arn = arn.replace('${SecurityProfileId}', securityProfileId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type hierarchy-group to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/agent-hierarchy.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param hierarchyGroupId - Identifier for the hierarchyGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onHierarchyGroup(instanceId, hierarchyGroupId, account, region, partition) {
        var arn = 'arn:${Partition}:connect:${Region}:${Account}:instance/${InstanceId}/agent-group/${HierarchyGroupId}';
        arn = arn.replace('${InstanceId}', instanceId);
        arn = arn.replace('${HierarchyGroupId}', hierarchyGroupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type queue to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/create-queue.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param queueId - Identifier for the queueId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onQueue(instanceId, queueId, account, region, partition) {
        var arn = 'arn:${Partition}:connect:${Region}:${Account}:instance/${InstanceId}/queue/${QueueId}';
        arn = arn.replace('${InstanceId}', instanceId);
        arn = arn.replace('${QueueId}', queueId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type contact-flow to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/connect-contact-flows.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param contactFlowId - Identifier for the contactFlowId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onContactFlow(instanceId, contactFlowId, account, region, partition) {
        var arn = 'arn:${Partition}:connect:${Region}:${Account}:instance/${InstanceId}/contact-flow/${ContactFlowId}';
        arn = arn.replace('${InstanceId}', instanceId);
        arn = arn.replace('${ContactFlowId}', contactFlowId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type hours-of-operation to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/set-hours-operation.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param hoursOfOperationId - Identifier for the hoursOfOperationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onHoursOfOperation(instanceId, hoursOfOperationId, account, region, partition) {
        var arn = 'arn:${Partition}:connect:${Region}:${Account}:instance/${InstanceId}/operating-hours/${HoursOfOperationId}';
        arn = arn.replace('${InstanceId}', instanceId);
        arn = arn.replace('${HoursOfOperationId}', hoursOfOperationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type phone-number to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/contact-center-phone-number.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param phoneNumberId - Identifier for the phoneNumberId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPhoneNumber(instanceId, phoneNumberId, account, region, partition) {
        var arn = 'arn:${Partition}:connect:${Region}:${Account}:instance/${InstanceId}/phone-numbers/${PhoneNumberId}';
        arn = arn.replace('${InstanceId}', instanceId);
        arn = arn.replace('${PhoneNumberId}', phoneNumberId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Connect = Connect;
//# sourceMappingURL=data:application/json;base64,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