"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Dax = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [dax](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazondynamodbacceleratordax.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Dax extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [dax](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazondynamodbacceleratordax.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'dax';
        this.accessLevelList = {
            "Read": [
                "BatchGetItem",
                "ConditionCheckItem",
                "DescribeParameters",
                "GetItem",
                "ListTags",
                "Query",
                "Scan"
            ],
            "Write": [
                "BatchWriteItem",
                "CreateCluster",
                "CreateParameterGroup",
                "CreateSubnetGroup",
                "DecreaseReplicationFactor",
                "DeleteCluster",
                "DeleteItem",
                "DeleteParameterGroup",
                "DeleteSubnetGroup",
                "IncreaseReplicationFactor",
                "PutItem",
                "RebootNode",
                "UpdateCluster",
                "UpdateItem",
                "UpdateParameterGroup",
                "UpdateSubnetGroup"
            ],
            "List": [
                "DescribeClusters",
                "DescribeDefaultParameters",
                "DescribeEvents",
                "DescribeParameterGroups",
                "DescribeSubnetGroups"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * The BatchGetItem action returns the attributes of one or more items from one or more tables.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_BatchGetItem.html
     */
    toBatchGetItem() {
        this.to('dax:BatchGetItem');
        return this;
    }
    /**
     * The BatchWriteItem action operation puts or deletes multiple items in one or more tables.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_BatchWriteItem.html
     */
    toBatchWriteItem() {
        this.to('dax:BatchWriteItem');
        return this;
    }
    /**
     * The ConditionCheckItem operation checks the existence of a set of attributes for the item with the given primary key
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ConditionCheckItem.html
     */
    toConditionCheckItem() {
        this.to('dax:ConditionCheckItem');
        return this;
    }
    /**
     * The CreateCluster action creates a DAX cluster.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - dax:CreateParameterGroup
     * - dax:CreateSubnetGroup
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - iam:GetRole
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_CreateCluster.html
     */
    toCreateCluster() {
        this.to('dax:CreateCluster');
        return this;
    }
    /**
     * The CreateParameterGroup action creates collection of parameters that you apply to all of the nodes in a DAX cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_CreateParameterGroup.html
     */
    toCreateParameterGroup() {
        this.to('dax:CreateParameterGroup');
        return this;
    }
    /**
     * The CreateSubnetGroup action creates a new subnet group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_CreateSubnetGroup.html
     */
    toCreateSubnetGroup() {
        this.to('dax:CreateSubnetGroup');
        return this;
    }
    /**
     * The DecreaseReplicationFactor action removes one or more nodes from a DAX cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DecreaseReplicationFactor.html
     */
    toDecreaseReplicationFactor() {
        this.to('dax:DecreaseReplicationFactor');
        return this;
    }
    /**
     * The DeleteCluster action deletes a previously provisioned DAX cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DeleteCluster.html
     */
    toDeleteCluster() {
        this.to('dax:DeleteCluster');
        return this;
    }
    /**
     * The DeleteItem action deletes a single item in a table by primary key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEnclosingOperation()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html
     */
    toDeleteItem() {
        this.to('dax:DeleteItem');
        return this;
    }
    /**
     * The DeleteParameterGroup action deletes the specified parameter group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DeleteParameterGroup.html
     */
    toDeleteParameterGroup() {
        this.to('dax:DeleteParameterGroup');
        return this;
    }
    /**
     * The DeleteSubnetGroup action deletes a subnet group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DeleteSubnetGroup.html
     */
    toDeleteSubnetGroup() {
        this.to('dax:DeleteSubnetGroup');
        return this;
    }
    /**
     * The DescribeClusters action returns information about all provisioned DAX clusters.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DescribeClusters.html
     */
    toDescribeClusters() {
        this.to('dax:DescribeClusters');
        return this;
    }
    /**
     * The DescribeDefaultParameters action returns the default system parameter information for DAX.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DescribeDefaultParameters.html
     */
    toDescribeDefaultParameters() {
        this.to('dax:DescribeDefaultParameters');
        return this;
    }
    /**
     * The DescribeEvents action returns events related to DAX clusters and parameter groups.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DescribeEvents.html
     */
    toDescribeEvents() {
        this.to('dax:DescribeEvents');
        return this;
    }
    /**
     * The DescribeParameterGroups action returns a list of parameter group descriptions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DescribeParameterGroups.html
     */
    toDescribeParameterGroups() {
        this.to('dax:DescribeParameterGroups');
        return this;
    }
    /**
     * The DescribeParameters action returns the detailed parameter list for a particular parameter group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DescribeParameters.html
     */
    toDescribeParameters() {
        this.to('dax:DescribeParameters');
        return this;
    }
    /**
     * The DescribeSubnetGroups action returns a list of subnet group descriptions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DescribeSubnetGroups.html
     */
    toDescribeSubnetGroups() {
        this.to('dax:DescribeSubnetGroups');
        return this;
    }
    /**
     * The GetItem action returns a set of attributes for the item with the given primary key.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifEnclosingOperation()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html
     */
    toGetItem() {
        this.to('dax:GetItem');
        return this;
    }
    /**
     * The IncreaseReplicationFactor action adds one or more nodes to a DAX cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_IncreaseReplicationFactor.html
     */
    toIncreaseReplicationFactor() {
        this.to('dax:IncreaseReplicationFactor');
        return this;
    }
    /**
     * The ListTags action returns a list all of the tags for a DAX cluster.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_ListTags.html
     */
    toListTags() {
        this.to('dax:ListTags');
        return this;
    }
    /**
     * The PutItem action creates a new item, or replaces an old item with a new item.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEnclosingOperation()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html
     */
    toPutItem() {
        this.to('dax:PutItem');
        return this;
    }
    /**
     * The Query action finds items based on primary key values. You can query any table or secondary index that has a composite primary key (a partition key and a sort key).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Query.html
     */
    toQuery() {
        this.to('dax:Query');
        return this;
    }
    /**
     * The RebootNode action reboots a single node of a DAX cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_RebootNode.html
     */
    toRebootNode() {
        this.to('dax:RebootNode');
        return this;
    }
    /**
     * The Scan action returns one or more items and item attributes by accessing every item in a table or a secondary index.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Scan.html
     */
    toScan() {
        this.to('dax:Scan');
        return this;
    }
    /**
     * The TagResource action associates a set of tags with a DAX resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_TagResource.html
     */
    toTagResource() {
        this.to('dax:TagResource');
        return this;
    }
    /**
     * The UntagResource action removes the association of tags from a DAX resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_UntagResource.html
     */
    toUntagResource() {
        this.to('dax:UntagResource');
        return this;
    }
    /**
     * The UpdateCluster action modifies the settings for a DAX cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_UpdateCluster.html
     */
    toUpdateCluster() {
        this.to('dax:UpdateCluster');
        return this;
    }
    /**
     * The UpdateItem action edits an existing item's attributes, or adds a new item to the table if it does not already exist.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEnclosingOperation()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html
     */
    toUpdateItem() {
        this.to('dax:UpdateItem');
        return this;
    }
    /**
     * The UpdateParameterGroup action modifies the parameters of a parameter group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_UpdateParameterGroup.html
     */
    toUpdateParameterGroup() {
        this.to('dax:UpdateParameterGroup');
        return this;
    }
    /**
     * The UpdateSubnetGroup action modifies an existing subnet group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_UpdateSubnetGroup.html
     */
    toUpdateSubnetGroup() {
        this.to('dax:UpdateSubnetGroup');
        return this;
    }
    /**
     * Adds a resource of type application to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/DAX.access-control.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onApplication(clusterName, account, region, partition) {
        var arn = 'arn:${Partition}:dax:${Region}:${Account}:cache/${ClusterName}';
        arn = arn.replace('${ClusterName}', clusterName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Used to block Transactions APIs calls and allow the non-Transaction APIs calls and vice-versa.
     *
     * Applies to actions:
     * - .toDeleteItem()
     * - .toGetItem()
     * - .toPutItem()
     * - .toUpdateItem()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnclosingOperation(value, operator) {
        return this.if(`dax:EnclosingOperation`, value, operator || 'StringLike');
    }
}
exports.Dax = Dax;
//# sourceMappingURL=data:application/json;base64,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