"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Autoscaling = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [autoscaling](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonec2autoscaling.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Autoscaling extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [autoscaling](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonec2autoscaling.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'autoscaling';
        this.accessLevelList = {
            "Write": [
                "AttachInstances",
                "AttachLoadBalancerTargetGroups",
                "AttachLoadBalancers",
                "BatchDeleteScheduledAction",
                "BatchPutScheduledUpdateGroupAction",
                "CancelInstanceRefresh",
                "CompleteLifecycleAction",
                "CreateLaunchConfiguration",
                "DeleteAutoScalingGroup",
                "DeleteLaunchConfiguration",
                "DeleteLifecycleHook",
                "DeleteNotificationConfiguration",
                "DeletePolicy",
                "DeleteScheduledAction",
                "DetachInstances",
                "DetachLoadBalancerTargetGroups",
                "DetachLoadBalancers",
                "DisableMetricsCollection",
                "EnableMetricsCollection",
                "EnterStandby",
                "ExecutePolicy",
                "ExitStandby",
                "PutLifecycleHook",
                "PutNotificationConfiguration",
                "PutScalingPolicy",
                "PutScheduledUpdateGroupAction",
                "RecordLifecycleActionHeartbeat",
                "ResumeProcesses",
                "SetDesiredCapacity",
                "SetInstanceHealth",
                "SetInstanceProtection",
                "StartInstanceRefresh",
                "SuspendProcesses",
                "TerminateInstanceInAutoScalingGroup",
                "UpdateAutoScalingGroup"
            ],
            "Tagging": [
                "CreateAutoScalingGroup",
                "CreateOrUpdateTags",
                "DeleteTags"
            ],
            "List": [
                "DescribeAccountLimits",
                "DescribeAdjustmentTypes",
                "DescribeAutoScalingGroups",
                "DescribeAutoScalingInstances",
                "DescribeAutoScalingNotificationTypes",
                "DescribeInstanceRefreshes",
                "DescribeLaunchConfigurations",
                "DescribeLifecycleHookTypes",
                "DescribeLifecycleHooks",
                "DescribeLoadBalancerTargetGroups",
                "DescribeLoadBalancers",
                "DescribeMetricCollectionTypes",
                "DescribeNotificationConfigurations",
                "DescribePolicies",
                "DescribeScalingActivities",
                "DescribeScalingProcessTypes",
                "DescribeScheduledActions",
                "DescribeTerminationPolicyTypes"
            ],
            "Read": [
                "DescribeTags"
            ]
        };
    }
    /**
     * Attaches one or more EC2 instances to the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_AttachInstances.html
     */
    toAttachInstances() {
        this.to('autoscaling:AttachInstances');
        return this;
    }
    /**
     * Attaches one or more target groups to the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTargetGroupARNs()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_AttachLoadBalancerTargetGroups.html
     */
    toAttachLoadBalancerTargetGroups() {
        this.to('autoscaling:AttachLoadBalancerTargetGroups');
        return this;
    }
    /**
     * Attaches one or more load balancers to the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifLoadBalancerNames()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_AttachLoadBalancers.html
     */
    toAttachLoadBalancers() {
        this.to('autoscaling:AttachLoadBalancers');
        return this;
    }
    /**
     * Deletes the specified scheduled actions.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_BatchDeleteScheduledAction.html
     */
    toBatchDeleteScheduledAction() {
        this.to('autoscaling:BatchDeleteScheduledAction');
        return this;
    }
    /**
     * Creates or updates multiple scheduled scaling actions for an Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_BatchPutScheduledUpdateGroupAction.html
     */
    toBatchPutScheduledUpdateGroupAction() {
        this.to('autoscaling:BatchPutScheduledUpdateGroupAction');
        return this;
    }
    /**
     * Grants permission to cancel an instance refresh operation in progress
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CancelInstanceRefresh.html
     */
    toCancelInstanceRefresh() {
        this.to('autoscaling:CancelInstanceRefresh');
        return this;
    }
    /**
     * Completes the lifecycle action for the specified token or instance with the specified result.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CompleteLifecycleAction.html
     */
    toCompleteLifecycleAction() {
        this.to('autoscaling:CompleteLifecycleAction');
        return this;
    }
    /**
     * Creates an Auto Scaling group with the specified name and attributes.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifInstanceTypes()
     * - .ifLaunchConfigurationName()
     * - .ifLaunchTemplateVersionSpecified()
     * - .ifLoadBalancerNames()
     * - .ifMaxSize()
     * - .ifMinSize()
     * - .ifTargetGroupARNs()
     * - .ifVPCZoneIdentifiers()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CreateAutoScalingGroup.html
     */
    toCreateAutoScalingGroup() {
        this.to('autoscaling:CreateAutoScalingGroup');
        return this;
    }
    /**
     * Creates a launch configuration.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifImageId()
     * - .ifInstanceType()
     * - .ifSpotPrice()
     * - .ifMetadataHttpTokens()
     * - .ifMetadataHttpPutResponseHopLimit()
     * - .ifMetadataHttpEndpoint()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CreateLaunchConfiguration.html
     */
    toCreateLaunchConfiguration() {
        this.to('autoscaling:CreateLaunchConfiguration');
        return this;
    }
    /**
     * Creates or updates tags for the specified Auto Scaling group.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CreateOrUpdateTags.html
     */
    toCreateOrUpdateTags() {
        this.to('autoscaling:CreateOrUpdateTags');
        return this;
    }
    /**
     * Deletes the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteAutoScalingGroup.html
     */
    toDeleteAutoScalingGroup() {
        this.to('autoscaling:DeleteAutoScalingGroup');
        return this;
    }
    /**
     * Deletes the specified launch configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteLaunchConfiguration.html
     */
    toDeleteLaunchConfiguration() {
        this.to('autoscaling:DeleteLaunchConfiguration');
        return this;
    }
    /**
     * Deletes the specified lifecycle hook.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteLifecycleHook.html
     */
    toDeleteLifecycleHook() {
        this.to('autoscaling:DeleteLifecycleHook');
        return this;
    }
    /**
     * Deletes the specified notification.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteNotificationConfiguration.html
     */
    toDeleteNotificationConfiguration() {
        this.to('autoscaling:DeleteNotificationConfiguration');
        return this;
    }
    /**
     * Deletes the specified Auto Scaling policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeletePolicy.html
     */
    toDeletePolicy() {
        this.to('autoscaling:DeletePolicy');
        return this;
    }
    /**
     * Deletes the specified scheduled action.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteScheduledAction.html
     */
    toDeleteScheduledAction() {
        this.to('autoscaling:DeleteScheduledAction');
        return this;
    }
    /**
     * Deletes the specified tags.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteTags.html
     */
    toDeleteTags() {
        this.to('autoscaling:DeleteTags');
        return this;
    }
    /**
     * Describes the current Auto Scaling resource limits for your AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAccountLimits.html
     */
    toDescribeAccountLimits() {
        this.to('autoscaling:DescribeAccountLimits');
        return this;
    }
    /**
     * Describes the policy adjustment types for use with PutScalingPolicy.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAdjustmentTypes.html
     */
    toDescribeAdjustmentTypes() {
        this.to('autoscaling:DescribeAdjustmentTypes');
        return this;
    }
    /**
     * Describes one or more Auto Scaling groups. If a list of names is not provided, the call describes all Auto Scaling groups.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAutoScalingGroups.html
     */
    toDescribeAutoScalingGroups() {
        this.to('autoscaling:DescribeAutoScalingGroups');
        return this;
    }
    /**
     * Describes one or more Auto Scaling instances. If a list is not provided, the call describes all instances.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAutoScalingInstances.html
     */
    toDescribeAutoScalingInstances() {
        this.to('autoscaling:DescribeAutoScalingInstances');
        return this;
    }
    /**
     * Describes the notification types that are supported by Auto Scaling.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAutoScalingNotificationTypes.html
     */
    toDescribeAutoScalingNotificationTypes() {
        this.to('autoscaling:DescribeAutoScalingNotificationTypes');
        return this;
    }
    /**
     * Grants permission to describe one or more instance refreshes for an Auto Scaling group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeInstanceRefreshes.html
     */
    toDescribeInstanceRefreshes() {
        this.to('autoscaling:DescribeInstanceRefreshes');
        return this;
    }
    /**
     * Describes one or more launch configurations. If you omit the list of names, then the call describes all launch configurations.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLaunchConfigurations.html
     */
    toDescribeLaunchConfigurations() {
        this.to('autoscaling:DescribeLaunchConfigurations');
        return this;
    }
    /**
     * Describes the available types of lifecycle hooks.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLifecycleHookTypes.html
     */
    toDescribeLifecycleHookTypes() {
        this.to('autoscaling:DescribeLifecycleHookTypes');
        return this;
    }
    /**
     * Describes the lifecycle hooks for the specified Auto Scaling group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLifecycleHooks.html
     */
    toDescribeLifecycleHooks() {
        this.to('autoscaling:DescribeLifecycleHooks');
        return this;
    }
    /**
     * Describes the target groups for the specified Auto Scaling group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLoadBalancerTargetGroups.html
     */
    toDescribeLoadBalancerTargetGroups() {
        this.to('autoscaling:DescribeLoadBalancerTargetGroups');
        return this;
    }
    /**
     * Describes the load balancers for the specified Auto Scaling group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLoadBalancers.html
     */
    toDescribeLoadBalancers() {
        this.to('autoscaling:DescribeLoadBalancers');
        return this;
    }
    /**
     * Describes the available CloudWatch metrics for Auto Scaling.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeMetricCollectionTypes.html
     */
    toDescribeMetricCollectionTypes() {
        this.to('autoscaling:DescribeMetricCollectionTypes');
        return this;
    }
    /**
     * Describes the notification actions associated with the specified Auto Scaling group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeNotificationConfigurations.html
     */
    toDescribeNotificationConfigurations() {
        this.to('autoscaling:DescribeNotificationConfigurations');
        return this;
    }
    /**
     * Describes the policies for the specified Auto Scaling group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribePolicies.html
     */
    toDescribePolicies() {
        this.to('autoscaling:DescribePolicies');
        return this;
    }
    /**
     * Describes one or more scaling activities for the specified Auto Scaling group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeScalingActivities.html
     */
    toDescribeScalingActivities() {
        this.to('autoscaling:DescribeScalingActivities');
        return this;
    }
    /**
     * Describes the scaling process types for use with ResumeProcesses and SuspendProcesses.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeScalingProcessTypes.html
     */
    toDescribeScalingProcessTypes() {
        this.to('autoscaling:DescribeScalingProcessTypes');
        return this;
    }
    /**
     * Describes the actions scheduled for your Auto Scaling group that haven't run.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeScheduledActions.html
     */
    toDescribeScheduledActions() {
        this.to('autoscaling:DescribeScheduledActions');
        return this;
    }
    /**
     * Describes the specified tags.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeTags.html
     */
    toDescribeTags() {
        this.to('autoscaling:DescribeTags');
        return this;
    }
    /**
     * Describes the termination policies supported by Auto Scaling.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeTerminationPolicyTypes.html
     */
    toDescribeTerminationPolicyTypes() {
        this.to('autoscaling:DescribeTerminationPolicyTypes');
        return this;
    }
    /**
     * Removes one or more instances from the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DetachInstances.html
     */
    toDetachInstances() {
        this.to('autoscaling:DetachInstances');
        return this;
    }
    /**
     * Detaches one or more target groups from the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTargetGroupARNs()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DetachLoadBalancerTargetGroups.html
     */
    toDetachLoadBalancerTargetGroups() {
        this.to('autoscaling:DetachLoadBalancerTargetGroups');
        return this;
    }
    /**
     * Removes one or more load balancers from the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifLoadBalancerNames()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DetachLoadBalancers.html
     */
    toDetachLoadBalancers() {
        this.to('autoscaling:DetachLoadBalancers');
        return this;
    }
    /**
     * Disables monitoring of the specified metrics for the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DisableMetricsCollection.html
     */
    toDisableMetricsCollection() {
        this.to('autoscaling:DisableMetricsCollection');
        return this;
    }
    /**
     * Enables monitoring of the specified metrics for the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_EnableMetricsCollection.html
     */
    toEnableMetricsCollection() {
        this.to('autoscaling:EnableMetricsCollection');
        return this;
    }
    /**
     * Moves the specified instances into Standby mode.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_EnterStandby.html
     */
    toEnterStandby() {
        this.to('autoscaling:EnterStandby');
        return this;
    }
    /**
     * Executes the specified policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_ExecutePolicy.html
     */
    toExecutePolicy() {
        this.to('autoscaling:ExecutePolicy');
        return this;
    }
    /**
     * Moves the specified instances out of Standby mode.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_ExitStandby.html
     */
    toExitStandby() {
        this.to('autoscaling:ExitStandby');
        return this;
    }
    /**
     * Creates or updates a lifecycle hook for the specified Auto Scaling Group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_PutLifecycleHook.html
     */
    toPutLifecycleHook() {
        this.to('autoscaling:PutLifecycleHook');
        return this;
    }
    /**
     * Configures an Auto Scaling group to send notifications when specified events take place.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_PutNotificationConfiguration.html
     */
    toPutNotificationConfiguration() {
        this.to('autoscaling:PutNotificationConfiguration');
        return this;
    }
    /**
     * Creates or updates a policy for an Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_PutScalingPolicy.html
     */
    toPutScalingPolicy() {
        this.to('autoscaling:PutScalingPolicy');
        return this;
    }
    /**
     * Creates or updates a scheduled scaling action for an Auto Scaling group.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifMaxSize()
     * - .ifMinSize()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_PutScheduledUpdateGroupAction.html
     */
    toPutScheduledUpdateGroupAction() {
        this.to('autoscaling:PutScheduledUpdateGroupAction');
        return this;
    }
    /**
     * Records a heartbeat for the lifecycle action associated with the specified token or instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_RecordLifecycleActionHeartbeat.html
     */
    toRecordLifecycleActionHeartbeat() {
        this.to('autoscaling:RecordLifecycleActionHeartbeat');
        return this;
    }
    /**
     * Resumes the specified suspended Auto Scaling processes, or all suspended process, for the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_ResumeProcesses.html
     */
    toResumeProcesses() {
        this.to('autoscaling:ResumeProcesses');
        return this;
    }
    /**
     * Sets the size of the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_SetDesiredCapacity.html
     */
    toSetDesiredCapacity() {
        this.to('autoscaling:SetDesiredCapacity');
        return this;
    }
    /**
     * Sets the health status of the specified instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_SetInstanceHealth.html
     */
    toSetInstanceHealth() {
        this.to('autoscaling:SetInstanceHealth');
        return this;
    }
    /**
     * Updates the instance protection settings of the specified instances.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_SetInstanceProtection.html
     */
    toSetInstanceProtection() {
        this.to('autoscaling:SetInstanceProtection');
        return this;
    }
    /**
     * Grants permission to start a new instance refresh operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_StartInstanceRefresh.html
     */
    toStartInstanceRefresh() {
        this.to('autoscaling:StartInstanceRefresh');
        return this;
    }
    /**
     * Suspends the specified Auto Scaling processes, or all processes, for the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_SuspendProcesses.html
     */
    toSuspendProcesses() {
        this.to('autoscaling:SuspendProcesses');
        return this;
    }
    /**
     * Terminates the specified instance and optionally adjusts the desired group size.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_TerminateInstanceInAutoScalingGroup.html
     */
    toTerminateInstanceInAutoScalingGroup() {
        this.to('autoscaling:TerminateInstanceInAutoScalingGroup');
        return this;
    }
    /**
     * Updates the configuration for the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceTypes()
     * - .ifLaunchConfigurationName()
     * - .ifLaunchTemplateVersionSpecified()
     * - .ifMaxSize()
     * - .ifMinSize()
     * - .ifVPCZoneIdentifiers()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_UpdateAutoScalingGroup.html
     */
    toUpdateAutoScalingGroup() {
        this.to('autoscaling:UpdateAutoScalingGroup');
        return this;
    }
    /**
     * Adds a resource of type autoScalingGroup to the statement
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-resources
     *
     * @param groupId - Identifier for the groupId.
     * @param groupFriendlyName - Identifier for the groupFriendlyName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifResourceTag()
     * - .ifAwsResourceTag()
     */
    onAutoScalingGroup(groupId, groupFriendlyName, account, region, partition) {
        var arn = 'arn:${Partition}:autoscaling:${Region}:${Account}:autoScalingGroup:${GroupId}:autoScalingGroupName/${GroupFriendlyName}';
        arn = arn.replace('${GroupId}', groupId);
        arn = arn.replace('${GroupFriendlyName}', groupFriendlyName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type launchConfiguration to the statement
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-resources
     *
     * @param id - Identifier for the id.
     * @param launchConfigurationName - Identifier for the launchConfigurationName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLaunchConfiguration(id, launchConfigurationName, account, region, partition) {
        var arn = 'arn:${Partition}:autoscaling:${Region}:${Account}:launchConfiguration:${Id}:launchConfigurationName/${LaunchConfigurationName}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${LaunchConfigurationName}', launchConfigurationName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The AMI used to create the instance.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateLaunchConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifImageId(value, operator) {
        return this.if(`autoscaling:ImageId`, value, operator || 'StringLike');
    }
    /**
     * The type of instance, in terms of the hardware resources available.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateLaunchConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstanceType(value, operator) {
        return this.if(`autoscaling:InstanceType`, value, operator || 'StringLike');
    }
    /**
     * The types of instances, in terms of the hardware resources available.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateAutoScalingGroup()
     * - .toUpdateAutoScalingGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstanceTypes(value, operator) {
        return this.if(`autoscaling:InstanceTypes`, value, operator || 'StringLike');
    }
    /**
     * The name of a launch configuration.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateAutoScalingGroup()
     * - .toUpdateAutoScalingGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLaunchConfigurationName(value, operator) {
        return this.if(`autoscaling:LaunchConfigurationName`, value, operator || 'StringLike');
    }
    /**
     * Filters access by whether users can specify any version of a launch template or only the Latest or Default version
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateAutoScalingGroup()
     * - .toUpdateAutoScalingGroup()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifLaunchTemplateVersionSpecified(value) {
        return this.if(`autoscaling:LaunchTemplateVersionSpecified`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * The name of the load balancer.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toAttachLoadBalancers()
     * - .toCreateAutoScalingGroup()
     * - .toDetachLoadBalancers()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLoadBalancerNames(value, operator) {
        return this.if(`autoscaling:LoadBalancerNames`, value, operator || 'StringLike');
    }
    /**
     * The maximum scaling size.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateAutoScalingGroup()
     * - .toPutScheduledUpdateGroupAction()
     * - .toUpdateAutoScalingGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMaxSize(value, operator) {
        return this.if(`autoscaling:MaxSize`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by whether the HTTP endpoint is enabled for the instance metadata service.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateLaunchConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMetadataHttpEndpoint(value, operator) {
        return this.if(`autoscaling:MetadataHttpEndpoint`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the allowed number of hops when calling the instance metadata service.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateLaunchConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMetadataHttpPutResponseHopLimit(value, operator) {
        return this.if(`autoscaling:MetadataHttpPutResponseHopLimit`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by whether tokens are required when calling the instance metadata service (optional or required)
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateLaunchConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMetadataHttpTokens(value, operator) {
        return this.if(`autoscaling:MetadataHttpTokens`, value, operator || 'StringLike');
    }
    /**
     * The minimum scaling size.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateAutoScalingGroup()
     * - .toPutScheduledUpdateGroupAction()
     * - .toUpdateAutoScalingGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMinSize(value, operator) {
        return this.if(`autoscaling:MinSize`, value, operator || 'NumericEquals');
    }
    /**
     * The value of a tag attached to a resource.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toAttachInstances()
     * - .toAttachLoadBalancerTargetGroups()
     * - .toAttachLoadBalancers()
     * - .toBatchDeleteScheduledAction()
     * - .toBatchPutScheduledUpdateGroupAction()
     * - .toCancelInstanceRefresh()
     * - .toCompleteLifecycleAction()
     * - .toCreateAutoScalingGroup()
     * - .toCreateOrUpdateTags()
     * - .toDeleteAutoScalingGroup()
     * - .toDeleteLifecycleHook()
     * - .toDeleteNotificationConfiguration()
     * - .toDeletePolicy()
     * - .toDeleteScheduledAction()
     * - .toDeleteTags()
     * - .toDetachInstances()
     * - .toDetachLoadBalancerTargetGroups()
     * - .toDetachLoadBalancers()
     * - .toDisableMetricsCollection()
     * - .toEnableMetricsCollection()
     * - .toEnterStandby()
     * - .toExecutePolicy()
     * - .toExitStandby()
     * - .toPutLifecycleHook()
     * - .toPutNotificationConfiguration()
     * - .toPutScalingPolicy()
     * - .toPutScheduledUpdateGroupAction()
     * - .toRecordLifecycleActionHeartbeat()
     * - .toResumeProcesses()
     * - .toSetDesiredCapacity()
     * - .toSetInstanceHealth()
     * - .toSetInstanceProtection()
     * - .toStartInstanceRefresh()
     * - .toSuspendProcesses()
     * - .toTerminateInstanceInAutoScalingGroup()
     * - .toUpdateAutoScalingGroup()
     *
     * Applies to resource types:
     * - autoScalingGroup
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey, value, operator) {
        return this.if(`autoscaling:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * The spot price associated with an instance.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateLaunchConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifSpotPrice(value, operator) {
        return this.if(`autoscaling:SpotPrice`, value, operator || 'NumericEquals');
    }
    /**
     * The ARN of a target group.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toAttachLoadBalancerTargetGroups()
     * - .toCreateAutoScalingGroup()
     * - .toDetachLoadBalancerTargetGroups()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifTargetGroupARNs(value, operator) {
        return this.if(`autoscaling:TargetGroupARNs`, value, operator || 'ArnLike');
    }
    /**
     * The identifier of a VPC zone.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateAutoScalingGroup()
     * - .toUpdateAutoScalingGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVPCZoneIdentifiers(value, operator) {
        return this.if(`autoscaling:VPCZoneIdentifiers`, value, operator || 'StringLike');
    }
}
exports.Autoscaling = Autoscaling;
//# sourceMappingURL=data:application/json;base64,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